const winston = require('winston');
const path = require('path');
const crypto = require('crypto');

class EnterpriseAuditLogger {
  constructor() {
    this.sensitiveFields = [
      'password', 'token', 'secret', 'key', 'authorization',
      'credit_card', 'ssn', 'social_security', 'passport'
    ];
    
    this.initializeLogger();
  }

  initializeLogger() {
    // Create custom format for structured logging
    const auditFormat = winston.format.combine(
      winston.format.timestamp({
        format: 'YYYY-MM-DD HH:mm:ss.SSS'
      }),
      winston.format.errors({ stack: true }),
      winston.format.printf(({ timestamp, level, message, ...meta }) => {
        const logEntry = {
          timestamp,
          level: level.toUpperCase(),
          event: message,
          correlationId: meta.correlationId || crypto.randomUUID(),
          ...this.sanitizeMetadata(meta)
        };
        
        return JSON.stringify(logEntry);
      })
    );

    // Configure transports
    const transports = [
      // Console logging for development
      new winston.transports.Console({
        level: process.env.LOG_LEVEL || 'info',
        format: winston.format.combine(
          winston.format.colorize(),
          winston.format.simple()
        ),
        silent: process.env.NODE_ENV === 'test'
      }),

      // File logging for audit trail
      new winston.transports.File({
        filename: path.join(process.cwd(), 'logs', 'audit.log'),
        level: 'info',
        format: auditFormat,
        maxsize: 50 * 1024 * 1024, // 50MB
        maxFiles: 10,
        tailable: true
      }),

      // Error logging
      new winston.transports.File({
        filename: path.join(process.cwd(), 'logs', 'error.log'),
        level: 'error',
        format: auditFormat,
        maxsize: 50 * 1024 * 1024, // 50MB
        maxFiles: 5,
        tailable: true
      }),

      // Security events logging
      new winston.transports.File({
        filename: path.join(process.cwd(), 'logs', 'security.log'),
        level: 'warn',
        format: auditFormat,
        maxsize: 50 * 1024 * 1024, // 50MB
        maxFiles: 20,
        tailable: true
      })
    ];

    // Add external logging services in production
    if (process.env.NODE_ENV === 'production') {
      // Add Elasticsearch transport if configured
      if (process.env.ELASTICSEARCH_URL) {
        const { ElasticsearchTransport } = require('winston-elasticsearch');
        transports.push(
          new ElasticsearchTransport({
            level: 'info',
            clientOpts: {
              node: process.env.ELASTICSEARCH_URL,
              auth: {
                username: process.env.ELASTICSEARCH_USERNAME,
                password: process.env.ELASTICSEARCH_PASSWORD
              }
            },
            index: 'edumetrix-audit-logs'
          })
        );
      }

      // Add Splunk transport if configured
      if (process.env.SPLUNK_URL) {
        const { SplunkStreamEvent } = require('winston-splunk-httplogger');
        transports.push(
          new SplunkStreamEvent({
            token: process.env.SPLUNK_TOKEN,
            url: process.env.SPLUNK_URL,
            level: 'info'
          })
        );
      }
    }

    this.logger = winston.createLogger({
      transports,
      exitOnError: false,
      rejectionHandlers: [
        new winston.transports.File({
          filename: path.join(process.cwd(), 'logs', 'rejections.log')
        })
      ],
      exceptionHandlers: [
        new winston.transports.File({
          filename: path.join(process.cwd(), 'logs', 'exceptions.log')
        })
      ]
    });

    // Ensure logs directory exists
    this.ensureLogDirectory();
  }

  ensureLogDirectory() {
    const fs = require('fs');
    const logsDir = path.join(process.cwd(), 'logs');
    
    if (!fs.existsSync(logsDir)) {
      fs.mkdirSync(logsDir, { recursive: true });
    }
  }

  // Sanitize sensitive data from logs
  sanitizeMetadata(metadata) {
    const sanitized = { ...metadata };
    
    for (const [key, value] of Object.entries(sanitized)) {
      if (this.isSensitiveField(key)) {
        sanitized[key] = this.maskSensitiveData(value);
      } else if (typeof value === 'object' && value !== null) {
        sanitized[key] = this.sanitizeMetadata(value);
      }
    }
    
    return sanitized;
  }

  isSensitiveField(fieldName) {
    const lowerField = fieldName.toLowerCase();
    return this.sensitiveFields.some(sensitive => 
      lowerField.includes(sensitive)
    );
  }

  maskSensitiveData(value) {
    if (typeof value !== 'string') {
      return '[REDACTED]';
    }
    
    if (value.length <= 4) {
      return '***';
    }
    
    // Show first 2 and last 2 characters for longer strings
    return value.substring(0, 2) + '*'.repeat(value.length - 4) + value.substring(value.length - 2);
  }

  // Authentication events
  logAuthenticationAttempt(success, metadata = {}) {
    const event = success ? 'AUTHENTICATION_SUCCESS' : 'AUTHENTICATION_FAILURE';
    const level = success ? 'info' : 'warn';
    
    this.logger.log(level, event, {
      category: 'AUTHENTICATION',
      success,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  logAuthorizationFailure(metadata = {}) {
    this.logger.warn('AUTHORIZATION_FAILURE', {
      category: 'AUTHORIZATION',
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  // Data access events
  logDataAccess(action, resource, metadata = {}) {
    this.logger.info('DATA_ACCESS', {
      category: 'DATA_ACCESS',
      action,
      resource,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  logDataModification(action, resource, metadata = {}) {
    this.logger.info('DATA_MODIFICATION', {
      category: 'DATA_MODIFICATION',
      action,
      resource,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  // Security events
  logSecurityEvent(event, severity = 'warn', metadata = {}) {
    this.logger.log(severity, event, {
      category: 'SECURITY',
      severity: severity.toUpperCase(),
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  logSuspiciousActivity(description, metadata = {}) {
    this.logger.warn('SUSPICIOUS_ACTIVITY', {
      category: 'SECURITY',
      description,
      severity: 'HIGH',
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  // System events
  logSystemEvent(event, metadata = {}) {
    this.logger.info('SYSTEM_EVENT', {
      category: 'SYSTEM',
      event,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  logError(error, metadata = {}) {
    this.logger.error('APPLICATION_ERROR', {
      category: 'ERROR',
      error: {
        message: error.message,
        stack: error.stack,
        name: error.name
      },
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  // Business events
  logBusinessEvent(event, metadata = {}) {
    this.logger.info('BUSINESS_EVENT', {
      category: 'BUSINESS',
      event,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  // Performance monitoring
  logPerformanceMetric(metric, value, metadata = {}) {
    this.logger.info('PERFORMANCE_METRIC', {
      category: 'PERFORMANCE',
      metric,
      value,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  // Custom log levels
  debug(message, metadata = {}) {
    this.logger.debug(message, this.addDefaultMetadata(metadata));
  }

  info(message, metadata = {}) {
    this.logger.info(message, this.addDefaultMetadata(metadata));
  }

  warn(message, metadata = {}) {
    this.logger.warn(message, this.addDefaultMetadata(metadata));
  }

  error(message, metadata = {}) {
    this.logger.error(message, this.addDefaultMetadata(metadata));
  }

  addDefaultMetadata(metadata) {
    return {
      service: 'edumetrix-platform',
      version: process.env.APP_VERSION || '1.0.0',
      environment: process.env.NODE_ENV || 'development',
      timestamp: new Date().toISOString(),
      ...metadata
    };
  }

  // Request logging middleware
  createRequestLogger() {
    return (req, res, next) => {
      const start = Date.now();
      const correlationId = req.headers['x-correlation-id'] || crypto.randomUUID();
      
      // Add correlation ID to request
      req.correlationId = correlationId;
      res.setHeader('X-Correlation-ID', correlationId);

      // Log request start
      this.info('HTTP_REQUEST_START', {
        correlationId,
        method: req.method,
        url: req.url,
        path: req.path,
        ip: req.ip,
        userAgent: req.get('User-Agent'),
        userId: req.user?.id,
        userType: req.user?.userType
      });

      // Log response
      res.on('finish', () => {
        const duration = Date.now() - start;
        const level = res.statusCode >= 400 ? 'warn' : 'info';
        
        this.logger.log(level, 'HTTP_REQUEST_COMPLETE', {
          correlationId,
          method: req.method,
          url: req.url,
          path: req.path,
          statusCode: res.statusCode,
          duration: `${duration}ms`,
          ip: req.ip,
          userId: req.user?.id,
          userType: req.user?.userType,
          responseSize: res.get('Content-Length')
        });

        // Log slow requests
        if (duration > 5000) { // 5 seconds
          this.warn('SLOW_REQUEST', {
            correlationId,
            method: req.method,
            path: req.path,
            duration: `${duration}ms`,
            threshold: '5000ms'
          });
        }
      });

      next();
    };
  }

  // Compliance logging for GDPR, HIPAA, etc.
  logComplianceEvent(type, details, metadata = {}) {
    this.logger.info('COMPLIANCE_EVENT', {
      category: 'COMPLIANCE',
      type,
      details,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  logDataProcessingEvent(action, dataType, legalBasis, metadata = {}) {
    this.logger.info('DATA_PROCESSING', {
      category: 'COMPLIANCE',
      action,
      dataType,
      legalBasis,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }

  // Health check logging
  logHealthCheck(status, checks, metadata = {}) {
    this.logger.info('HEALTH_CHECK', {
      category: 'MONITORING',
      status,
      checks,
      timestamp: new Date().toISOString(),
      ...metadata
    });
  }
}

// Create singleton instance
const auditLogger = new EnterpriseAuditLogger();

module.exports = auditLogger;