const request = require('supertest');
const express = require('express');
const mongoose = require('mongoose');
const jwt = require('jsonwebtoken');
const User = require('../../models/admin/User');
const reportsRoutes = require('../../routes/admin/reports');

const app = express();
app.use(express.json());
app.use('/api/reports', reportsRoutes);

describe('Reports API Integration Tests', () => {
  let adminToken;
  let adminUser;

  beforeAll(async () => {
    await mongoose.connect('mongodb://localhost:27017/edumetrix_test', {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });

    // Create admin user for testing
    adminUser = await User.create({
      username: 'admin@test.com',
      password: 'hashedPassword',
      userType: 'admin',
      profile: {
        name: 'Test Admin',
        email: 'admin@test.com'
      }
    });

    // Generate JWT token
    adminToken = jwt.sign(
      { userId: adminUser._id, userType: 'admin' },
      process.env.JWT_SECRET || 'test_secret',
      { expiresIn: '1h' }
    );
  });

  afterAll(async () => {
    await mongoose.connection.close();
  });

  describe('GET /api/reports/stats', () => {
    test('should return statistics for authorized admin', async () => {
      const response = await request(app)
        .get('/api/reports/stats')
        .set('Authorization', `Bearer ${adminToken}`);

      expect(response.status).toBe(200);
      expect(response.body).toHaveProperty('totalStudents');
      expect(response.body).toHaveProperty('totalTeachers');
      expect(response.body).toHaveProperty('totalClasses');
      expect(response.body).toHaveProperty('totalRevenue');
      expect(typeof response.body.totalStudents).toBe('number');
    });

    test('should reject unauthorized requests', async () => {
      const response = await request(app)
        .get('/api/reports/stats');

      expect(response.status).toBe(401);
    });

    test('should reject invalid token', async () => {
      const response = await request(app)
        .get('/api/reports/stats')
        .set('Authorization', 'Bearer invalid_token');

      expect(response.status).toBe(401);
    });
  });

  describe('GET /api/reports/students', () => {
    test('should return students report with date filtering', async () => {
      const startDate = '2025-01-01';
      const endDate = '2025-12-31';

      const response = await request(app)
        .get('/api/reports/students')
        .query({ startDate, endDate })
        .set('Authorization', `Bearer ${adminToken}`);

      expect(response.status).toBe(200);
      expect(Array.isArray(response.body)).toBe(true);
    });

    test('should return students report without date filtering', async () => {
      const response = await request(app)
        .get('/api/reports/students')
        .set('Authorization', `Bearer ${adminToken}`);

      expect(response.status).toBe(200);
      expect(Array.isArray(response.body)).toBe(true);
    });
  });

  describe('GET /api/reports/teachers', () => {
    test('should return teachers report', async () => {
      const response = await request(app)
        .get('/api/reports/teachers')
        .set('Authorization', `Bearer ${adminToken}`);

      expect(response.status).toBe(200);
      expect(Array.isArray(response.body)).toBe(true);
    });
  });

  describe('GET /api/reports/classes', () => {
    test('should return classes report', async () => {
      const response = await request(app)
        .get('/api/reports/classes')
        .set('Authorization', `Bearer ${adminToken}`);

      expect(response.status).toBe(200);
      expect(Array.isArray(response.body)).toBe(true);
    });
  });

  describe('GET /api/reports/finance', () => {
    test('should return finance report', async () => {
      const response = await request(app)
        .get('/api/reports/finance')
        .set('Authorization', `Bearer ${adminToken}`);

      expect(response.status).toBe(200);
      expect(Array.isArray(response.body)).toBe(true);
    });
  });

  describe('GET /api/reports/attendance', () => {
    test('should return attendance report', async () => {
      const response = await request(app)
        .get('/api/reports/attendance')
        .set('Authorization', `Bearer ${adminToken}`);

      expect(response.status).toBe(200);
      expect(Array.isArray(response.body)).toBe(true);
    });
  });
});