const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');
const readline = require('readline');
require('dotenv').config();

const User = require('./models/admin/User');

const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout
});

function askQuestion(question) {
  return new Promise((resolve) => {
    rl.question(question, (answer) => {
      resolve(answer);
    });
  });
}

async function setupProductionAdmin() {
  try {
    console.log('🔧 Setting up EduMetrix UK Admin Account for Production\n');
    
    // Connect to MongoDB
    await mongoose.connect(process.env.MONGODB_URI);
    console.log('✅ Connected to MongoDB\n');

    // Get admin details
    const adminName = await askQuestion('👤 Enter admin full name: ');
    const adminEmail = await askQuestion('📧 Enter admin email: ');
    const adminUsername = await askQuestion('🔑 Enter admin username: ');
    const adminPassword = await askQuestion('🔐 Enter admin password: ');
    
    console.log('\n⏳ Creating admin account...');

    // Hash password
    const hashedPassword = await bcrypt.hash(adminPassword, 12);

    // Update or create admin user
    const adminUser = await User.findOneAndUpdate(
      { userType: 'admin' },
      {
        username: adminUsername,
        password: hashedPassword,
        userType: 'admin',
        profile: {
          name: adminName,
          email: adminEmail,
          position: 'System Administrator',
          department: 'Administration'
        },
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date()
      },
      { upsert: true, new: true }
    );

    console.log('\n✅ Admin account created successfully!');
    console.log('📋 Admin Details:');
    console.log(`   Name: ${adminName}`);
    console.log(`   Email: ${adminEmail}`);
    console.log(`   Username: ${adminUsername}`);
    console.log(`   User Type: admin`);
    console.log(`   Status: Active\n`);

    console.log('🚀 Your EduMetrix UK platform is now ready for production use!');
    console.log('\n📱 Access the admin portal at: http://localhost:3001');
    console.log('🔑 Login with the credentials you just created');

  } catch (error) {
    console.error('❌ Error setting up admin account:', error);
  } finally {
    rl.close();
    await mongoose.connection.close();
  }
}

setupProductionAdmin();