const express = require('express');
const mongoose = require('mongoose');
const dotenv = require('dotenv');
const http = require('http');
const socketIo = require('socket.io');
const path = require('path');
const fs = require('fs');

// Load environment variables
dotenv.config();

// Import security middleware
const securityPipeline = require('./middleware/securityPipeline');
const enterpriseAuth = require('./middleware/enterpriseAuth');
const rbac = require('./middleware/rbac');
const validation = require('./middleware/validation');
const auditLogger = require('./utils/auditLogger');
const encryption = require('./utils/encryption');

class SecureEduMetrixServer {
  constructor() {
    this.app = express();
    this.server = http.createServer(this.app);
    this.io = null;
    
    this.initializeServer();
  }

  async initializeServer() {
    try {
      // Initialize logging
      this.setupLogging();
      
      // Setup security infrastructure
      this.setupSecurity();
      
      // Initialize database
      await this.connectDatabase();
      
      // Setup socket.io with security
      this.setupSocketIO();
      
      // Setup routes with security
      this.setupRoutes();
      
      // Setup error handling
      this.setupErrorHandling();
      
      // Start server
      this.startServer();
      
    } catch (error) {
      auditLogger.error('Server initialization failed', {
        error: error.message,
        stack: error.stack
      });
      process.exit(1);
    }
  }

  setupLogging() {
    // Ensure logs directory exists
    const logsDir = path.join(__dirname, 'logs');
    if (!fs.existsSync(logsDir)) {
      fs.mkdirSync(logsDir, { recursive: true });
    }

    auditLogger.logSystemEvent('SERVER_STARTING', {
      nodeVersion: process.version,
      environment: process.env.NODE_ENV,
      pid: process.pid
    });
  }

  setupSecurity() {
    // Trust proxy for accurate IP addresses
    this.app.set('trust proxy', 1);
    
    // Set security context
    this.app.set('securityPipeline', securityPipeline);
    this.app.set('rbac', rbac);
    this.app.set('auditLogger', auditLogger);
    
    // Parse JSON with size limits
    this.app.use(express.json({ 
      limit: '1mb',
      verify: (req, res, buf) => {
        // Verify JSON integrity
        try {
          JSON.parse(buf);
        } catch (error) {
          const err = new Error('Invalid JSON payload');
          err.status = 400;
          throw err;
        }
      }
    }));
    
    this.app.use(express.urlencoded({ 
      extended: true, 
      limit: '1mb' 
    }));

    // Apply default security pipeline to all requests
    this.app.use('*', securityPipeline.getSecurityPipeline('public'));

    auditLogger.logSystemEvent('SECURITY_INITIALIZED', {
      components: [
        'SecurityHeaders',
        'RateLimiting', 
        'Authentication',
        'Authorization',
        'Validation',
        'Encryption',
        'AuditLogging'
      ]
    });
  }

  async connectDatabase() {
    try {
      const mongoOptions = {
        useNewUrlParser: true,
        useUnifiedTopology: true,
        maxPoolSize: 10,
        serverSelectionTimeoutMS: 5000,
        socketTimeoutMS: 45000,
      };

      await mongoose.connect(
        process.env.MONGODB_URI || 'mongodb://localhost:27017/edumetrix',
        mongoOptions
      );

      auditLogger.logSystemEvent('DATABASE_CONNECTED', {
        host: mongoose.connection.host,
        database: mongoose.connection.name
      });

      // Handle database events
      mongoose.connection.on('error', (error) => {
        auditLogger.error('Database error', { error: error.message });
      });

      mongoose.connection.on('disconnected', () => {
        auditLogger.warn('Database disconnected');
      });

    } catch (error) {
      auditLogger.error('Database connection failed', {
        error: error.message,
        host: process.env.MONGODB_URI
      });
      throw error;
    }
  }

  setupSocketIO() {
    this.io = socketIo(this.server, {
      cors: {
        origin: [
          "http://localhost:3001",
          "http://localhost:3002", 
          "http://localhost:3003",
          process.env.FRONTEND_URL
        ].filter(Boolean),
        methods: ["GET", "POST"],
        credentials: true
      },
      allowEIO3: true,
      transports: ['websocket', 'polling']
    });

    // Socket.io authentication middleware
    this.io.use(async (socket, next) => {
      try {
        const token = socket.handshake.auth.token;
        
        if (!token) {
          throw new Error('Authentication token required');
        }

        // Verify token (simplified for socket.io)
        if (token.startsWith('dev-')) {
          // Development mode
          socket.user = { id: 'dev-user', userType: 'admin' };
        } else {
          // Production token verification would go here
          socket.user = { id: 'authenticated-user' };
        }

        auditLogger.logSecurityEvent('SOCKET_AUTHENTICATED', 'info', {
          socketId: socket.id,
          userId: socket.user.id
        });

        next();
      } catch (error) {
        auditLogger.logSecurityEvent('SOCKET_AUTH_FAILED', 'warn', {
          socketId: socket.id,
          error: error.message
        });
        next(new Error('Authentication failed'));
      }
    });

    // Socket.io event handlers with security
    this.io.on('connection', (socket) => {
      auditLogger.logSecurityEvent('SOCKET_CONNECTED', 'info', {
        socketId: socket.id,
        userId: socket.user?.id
      });

      socket.on('join-room', (data) => {
        // Validate room access
        if (this.validateRoomAccess(socket.user, data.room)) {
          socket.join(data.room);
          auditLogger.logDataAccess('ROOM_JOINED', data.room, {
            socketId: socket.id,
            userId: socket.user?.id
          });
        } else {
          socket.emit('error', { message: 'Access denied to room' });
        }
      });

      socket.on('disconnect', () => {
        auditLogger.logSecurityEvent('SOCKET_DISCONNECTED', 'info', {
          socketId: socket.id,
          userId: socket.user?.id
        });
      });
    });

    this.app.set('io', this.io);
  }

  validateRoomAccess(user, room) {
    // Implement room access validation logic
    if (!user || !room) return false;
    
    // Allow development access
    if (process.env.NODE_ENV === 'development') return true;
    
    // Implement actual room access logic based on user permissions
    return true;
  }

  setupRoutes() {
    // Health check endpoint (public)
    this.app.get('/health', securityPipeline.healthCheck());
    
    // Security configuration endpoint (admin only)
    this.app.get('/api/security/config', ...securityPipeline.getSecurityConfig());

    // Authentication routes with enhanced security
    this.app.use('/api/auth', 
      securityPipeline.getSecurityPipeline('auth'),
      require('./routes/admin/auth')
    );

    // Refresh token endpoint
    this.app.post('/api/auth/refresh',
      securityPipeline.getSecurityPipeline('auth'),
      enterpriseAuth.refreshAccessToken
    );

    // Logout endpoint
    this.app.post('/api/auth/logout',
      securityPipeline.getSecurityPipeline('api'),
      enterpriseAuth.logout
    );

    // Admin routes with maximum security
    this.app.use('/api/admin',
      securityPipeline.getSecurityPipeline('admin'),
      rbac.requirePermission('users.read'),
      require('./routes/admin/admin')
    );

    // Finance routes with strict access control
    this.app.use('/api/finance',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requireAnyPermission(['finance.read', 'finance.create']),
      require('./routes/admin/finance')
    );

    // Teacher routes with role-based access
    this.app.use('/api/teacher',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('classes.read'),
      require('./routes/teacher/teacher')
    );

    // Student routes with restricted access
    this.app.use('/api/student',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('profile.read'),
      require('./routes/student/student')
    );

    // File upload routes with enhanced security
    this.app.use('/api/upload',
      securityPipeline.getSecurityPipeline('upload'),
      rbac.requirePermission('files.upload'),
      this.setupFileUploadRoutes()
    );

    // Chat routes with communication permissions
    this.app.use('/api/chat',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('chat.read'),
      require('./routes/admin/chat')
    );

    // Class management routes
    this.app.use('/api/classes',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('classes.read'),
      require('./routes/teacher/classes')
    );

    // Assignment routes with proper permissions
    this.app.use('/api/teacher/assignments',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('assignments.create'),
      require('./routes/teacher/assignments')
    );

    this.app.use('/api/student/assignments',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('assignments.read'),
      require('./routes/student/assignments')
    );

    // Exam routes with role-based permissions
    this.app.use('/api/teacher/exams',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('exams.create'),
      require('./routes/teacher/exams')
    );

    this.app.use('/api/student/exams',
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('exams.take'),
      require('./routes/student/exams')
    );

    this.app.use('/api/admin/exams',
      securityPipeline.getSecurityPipeline('admin'),
      rbac.requirePermission('exams.read'),
      require('./routes/admin/exams')
    );

    // Static file serving with security
    this.app.use('/uploads', 
      securityPipeline.getSecurityPipeline('api'),
      rbac.requirePermission('files.download'),
      express.static('uploads', {
        setHeaders: (res, path) => {
          res.setHeader('X-Content-Type-Options', 'nosniff');
          res.setHeader('Cache-Control', 'private, no-cache');
        }
      })
    );

    // 404 handler
    this.app.use('*', (req, res) => {
      auditLogger.warn('Route not found', {
        path: req.path,
        method: req.method,
        ip: req.ip,
        userAgent: req.get('User-Agent')
      });

      res.status(404).json({
        error: 'Route not found',
        code: 'ROUTE_NOT_FOUND',
        path: req.path
      });
    });
  }

  setupFileUploadRoutes() {
    const router = express.Router();

    router.post('/', (req, res) => {
      if (!req.files || req.files.length === 0) {
        return res.status(400).json({
          error: 'No files uploaded',
          code: 'NO_FILES'
        });
      }

      const uploadedFiles = req.files.map(file => ({
        filename: file.filename,
        originalName: file.originalname,
        size: file.size,
        mimetype: file.mimetype,
        url: `/uploads/${file.filename}`
      }));

      auditLogger.logDataModification('FILE_UPLOAD', 'files', {
        fileCount: uploadedFiles.length,
        totalSize: uploadedFiles.reduce((sum, f) => sum + f.size, 0),
        userId: req.user?.id
      });

      res.json({
        message: 'Files uploaded successfully',
        files: uploadedFiles
      });
    });

    return router;
  }

  setupErrorHandling() {
    // Global error handler
    this.app.use(securityPipeline.errorHandler());

    // Handle uncaught exceptions
    process.on('uncaughtException', (error) => {
      auditLogger.error('Uncaught exception', {
        error: error.message,
        stack: error.stack
      });
      
      // Graceful shutdown
      this.gracefulShutdown('UNCAUGHT_EXCEPTION');
    });

    // Handle unhandled promise rejections
    process.on('unhandledRejection', (reason, promise) => {
      auditLogger.error('Unhandled promise rejection', {
        reason: reason?.message || reason,
        stack: reason?.stack
      });
      
      // Graceful shutdown
      this.gracefulShutdown('UNHANDLED_REJECTION');
    });

    // Handle shutdown signals
    process.on('SIGTERM', () => this.gracefulShutdown('SIGTERM'));
    process.on('SIGINT', () => this.gracefulShutdown('SIGINT'));
  }

  gracefulShutdown(signal) {
    auditLogger.logSystemEvent('SERVER_SHUTDOWN_INITIATED', { signal });

    this.server.close(() => {
      auditLogger.logSystemEvent('SERVER_SHUTDOWN_COMPLETE');
      
      mongoose.connection.close(() => {
        auditLogger.logSystemEvent('DATABASE_CONNECTION_CLOSED');
        process.exit(0);
      });
    });

    // Force shutdown after 30 seconds
    setTimeout(() => {
      auditLogger.error('Force shutdown after timeout');
      process.exit(1);
    }, 30000);
  }

  startServer() {
    const PORT = process.env.PORT || 5001;
    
    this.server.listen(PORT, () => {
      auditLogger.logSystemEvent('SERVER_STARTED', {
        port: PORT,
        environment: process.env.NODE_ENV,
        pid: process.pid,
        nodeVersion: process.version
      });

    });
  }
}

// Start the secure server
new SecureEduMetrixServer();

module.exports = SecureEduMetrixServer;