import express from 'express';
import mongoose from 'mongoose';
import crypto from 'crypto';
import QuickTest from '../../models/teacher/QuickTest.js';
import Class from '../../models/Class.js';
import Student from '../../models/student/Student.js';
import { validationResult } from 'express-validator';
import Teacher from '../../models/teacher/Teacher.js'

const router = express.Router();

// Helper function to convert development IDs to valid ObjectIds
const getValidObjectId = (id) => {
  if (mongoose.Types.ObjectId.isValid(id)) {
    return id; // Valid ObjectId
  } else if (typeof id === 'string' && id.startsWith('dev-')) {
    // For development mode with string IDs, create a consistent ObjectId
    const hash = crypto.createHash('md5').update(id).digest('hex');
    const objectIdHex = hash.substring(0, 24);
    return new mongoose.Types.ObjectId(objectIdHex);
  }
  return id; // Return as-is if not a dev string
};

// Get teacher's students with their subjects
router.get('/students', async (req, res) => {
  try {

    const teacherId = getValidObjectId(req.user._id);

    // Fetch students from both sources: Teacher schema (assigned students) and Class schema (scheduled classes)

    // 1. Get assigned students from Teacher schema
    const teacher = await Teacher.findById(teacherId)
      .populate({
        path: 'assignedStudents.studentId',
        select: 'studentName email class subjects phoneNumber'
      });

    // 2. Get students from Class schema (students with scheduled classes)
    const classStudents = await Class.find({ teacherId,      status: { $in: ['scheduled', 'ongoing', 'completed'] }
 })
      .populate({
        path: 'studentId',
        select: 'studentName email class subjects phoneNumber'
      })
      .select('studentId subject scheduledDate status')
      .sort({ scheduledDate: -1 });

    let students = [];
    const studentMap = new Map(); // To avoid duplicates and merge data
    // Process assigned students from Teacher schema
    if (teacher && teacher.assignedStudents && teacher.assignedStudents.length > 0) {
      teacher.assignedStudents.forEach(assignment => {
        if (assignment.studentId) {
          const studentKey = assignment.studentId._id.toString();
          studentMap.set(studentKey, {
            student: {
              _id: assignment.studentId._id,
              studentName: assignment.studentId.studentName,
              email: assignment.studentId.email,
              class: assignment.studentId.class || 'No Class',
              subjects: assignment.studentId.subjects || [assignment.subject],
              phoneNumber: assignment.studentId.phoneNumber
            },
            subject: assignment.subject,
            canChat: true,
            attendanceRate: Math.floor(Math.random() * 20) + 80, // Mock attendance rate
            lastClass: new Date().toISOString().split('T')[0],
            source: 'teacher_assignment'
          });
        }
      });
    }

    // Process students from Class schema
    if (classStudents && classStudents.length > 0) {
      classStudents.forEach(classItem => {
        if (classItem.studentId) {
          const studentKey = classItem.studentId._id.toString();

          // Calculate last class date from scheduled classes
          const lastClassDate = classItem.scheduledDate ? classItem.scheduledDate.toISOString().split('T')[0] : new Date().toISOString().split('T')[0];

          if (studentMap.has(studentKey)) {
            // Update existing student with class information
            const existingStudent = studentMap.get(studentKey);
            existingStudent.lastClass = lastClassDate; // Update with actual class date
            existingStudent.source = 'both'; // Mark as found in both sources
          } else {
            // Add new student from class schedule
            studentMap.set(studentKey, {
              student: {
                _id: classItem.studentId._id,
                studentName: classItem.studentId.studentName,
                email: classItem.studentId.email,
                class: classItem.studentId.class || 'No Class',
                subjects: classItem.studentId.subjects || [classItem.subject],
                phoneNumber: classItem.studentId.phoneNumber
              },
              subject: classItem.subject,
              canChat: true,
              attendanceRate: Math.floor(Math.random() * 20) + 80, // Mock attendance rate
              lastClass: lastClassDate,
              source: 'class_schedule'
            });
          }
        }
      });
    }

    // Convert Map to array
    students = Array.from(studentMap.values());



    res.json({
      success: true,
      data: students,
      meta: {
        totalStudents: students.length,
        sources: {
          teacherAssignments: students.filter(s => s.source === 'teacher_assignment' || s.source === 'both').length,
          classSchedules: students.filter(s => s.source === 'class_schedule' || s.source === 'both').length,
          both: students.filter(s => s.source === 'both').length
        }
      }
    });

  } catch (error) {
    console.error('Get students error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch students',
      error: error.message
    });
  }
});

// Create a new quick test
router.post('/', async (req, res) => {
  try {
    console.log('🚀 POST /quicktest endpoint called');
    console.log('🔍 Request body:', JSON.stringify(req.body, null, 2));
    console.log('🔍 User:', req.user);
    const {
      title,
      description,
      studentId,
      subject,
      topic,
      questions,
      duration
    } = req.body;

    // Validate input
    if (!title || !studentId || !subject || !topic || !questions || !Array.isArray(questions) || questions.length === 0) {
      return res.status(400).json({
        success: false,
        message: 'Title, student, subject, topic, and questions are required'
      });
    }

    // Validate duration
    if (!duration || ![5, 10, 15].includes(duration)) {
      return res.status(400).json({
        success: false,
        message: 'Duration must be 5, 10, or 15 minutes'
      });
    }

    // Validate questions format
    for (let i = 0; i < questions.length; i++) {
      const q = questions[i];
      if (!q.question || !q.options || !Array.isArray(q.options) || q.options.length !== 4) {
        return res.status(400).json({
          success: false,
          message: `Question ${i + 1} must have a question text and exactly 4 options`
        });
      }

      // Check if exactly one option is marked as correct
      const correctOptions = q.options.filter(opt => opt.isCorrect);
      if (correctOptions.length !== 1) {
        return res.status(400).json({
          success: false,
          message: `Question ${i + 1} must have exactly one correct answer`
        });
      }

      // Set correct answer index
      questions[i].correctAnswer = q.options.findIndex(opt => opt.isCorrect);
      questions[i].marks = q.marks || 1;
    }

    // Calculate total marks
    const totalMarks = questions.reduce((total, q) => total + (q.marks || 1), 0);
    
    const quickTest = new QuickTest({
      title,
      description,
      teacherId: getValidObjectId(req.user._id),
      studentId: getValidObjectId(studentId),
      subject,
      topic,
      questions,
      duration,
      totalMarks
    });

    await quickTest.save();

    // Create notification for student
    const io = req.app.get('io');
    const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');

    if (io) {
      console.log('📢 Creating QuickTest assignment notification for student:', studentId);

      const notificationData = NotificationTemplates.quicktestAssigned(
        title,
        'No specific due date'
      );

      await createNotification({
        userId: getValidObjectId(studentId),
        userType: 'student',
        type: notificationData.type,
        title: notificationData.title,
        message: `New quick test assigned: ${title} (${subject} - ${topic}). Duration: ${duration} minutes`,
        relatedId: quickTest._id,
        relatedModel: 'QuickTest',
        priority: notificationData.priority,
        actionUrl: '/student/quick-test',
        io
      });

      console.log('✅ QuickTest assignment notification created successfully');
    } else {
      console.warn('⚠️ Socket.IO instance not available for QuickTest notification');
    }

    res.status(201).json({
      success: true,
      message: 'Quick test created successfully',
      data: quickTest
    });

  } catch (error) {
    console.error('Create quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create quick test',
      error: error.message
    });
  }
});

// Get all quick tests for teacher
router.get('/', async (req, res) => {
  try {
    const { status, studentId, limit = 50, page = 1 } = req.query;
    
    const filter = { teacherId: getValidObjectId(req.user._id) };

    if (status) filter.status = status;
    if (studentId) filter.studentId = getValidObjectId(studentId);

    const skip = (page - 1) * limit;

    const quickTests = await QuickTest.find(filter)
      .populate('studentId', 'studentName email')
      .populate('submissions.studentId', 'studentName email')
      .sort({ createdAt: -1 })
      .limit(parseInt(limit))
      .skip(skip);

    const total = await QuickTest.countDocuments(filter);

    res.json({
      success: true,
      data: quickTests,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get quick tests error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick tests',
      error: error.message
    });
  }
});

// Get specific quick test
router.get('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: getValidObjectId(req.user._id)
    })
    .populate('studentId', 'studentName email')
    .populate('submissions.studentId', 'studentName email');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    res.json({
      success: true,
      data: quickTest
    });

  } catch (error) {
    console.error('Get quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick test',
      error: error.message
    });
  }
});

// Update quick test
router.put('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;
    const updates = req.body;

    // Don't allow updating if test is active or completed
    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: getValidObjectId(req.user._id)
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status !== 'draft') {
      return res.status(400).json({
        success: false,
        message: 'Cannot update test that is not in draft status'
      });
    }

    // Validate questions if updating
    if (updates.questions && Array.isArray(updates.questions)) {
      for (let i = 0; i < updates.questions.length; i++) {
        const q = updates.questions[i];
        if (!q.question || !q.options || !Array.isArray(q.options) || q.options.length !== 4) {
          return res.status(400).json({
            success: false,
            message: `Question ${i + 1} must have a question text and exactly 4 options`
          });
        }

        const correctOptions = q.options.filter(opt => opt.isCorrect);
        if (correctOptions.length !== 1) {
          return res.status(400).json({
            success: false,
            message: `Question ${i + 1} must have exactly one correct answer`
          });
        }

        updates.questions[i].correctAnswer = q.options.findIndex(opt => opt.isCorrect);
        updates.questions[i].marks = q.marks || 1;
      }
    }

    const updatedTest = await QuickTest.findByIdAndUpdate(
      testId,
      { ...updates, updatedAt: new Date() },
      { new: true, runValidators: true }
    )
    .populate('studentId', 'studentName email');

    res.json({
      success: true,
      message: 'Quick test updated successfully',
      data: updatedTest
    });

  } catch (error) {
    console.error('Update quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update quick test',
      error: error.message
    });
  }
});

// Start quick test (make it active)
router.post('/:testId/start', async (req, res) => {
  try {
    const { testId } = req.params;
    const { duration } = req.body; // duration in minutes

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: getValidObjectId(req.user._id)
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status !== 'draft') {
      return res.status(400).json({
        success: false,
        message: 'Quick test is not in draft status'
      });
    }

    if (quickTest.questions.length === 0) {
      return res.status(400).json({
        success: false,
        message: 'Cannot start test with no questions'
      });
    }

    const startTime = new Date();

    quickTest.status = 'active';
    quickTest.startTime = startTime;
    // No longer setting a global endTime - timer starts individually for each student
    quickTest.duration = duration || quickTest.duration;

    await quickTest.save();

    // Get Socket.IO and notification helper
    const io = req.app.get('io');
    const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');

    if (io) {
      console.log(`📢 QuickTest started - Sending notification to student: ${quickTest.studentId}`);

      // Send notification to student that test has started
      try {
        const studentIdForNotification = getValidObjectId(quickTest.studentId);

        await createNotification({
          userId: studentIdForNotification,
          userType: 'student',
          type: 'quicktest_assigned',
          title: 'Quick Test Started!',
          message: `Your teacher has started "${quickTest.title}". Duration: ${quickTest.duration} minutes. Please join now!`,
          relatedId: quickTest._id,
          relatedModel: 'QuickTest',
          priority: 'high',
          actionUrl: '/student/quick-test',
          io
        });

        console.log(`✅ QuickTest start notification sent to student`);
      } catch (notifError) {
        console.error('❌ Failed to send QuickTest start notification:', notifError);
      }

      // Also emit socket event for real-time UI update
      console.log(`📢 Emitting quick-test-started socket event to student-${quickTest.studentId}`);
      io.to(`student-${quickTest.studentId}`).emit('quick-test-started', {
        testId: quickTest._id,
        title: quickTest.title,
        subject: quickTest.subject,
        topic: quickTest.topic,
        duration: quickTest.duration,
        startTime
      });
    } else {
      console.warn('⚠️ Socket.IO instance not available for QuickTest start');
    }

    res.json({
      success: true,
      message: 'Quick test started successfully',
      data: {
        testId: quickTest._id,
        status: quickTest.status,
        startTime,
        duration: quickTest.duration,
        note: 'Timer will start individually for each student when they begin the test'
      }
    });

  } catch (error) {
    console.error('Start quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to start quick test',
      error: error.message
    });
  }
});

// Stop quick test
router.post('/:testId/stop', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: getValidObjectId(req.user._id)
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status !== 'active') {
      return res.status(400).json({
        success: false,
        message: 'Quick test is not active'
      });
    }

    quickTest.status = 'completed';
    quickTest.endTime = new Date();

    await quickTest.save();

    // Emit to the specific student
    const io = req.app.get('io');
    if (io) {
      io.to(`student-${quickTest.studentId}`).emit('quick-test-stopped', {
        testId: quickTest._id,
        endTime: quickTest.endTime
      });
    }

    res.json({
      success: true,
      message: 'Quick test stopped successfully',
      data: {
        testId: quickTest._id,
        status: quickTest.status,
        endTime: quickTest.endTime
      }
    });

  } catch (error) {
    console.error('Stop quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to stop quick test',
      error: error.message
    });
  }
});

// Get quick test results
router.get('/:testId/results', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: getValidObjectId(req.user._id)
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Calculate statistics
    const submissions = quickTest.submissions;
    const totalStudents = submissions.length;
    const averageScore = totalStudents > 0 ? 
      submissions.reduce((sum, sub) => sum + sub.percentage, 0) / totalStudents : 0;

    const passedStudents = submissions.filter(sub => sub.percentage >= 50).length;
    const passRate = totalStudents > 0 ? (passedStudents / totalStudents) * 100 : 0;

    const stats = {
      totalStudents,
      submittedCount: submissions.length,
      averageScore: Math.round(averageScore),
      passRate: Math.round(passRate),
      highestScore: totalStudents > 0 ? Math.max(...submissions.map(s => s.percentage)) : 0,
      lowestScore: totalStudents > 0 ? Math.min(...submissions.map(s => s.percentage)) : 0
    };

    res.json({
      success: true,
      data: {
        test: {
          _id: quickTest._id,
          title: quickTest.title,
          subject: quickTest.subject,
          topic: quickTest.topic,
          totalMarks: quickTest.totalMarks,
          status: quickTest.status,
          startTime: quickTest.startTime,
          endTime: quickTest.endTime,
          studentName: 'Student' // Default name for development
        },
        submissions: submissions.map(sub => ({
          studentId: sub.studentId,
          studentName: 'Student', // Default name for development
          studentEmail: 'student@example.com', // Default email for development
          totalMarks: sub.totalMarks,
          percentage: sub.percentage,
          submittedAt: sub.submittedAt,
          duration: sub.duration,
          answers: sub.answers
        })),
        statistics: stats
      }
    });

  } catch (error) {
    console.error('Get quick test results error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch results',
      error: error.message
    });
  }
});

// Delete quick test
router.delete('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: getValidObjectId(req.user._id)
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status === 'active') {
      return res.status(400).json({
        success: false,
        message: 'Cannot delete active quick test'
      });
    }

    await QuickTest.findByIdAndDelete(testId);

    res.json({
      success: true,
      message: 'Quick test deleted successfully'
    });

  } catch (error) {
    console.error('Delete quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete quick test',
      error: error.message
    });
  }
});

export default router;