import express from 'express';
import mongoose from 'mongoose';
import Homework from '../../models/teacher/Homework.js';
import multer from 'multer';
import fs from 'fs';
import crypto from 'crypto';
import { fileURLToPath } from 'url';
import path from 'path';
import { createNotification } from '../../utils/notificationHelper.js';
import moment from 'moment';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);


const router = express.Router();

// Helper function to convert development IDs to valid ObjectIds
const getValidObjectId = (id) => {
  if (mongoose.Types.ObjectId.isValid(id)) {
    return id;
  } else if (typeof id === 'string' && id.startsWith('dev-')) {
    const hash = crypto.createHash('md5').update(id).digest('hex');
    const objectIdHex = hash.substring(0, 24);
    return new mongoose.Types.ObjectId(objectIdHex);
  }
  return id;
};

// Configure multer for file uploads
// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadPath = path.join(__dirname, '../../uploads/homework');
    if (!fs.existsSync(uploadPath)) {
      fs.mkdirSync(uploadPath, { recursive: true });
    }
    cb(null, uploadPath);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 10 * 1024 * 1024 // 10MB limit
  },
  fileFilter: function (req, file, cb) {
    const allowedTypes = /jpeg|jpg|png|gif|pdf|doc|docx|txt|zip|rar/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Only images, documents, and archive files are allowed'));
    }
  }
});

// Get all homework for teacher with filters
router.get('/', async (req, res) => {
  try {
    const { status = 'all', subject = 'all', studentId = 'all', limit = 50, page = 1 } = req.query;
    const teacherId = req.user._id;


    // Build query with teacherId as base
    const query = { teacherId };

    if (status !== 'all') {
      query.status = status;
    }

    if (subject !== 'all') {
      query.subject = subject;
    }

    if (studentId !== 'all') {
      query.studentId = getValidObjectId(studentId);
    }


    const skip = (page - 1) * limit;

    const homework = await Homework.find(query)
      .populate('studentId', 'studentName email class')
      .populate('teacherId', 'name')
      .sort({ createdAt: -1 })
      .limit(parseInt(limit))
      .skip(skip);


    const total = await Homework.countDocuments(query);

    // Format homework for frontend
    const formattedHomework = homework.map(hw => {
      const now = new Date();
      const isOverdue = hw.status === 'assigned' && hw.dueDate < now;

      return {
        _id: hw._id,
        title: hw.title,
        description: hw.description,
        subject: hw.subject,
        instructions: hw.instructions,
        teacherId: hw.teacherId,
        studentId: hw.studentId,
        dueDate: hw.dueDate,
        points: hw.points,
        status: isOverdue ? 'overdue' : hw.status,
        attachment: hw.attachment,
        submission: hw.submission,
        createdAt: hw.createdAt,
        updatedAt: hw.updatedAt,
        isOverdue,
        canSubmit: hw.canSubmit(),
        isSubmitted: hw.isSubmitted(),
        isGraded: hw.isGraded()
      };
    });

    res.json({
      success: true,
      data: formattedHomework,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get teacher homework error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework',
      error: error.message
    });
  }
});

// Get specific homework by ID
router.get('/:homeworkId', async (req, res) => {
  try {
    const { homeworkId } = req.params;
    const teacherId = req.user._id;

    const homework = await Homework.findOne({
      _id: homeworkId,
      teacherId
    }).populate('studentId', 'name email class')
      .populate('teacherId', 'name');

    if (!homework) {
      return res.status(404).json({
        success: false,
        message: 'Homework not found'
      });
    }

    const now = new Date();
    const isOverdue = homework.status === 'assigned' && homework.dueDate < now;

    const formattedHomework = {
      _id: homework._id,
      title: homework.title,
      description: homework.description,
      subject: homework.subject,
      instructions: homework.instructions,
      teacherId: homework.teacherId,
      studentId: homework.studentId,
      dueDate: homework.dueDate,
      points: homework.points,
      status: isOverdue ? 'overdue' : homework.status,
      attachment: homework.attachment,
      submission: homework.submission,
      createdAt: homework.createdAt,
      updatedAt: homework.updatedAt,
      isOverdue,
      canSubmit: homework.canSubmit(),
      isSubmitted: homework.isSubmitted(),
      isGraded: homework.isGraded()
    };

    res.json({
      success: true,
      data: formattedHomework
    });

  } catch (error) {
    console.error('Get homework by ID error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework',
      error: error.message
    });
  }
});

// Create new homework
router.post('/', upload.single('attachment'), async (req, res) => {
  try {
    const {
      title,
      description,
      subject,
      studentId,
      dueDate,
      points = 10,
      instructions
    } = req.body;
        console.log('in create homework',req.body)

    if (!title || !description || !subject || !studentId || !dueDate) {
      return res.status(400).json({
        success: false,
        message: 'Required fields: title, description, subject, studentId, dueDate'
      });
    }

    const teacherId = req.user._id;

    const homeworkData = {
      title,
      description,
      subject,
      teacherId,
      studentId: getValidObjectId(studentId),
      dueDate: new Date(dueDate),
      points: parseInt(points),
      instructions,
      status: 'assigned'
    };

    // Handle file attachment
    if (req.file) {
      homeworkData.attachment = {
        filename: req.file.filename,
        originalName: req.file.originalname,
        path: req.file.path,
        size: req.file.size,
        mimetype: req.file.mimetype
      };
    }

    const homework = new Homework(homeworkData);
    await homework.save();

    // Populate the student and teacher info for response
    await homework.populate('studentId', 'name email class');
    await homework.populate('teacherId', 'name');

    // Create notification for student
    const io = req.app.get('io');
        console.log('in create homework',homework,io)

    if (homework.studentId && io) {
      console.log('creating notifiacation for student',homework.studentId._id,io)
      await createNotification({
        userId: homework.studentId._id,
        userType: 'student',
        type: 'homework_assigned',
        title: 'New Homework Assigned',
        message: `New ${homework.subject} homework assigned: ${homework.title}. Due: ${moment(homework.dueDate).format('MMM DD, YYYY')}`,
        relatedId: homework._id,
        relatedModel: 'Homework',
        priority: 'high',
        actionUrl: '/student/homework',
        io
      });
    }

    res.status(201).json({
      success: true,
      message: 'Homework created successfully',
      data: homework
    });

  } catch (error) {
    console.error('Create homework error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create homework',
      error: error.message
    });
  }
});

// Update homework
router.put('/:homeworkId', upload.single('attachment'), async (req, res) => {
  try {
    const { homeworkId } = req.params;
    const teacherId = req.user._id;

    const homework = await Homework.findOne({
      _id: homeworkId,
      teacherId
    });

    if (!homework) {
      return res.status(404).json({
        success: false,
        message: 'Homework not found'
      });
    }

    // Check if homework has submissions - restrict editing if submitted
    if (homework.isSubmitted()) {
      return res.status(400).json({
        success: false,
        message: 'Cannot edit homework that has been submitted'
      });
    }

    const {
      title,
      description,
      subject,
      dueDate,
      points,
      instructions
    } = req.body;

    // Update fields
    if (title) homework.title = title;
    if (description) homework.description = description;
    if (subject) homework.subject = subject;
    if (dueDate) homework.dueDate = new Date(dueDate);
    if (points) homework.points = parseInt(points);
    if (instructions !== undefined) homework.instructions = instructions;

    // Handle new file attachment
    if (req.file) {
      // Remove old file if exists
      if (homework.attachment && homework.attachment.path) {
        try {
          fs.unlinkSync(homework.attachment.path);
        } catch (err) {
          console.warn('Could not delete old file:', homework.attachment.path);
        }
      }

      homework.attachment = {
        filename: req.file.filename,
        originalName: req.file.originalname,
        path: req.file.path,
        size: req.file.size,
        mimetype: req.file.mimetype
      };
    }

    await homework.save();

    // Populate for response
    await homework.populate('studentId', 'name email class');
    await homework.populate('teacherId', 'name');

    res.json({
      success: true,
      message: 'Homework updated successfully',
      data: homework
    });

  } catch (error) {
    console.error('Update homework error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update homework',
      error: error.message
    });
  }
});

// Delete homework
router.delete('/:homeworkId', async (req, res) => {
  try {
    const { homeworkId } = req.params;
    const teacherId = req.user._id;

    const homework = await Homework.findOne({
      _id: homeworkId,
      teacherId
    });

    if (!homework) {
      return res.status(404).json({
        success: false,
        message: 'Homework not found'
      });
    }

    // Delete associated question attachment file
    if (homework.attachment && homework.attachment.path) {
      try {
        fs.unlinkSync(homework.attachment.path);
      } catch (err) {
        console.warn('Could not delete question file:', homework.attachment.path);
      }
    }

    // Delete submission attachment file if exists
    if (homework.submission && homework.submission.attachment && homework.submission.attachment.path) {
      try {
        fs.unlinkSync(homework.submission.attachment.path);
      } catch (err) {
        console.warn('Could not delete submission file:', homework.submission.attachment.path);
      }
    }

    await Homework.findByIdAndDelete(homeworkId);

    res.json({
      success: true,
      message: 'Homework deleted successfully'
    });

  } catch (error) {
    console.error('Delete homework error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete homework',
      error: error.message
    });
  }
});

// Grade homework submission
router.post('/:homeworkId/grade', async (req, res) => {
  try {
    const { homeworkId } = req.params;
    const { grade, feedback, corrections } = req.body;
    const teacherId = req.user._id;

    if (grade === undefined || grade === null) {
      return res.status(400).json({
        success: false,
        message: 'Grade is required'
      });
    }

    const homework = await Homework.findOne({
      _id: homeworkId,
      teacherId
    }).populate('studentId', 'name email class');

    if (!homework) {
      return res.status(404).json({
        success: false,
        message: 'Homework not found'
      });
    }

    if (!homework.isSubmitted()) {
      return res.status(400).json({
        success: false,
        message: 'Cannot grade homework that has not been submitted'
      });
    }

    if (grade < 0 || grade > homework.points) {
      return res.status(400).json({
        success: false,
        message: `Grade must be between 0 and ${homework.points}`
      });
    }

    await homework.gradeHomework({
      grade: parseFloat(grade),
      feedback,
      corrections
    }, teacherId);

    // Create notification for student
    const io = req.app.get('io');
    if (homework.studentId && io) {
      const percentage = Math.round((parseFloat(grade) / homework.points) * 100);
      await createNotification({
        userId: homework.studentId._id,
        userType: 'student',
        type: 'homework_graded',
        title: 'Homework Graded',
        message: `Your ${homework.subject} homework "${homework.title}" has been graded. Score: ${grade}/${homework.points} (${percentage}%)`,
        relatedId: homework._id,
        relatedModel: 'Homework',
        priority: 'medium',
        actionUrl: '/student/homework',
        io
      });
    }

    res.json({
      success: true,
      message: 'Homework graded successfully',
      data: {
        homeworkId: homework._id,
        studentName: homework.studentId.name,
        grade: homework.submission.grade,
        totalPoints: homework.points,
        percentage: Math.round((homework.submission.grade / homework.points) * 100),
        feedback: homework.submission.feedback,
        corrections: homework.submission.corrections,
        gradedAt: homework.submission.gradedAt
      }
    });

  } catch (error) {
    console.error('Grade homework error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to grade homework',
      error: error.message
    });
  }
});

// Get homework statistics for teacher dashboard
router.get('/stats/overview', async (req, res) => {
  try {
    const teacherId = req.user._id;
    const now = new Date();

    const stats = await Promise.all([
      // Total homework count
      Homework.countDocuments({ teacherId }),

      // Active homework (assigned and not overdue)
      Homework.countDocuments({
        teacherId,
        status: 'assigned',
        dueDate: { $gte: now }
      }),

      // Submitted homework waiting for review
      Homework.countDocuments({
        teacherId,
        status: 'submitted'
      }),

      // Overdue homework
      Homework.countDocuments({
        teacherId,
        $or: [
          { status: 'overdue' },
          { status: 'assigned', dueDate: { $lt: now } }
        ]
      }),

      // Graded homework
      Homework.countDocuments({
        teacherId,
        status: 'graded'
      })
    ]);

    const [total, active, pendingReview, overdue, graded] = stats;

    res.json({
      success: true,
      data: {
        total,
        active,
        pendingReview,
        overdue,
        graded,
        completionRate: total > 0 ? Math.round((graded / total) * 100) : 0
      }
    });

  } catch (error) {
    console.error('Get homework stats error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework statistics',
      error: error.message
    });
  }
});

export default router;