const express = require('express');
const Class = require('../../models/Class');
const Student = require('../../models/student/Student');
const Teacher = require('../../models/teacher/Teacher');
const { studentAuth } = require('../../middleware/auth');

// Try to load Invoice model, fallback if not available
let Invoice;
try {
  Invoice = require('../../models/admin/Invoice');
} catch (error) {
  console.warn('Invoice model not available, using fallback');
  Invoice = { countDocuments: () => Promise.resolve(0) };
}

const router = express.Router();

router.get('/dashboard', studentAuth, async (req, res) => {
  try {
    res.json({
        student: {
          studentName: 'Development Student',
          email: 'student@edumetrix.uk',
          grade: '10th Grade',
          subjects: ['Mathematics', 'Physics', 'Chemistry'],
          assignedTeachersCount: 3
        },
        todaysClasses: [
          {
            _id: 'class1',
            subject: 'Mathematics',
            scheduledDate: new Date(),
            duration: 60,
            teacher: { name: 'Mr. Smith' },
            status: 'scheduled'
          },
          {
            _id: 'class2',
            subject: 'Physics', 
            scheduledDate: new Date(Date.now() + 3 * 60 * 60 * 1000),
            duration: 90,
            teacher: { name: 'Ms. Johnson' },
            status: 'scheduled'
          }
        ],
        upcomingClasses: [
          {
            _id: 'class3',
            subject: 'Chemistry',
            scheduledDate: new Date(Date.now() + 24 * 60 * 60 * 1000),
            duration: 75,
            teacher: { name: 'Dr. Brown' }
          }
        ],
        monthlyStats: {
          classes: 22,
          attendance: 95,
          assignments: 8,
          pendingInvoices: 2
        },
        recentAssignments: [
          { title: 'Algebra Problems', subject: 'Mathematics', dueDate: new Date(Date.now() + 3 * 24 * 60 * 60 * 1000), status: 'pending' },
          { title: 'Physics Lab Report', subject: 'Physics', dueDate: new Date(Date.now() + 5 * 24 * 60 * 60 * 1000), status: 'in_progress' }
        ]
      });
  } catch (error) {
    console.error('Error fetching student dashboard:', error);
    res.status(500).json({ message: 'Internal server error' });
  }
});

router.get('/dashboard-old', studentAuth, async (req, res) => {
  try {
    // Use the student data already available in req.user from middleware
    const student = req.user;

    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const today = new Date();
    const startOfDay = new Date(today.setHours(0, 0, 0, 0));
    const endOfDay = new Date(today.setHours(23, 59, 59, 999));

    const todaysClasses = await Class.find({
      studentId: student._id,
      scheduledDate: { $gte: startOfDay, $lte: endOfDay },
      status: { $in: ['scheduled', 'ongoing'] }
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    const upcomingClasses = await Class.find({
      studentId: student._id,
      scheduledDate: { $gt: endOfDay },
      status: 'scheduled'
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 })
    .limit(5);

    const currentMonth = new Date();
    const startOfMonth = new Date(currentMonth.getFullYear(), currentMonth.getMonth(), 1);
    const endOfMonth = new Date(currentMonth.getFullYear(), currentMonth.getMonth() + 1, 0);

    const monthlyClasses = await Class.countDocuments({
      studentId: student._id,
      scheduledDate: { $gte: startOfMonth, $lte: endOfMonth },
      status: 'completed'
    });

    const pendingInvoices = await Invoice.countDocuments({
      studentId: student._id,
      status: { $in: ['sent', 'overdue'] }
    });

    // Get unique teachers from classes
    const uniqueTeachers = await Class.aggregate([
      { $match: { studentId: student._id, status: { $in: ['scheduled', 'ongoing', 'completed'] } } },
      { $group: { _id: '$teacherId' } },
      { $count: 'total' }
    ]);

    const totalTeachers = uniqueTeachers.length > 0 ? uniqueTeachers[0].total : 0;

    res.json({
      student: {
        studentName: student.studentName,
        email: student.email,
        grade: student.class,
        subjects: student.subjects,
        assignedTeachersCount: totalTeachers
      },
      todaysClasses: todaysClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        duration: cls.duration,
        teacher: cls.teacherId,
        status: cls.status
      })),
      upcomingClasses: upcomingClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        duration: cls.duration,
        teacher: cls.teacherId
      })),
      monthlyStats: {
        classes: monthlyClasses,
        attendance: monthlyClasses > 0 ? Math.round((monthlyClasses / (monthlyClasses + 1)) * 100) : 100,
        assignments: await Class.countDocuments({
          studentId: student._id,
          homeworkAssigned: { $ne: null, $ne: '' },
          scheduledDate: { $gte: startOfMonth, $lte: endOfMonth }
        }),
        pendingInvoices
      },
      recentAssignments: (await Class.find({
        studentId: student._id,
        homeworkAssigned: { $ne: null, $ne: '' },
        scheduledDate: { $gte: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000) }
      })
      .populate('teacherId', 'name')
      .sort({ scheduledDate: -1 })
      .limit(5)
      .select('subject homeworkAssigned scheduledDate teacherId'))
      .map(cls => ({
        title: cls.homeworkAssigned.substring(0, 50) + (cls.homeworkAssigned.length > 50 ? '...' : ''),
        subject: cls.subject,
        dueDate: new Date(cls.scheduledDate.getTime() + 7 * 24 * 60 * 60 * 1000), // Due 7 days after class
        status: 'pending',
        teacherName: cls.teacherId?.name
      }))
    });
  } catch (error) {
    console.error('Student dashboard error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/classes', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        classes: [
          {
            _id: 'class1',
            subject: 'Mathematics',
            scheduledDate: new Date(),
            duration: 60,
            teacherId: { name: 'Mr. Smith' },
            status: 'scheduled'
          },
          {
            _id: 'class2',
            subject: 'Physics',
            scheduledDate: new Date(Date.now() + 24 * 60 * 60 * 1000),
            duration: 90,
            teacherId: { name: 'Ms. Johnson' },
            status: 'completed'
          }
        ],
        totalPages: 1,
        currentPage: 1,
        total: 2
      });
    }

    // For real students, the authentication middleware already merged the student data into req.user
    const student = req.user;
    
    if (!student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { page = 1, limit = 10, status = 'all', subject = 'all', date } = req.query;
    
    let query = { studentId: student._id };
    
    if (status !== 'all') {
      query.status = status;
    }
    
    if (subject !== 'all') {
      query.subject = subject;
    }
    
    if (date) {
      const startDate = new Date(date);
      const endDate = new Date(date);
      endDate.setDate(endDate.getDate() + 1);
      query.scheduledDate = { $gte: startDate, $lt: endDate };
    }

    const classes = await Class.find(query)
      .populate('teacherId', 'name')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ scheduledDate: -1 });

    const total = await Class.countDocuments(query);

    res.json({
      classes,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get student classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/teachers', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        teachers: [
          {
            teacher: {
              _id: 'dev-teacher-123',
              name: 'Mr. Smith',
              email: 'smith@edumetrix.uk',
              subjects: ['Mathematics'],
              availableTime: '10:00-16:00',
              availableDays: ['Monday', 'Wednesday', 'Friday']
            },
            subject: 'Mathematics',
            canChat: true
          },
          {
            teacher: {
              _id: 'dev-teacher-456',
              name: 'Ms. Johnson',
              email: 'johnson@edumetrix.uk',
              subjects: ['Physics'],
              availableTime: '12:00-18:00',
              availableDays: ['Tuesday', 'Thursday']
            },
            subject: 'Physics',
            canChat: true
          }
        ]
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;

    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    // Get teachers from both assigned teachers array AND from scheduled classes
    const populatedStudent = await Student.findById(student._id)
      .populate({
        path: 'assignedTeachers.teacherId',
        select: 'name email subjects availableTime availableDays'
      });

    // Get teachers from scheduled classes
    const classTeachers = await Class.find({
      studentId: student._id,
      status: { $in: ['scheduled', 'ongoing', 'completed'] }
    })
    .populate('teacherId', 'name email subjects availableTime availableDays')
    .select('teacherId subject');

    // Combine teachers from both sources
    const teacherMap = new Map();
    
    // Add assigned teachers first
    if (populatedStudent && populatedStudent.assignedTeachers) {
      populatedStudent.assignedTeachers.forEach(assignment => {
        if (assignment.teacherId) {
          const key = `${assignment.teacherId._id}-${assignment.subject}`;
          teacherMap.set(key, {
            teacher: assignment.teacherId,
            subject: assignment.subject,
            canChat: true,
            source: 'assigned'
          });
        }
      });
    }

    // Add teachers from classes (this will override assigned teachers if same teacher-subject combo)
    classTeachers.forEach(classDoc => {
      if (classDoc.teacherId) {
        const key = `${classDoc.teacherId._id}-${classDoc.subject}`;
        teacherMap.set(key, {
          teacher: classDoc.teacherId,
          subject: classDoc.subject,
          canChat: true,
          source: 'class'
        });
      }
    });

    const teachersWithSubjects = Array.from(teacherMap.values());

    res.json({ teachers: teachersWithSubjects });
  } catch (error) {
    console.error('Get student teachers error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/invoices', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        invoices: [
          {
            _id: 'invoice1',
            invoiceNumber: 'INV-2025-001',
            amount: 500,
            currency: 'AED',
            status: 'sent',
            dueDate: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000),
            createdAt: new Date(),
            items: [{ description: 'Mathematics Classes', quantity: 4, rate: 125 }]
          },
          {
            _id: 'invoice2',
            invoiceNumber: 'INV-2025-002',
            amount: 750,
            currency: 'AED',
            status: 'paid',
            dueDate: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000),
            createdAt: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000),
            items: [{ description: 'Physics Classes', quantity: 6, rate: 125 }]
          }
        ],
        totalPages: 1,
        currentPage: 1,
        total: 2
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { page = 1, limit = 10, status = 'all' } = req.query;
    
    let query = { studentId: student._id };
    if (status !== 'all') {
      query.status = status;
    }

    const invoices = await Invoice.find(query)
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ createdAt: -1 });

    const total = await Invoice.countDocuments(query);

    res.json({
      invoices,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get student invoices error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/invoices/:id', studentAuth, async (req, res) => {
  try {
    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const invoice = await Invoice.findOne({
      _id: req.params.id,
      studentId: student._id
    });

    if (!invoice) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    res.json({ invoice });
  } catch (error) {
    console.error('Get student invoice error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/profile', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        student: {
          _id: 'dev-student-profile-123',
          studentName: 'Development Student',
          parentName: 'Development Parent',
          email: 'student@edumetrix.uk',
          phoneNumber: {
            countryCode: '+971',
            number: '501234567'
          },
          profilePicture: null,
          hourlyRate: 50,
          currency: 'AED',
          country: 'UAE',
          syllabus: 'CBSE',
          class: 'Class 10',
          subjects: ['Mathematics', 'Physics', 'Chemistry', 'Biology'],
          googleMeetLink: 'https://meet.google.com/dev-link',
          assignedTeachers: [
            {
              teacherId: {
                _id: 'dev-teacher-123',
                name: 'Mr. Smith'
              },
              subject: 'Mathematics'
            },
            {
              teacherId: {
                _id: 'dev-teacher-456',
                name: 'Ms. Johnson'
              },
              subject: 'Physics'
            }
          ],
          userId: {
            username: 'student@edumetrix.uk'
          }
        }
      });
    }

    // For real students, the authentication middleware already merged the student data into req.user
    const student = req.user;
    
    if (!student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    res.json({ student });
  } catch (error) {
    console.error('Get student profile error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/homework', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        homework: [
          {
            _id: 'homework1',
            subject: 'Mathematics',
            homeworkAssigned: 'Complete exercises 1-10 from Chapter 5',
            scheduledDate: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000),
            teacherId: { name: 'Mr. Smith' },
            classNotes: 'Focus on quadratic equations'
          },
          {
            _id: 'homework2',
            subject: 'Physics',
            homeworkAssigned: 'Write lab report on motion experiments',
            scheduledDate: new Date(Date.now() - 5 * 24 * 60 * 60 * 1000),
            teacherId: { name: 'Ms. Johnson' },
            classNotes: 'Include graphs and calculations'
          }
        ],
        totalPages: 1,
        currentPage: 1,
        total: 2
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { page = 1, limit = 10, subject = 'all' } = req.query;
    
    let query = { 
      studentId: student._id,
      homeworkAssigned: { $ne: null, $ne: '' },
      status: 'completed'
    };
    
    if (subject !== 'all') {
      query.subject = subject;
    }

    const homework = await Class.find(query)
      .populate('teacherId', 'name')
      .select('subject homeworkAssigned scheduledDate teacherId classNotes')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ scheduledDate: -1 });

    const total = await Class.countDocuments(query);

    res.json({
      homework,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get student homework error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/join-class/:classId', studentAuth, async (req, res) => {
  try {
    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const classDoc = await Class.findOne({
      _id: req.params.classId,
      studentId: student._id
    })
    .populate('teacherId', 'name');

    if (!classDoc) {
      return res.status(404).json({ message: 'Class not found' });
    }

    const now = new Date();
    const classTime = new Date(classDoc.scheduledDate);
    const timeDiff = Math.abs(now - classTime) / 60000; // difference in minutes

    if (timeDiff > 15) {
      return res.status(400).json({ 
        message: 'Class can only be joined 15 minutes before or after scheduled time' 
      });
    }

    res.json({
      class: {
        id: classDoc._id,
        subject: classDoc.subject,
        teacher: classDoc.teacherId.name,
        scheduledDate: classDoc.scheduledDate,
        meetingLink: classDoc.meetingLink || student.googleMeetLink,
        duration: classDoc.duration
      }
    });
  } catch (error) {
    console.error('Join class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/attendance-report', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        month: '8/2025',
        summary: {
          totalClasses: 20,
          attendedClasses: 18,
          missedClasses: 1,
          pendingClasses: 1,
          attendancePercentage: '90.00'
        },
        classes: [
          {
            date: '2025-08-01',
            subject: 'Mathematics',
            teacher: 'Mr. Smith',
            status: 'completed',
            attended: true,
            notes: 'Good progress on algebra',
            homework: 'Practice problems 1-20'
          },
          {
            date: '2025-08-03',
            subject: 'Physics',
            teacher: 'Ms. Johnson',
            status: 'completed',
            attended: false,
            notes: null,
            homework: null
          }
        ]
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { month, year } = req.query;
    
    const currentDate = new Date();
    const reportMonth = month || (currentDate.getMonth() + 1);
    const reportYear = year || currentDate.getFullYear();

    const startDate = new Date(reportYear, reportMonth - 1, 1);
    const endDate = new Date(reportYear, reportMonth, 0);

    const classes = await Class.find({
      studentId: student._id,
      scheduledDate: { $gte: startDate, $lte: endDate }
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    const totalClasses = classes.length;
    const attendedClasses = classes.filter(cls => cls.attendanceStatus?.studentAttended === true).length;
    const missedClasses = classes.filter(cls => cls.attendanceStatus?.studentAttended === false).length;
    const pendingClasses = classes.filter(cls => cls.status === 'scheduled').length;

    const attendancePercentage = totalClasses > 0 ? ((attendedClasses / totalClasses) * 100).toFixed(2) : 0;

    const classDetails = classes.map(cls => ({
      date: cls.scheduledDate.toISOString().split('T')[0],
      subject: cls.subject,
      teacher: cls.teacherId.name,
      status: cls.status,
      attended: cls.attendanceStatus?.studentAttended,
      notes: cls.classNotes,
      homework: cls.homeworkAssigned
    }));

    res.json({
      month: `${reportMonth}/${reportYear}`,
      summary: {
        totalClasses,
        attendedClasses,
        missedClasses,
        pendingClasses,
        attendancePercentage
      },
      classes: classDetails
    });
  } catch (error) {
    console.error('Get student attendance report error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/calendar', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        classes: [
          {
            _id: 'class1',
            subject: 'Mathematics',
            scheduledDate: new Date(),
            duration: 60,
            teacherId: { name: 'Mr. Smith' },
            status: 'scheduled',
            meetingLink: 'https://meet.google.com/dev-math-class'
          },
          {
            _id: 'class2',
            subject: 'Physics',
            scheduledDate: new Date(Date.now() + 24 * 60 * 60 * 1000),
            duration: 90,
            teacherId: { name: 'Ms. Johnson' },
            status: 'scheduled',
            meetingLink: 'https://meet.google.com/dev-physics-class'
          }
        ],
        view: req.query.view || 'month',
        startDate: new Date().toISOString().split('T')[0],
        endDate: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { view = 'month', date = new Date(), timezone = 'UTC' } = req.query;
    
    let startDate, endDate;
    const moment = require('moment-timezone');
    const currentDate = moment.tz(date, timezone);

    switch (view) {
      case 'day':
        startDate = currentDate.clone().startOf('day');
        endDate = currentDate.clone().endOf('day');
        break;
      case 'week':
        startDate = currentDate.clone().startOf('week');
        endDate = currentDate.clone().endOf('week');
        break;
      case 'month':
        startDate = currentDate.clone().startOf('month');
        endDate = currentDate.clone().endOf('month');
        break;
      default:
        startDate = currentDate.clone().startOf('month');
        endDate = currentDate.clone().endOf('month');
    }

    const classes = await Class.find({
      studentId: student._id,
      scheduledDate: {
        $gte: startDate.toDate(),
        $lte: endDate.toDate()
      },
      status: { $in: ['scheduled', 'ongoing', 'completed'] }
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    const formattedClasses = classes.map(cls => ({
      ...cls.toObject(),
      formattedDate: moment.tz(cls.scheduledDate, timezone).format('YYYY-MM-DD HH:mm'),
      localTime: moment.tz(cls.scheduledDate, timezone).format('HH:mm'),
      meetingLink: cls.meetingLink || student.googleMeetLink
    }));

    res.json({
      classes: formattedClasses,
      view,
      startDate: startDate.format('YYYY-MM-DD'),
      endDate: endDate.format('YYYY-MM-DD')
    });
  } catch (error) {
    console.error('Get student calendar error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/chat/users', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        users: [
          {
            _id: 'dev-teacher-123',
            username: 'teacher@edumetrix.uk',
            profile: {
              name: 'Mr. Smith',
              subject: 'Mathematics'
            },
            isOnline: true,
            unreadCount: 2
          },
          {
            _id: 'dev-teacher-456',
            username: 'teacher2@edumetrix.uk',
            profile: {
              name: 'Ms. Johnson',
              subject: 'Physics'
            },
            isOnline: false,
            unreadCount: 0
          }
        ]
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    // Find all teachers assigned to this student
    const populatedStudent = await Student.findById(student._id)
      .populate({
        path: 'assignedTeachers.teacherId',
        select: 'name email subjects userId',
        populate: {
          path: 'userId',
          select: 'username isActive'
        }
      });

    if (!populatedStudent) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const teachers = populatedStudent.assignedTeachers
      .filter(assignment => assignment.teacherId && assignment.teacherId.userId && assignment.teacherId.userId.isActive)
      .map(assignment => ({
        _id: assignment.teacherId._id,
        username: assignment.teacherId.userId.username,
        profile: {
          name: assignment.teacherId.name,
          subject: assignment.subject
        },
        isOnline: true, // You can implement real-time status later
        unreadCount: 0  // You can implement unread message counts later
      }));

    res.json({ users: teachers });
  } catch (error) {
    console.error('Get student chat users error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/chat/:chatId/messages', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        messages: [
          {
            _id: 'msg1',
            senderId: 'dev-teacher-123',
            content: 'Hello! How are you doing with your math homework?',
            timestamp: new Date(Date.now() - 2 * 60 * 60 * 1000),
            type: 'text'
          },
          {
            _id: 'msg2',
            senderId: 'dev-student-123',
            content: 'Hi! I\'m struggling a bit with quadratic equations. Could you help?',
            timestamp: new Date(Date.now() - 1 * 60 * 60 * 1000),
            type: 'text'
          },
          {
            _id: 'msg3',
            senderId: 'dev-teacher-123',
            content: 'Of course! Let\'s schedule a quick session to go through them together.',
            timestamp: new Date(Date.now() - 30 * 60 * 1000),
            type: 'text'
          }
        ]
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const Chat = require('../../models/admin/Chat');
    const { chatId } = req.params;

    // Find chat where student is a participant and chatId matches teacher ID
    const chat = await Chat.findOne({
      $and: [
        { 'participants.userId': student.userId },
        { 'participants.userId': chatId }
      ]
    }).populate('messages.senderId', 'username');

    if (!chat) {
      return res.json({ messages: [] });
    }

    res.json({ messages: chat.messages });
  } catch (error) {
    console.error('Get student chat messages error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/chat/:chatId/messages', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock response
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        message: {
          _id: 'new-msg-' + Date.now(),
          senderId: 'dev-student-123',
          content: req.body.message,
          timestamp: new Date(),
          type: req.body.type || 'text'
        }
      });
    }

    // Use the student data already available in req.user from middleware
    const student = req.user;
    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const Chat = require('../../models/admin/Chat');
    const { chatId } = req.params;
    const { message, type = 'text' } = req.body;

    if (!message || !message.trim()) {
      return res.status(400).json({ message: 'Message content is required' });
    }

    // Find or create chat between student and teacher
    let chat = await Chat.findOne({
      $and: [
        { 'participants.userId': student.userId },
        { 'participants.userId': chatId }
      ]
    });

    if (!chat) {
      // Create new chat
      chat = new Chat({
        participants: [
          { userId: student.userId, userType: 'student' },
          { userId: chatId, userType: 'teacher' }
        ],
        chatType: 'student-teacher',
        subject: null, // Allow null subject for general chat
        messages: []
      });
    }

    // Add message
    const newMessage = {
      senderId: student.userId,
      senderType: 'student',
      message: message.trim(),
      timestamp: new Date(),
      readBy: [{ userId: student.userId, readAt: new Date() }]
    };

    chat.messages.push(newMessage);
    chat.lastActivity = new Date();
    await chat.save();

    res.json({ 
      message: {
        ...newMessage,
        _id: newMessage._id,
        content: newMessage.message
      }
    });
  } catch (error) {
    console.error('Send student chat message error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;