import express from 'express';
import Class from '../../models/Class.js';
import Student from '../../models/student/Student.js';
import Teacher from '../../models/teacher/Teacher.js';
import Invoice from '../../models/admin/Invoice.js';
import Homework from '../../models/teacher/Homework.js';
import QuickTest from '../../models/teacher/QuickTest.js';
import { studentAuth } from '../../middleware/auth.js';
import moment from 'moment-timezone';
import Chat from '../../models/admin/Chat.js';
import RescheduleRequest from '../../models/shared/RescheduleRequest.js';

const router = express.Router();

router.get('/dashboard', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    console.log('📚 Student found:', student.studentName);
    console.log('🌍 Student timezone:', student.timezone, '| Country:', student.country);

    // Ensure student has timezone set based on country (fallback for existing students)
    if (!student.timezone && student.country) {
      const countryTimezoneMap = {
        'UAE': 'Asia/Dubai',
        'Saudi Arabia': 'Asia/Riyadh',
        'Qatar': 'Asia/Qatar',
        'Kuwait': 'Asia/Kuwait',
        'Bahrain': 'Asia/Bahrain',
        'Oman': 'Asia/Muscat',
        'India': 'Asia/Kolkata',
        'UK': 'Europe/London',
        'Australia': 'Australia/Sydney',
        'New Zealand': 'Pacific/Auckland'
      };
      student.timezone = countryTimezoneMap[student.country] || 'Asia/Dubai';
      await student.save();
      console.log('✅ Updated student timezone to:', student.timezone);
    }

    // Get today's date range
    const startOfToday = new Date();
    startOfToday.setHours(0, 0, 0, 0);
    const endOfToday = new Date();
    endOfToday.setHours(23, 59, 59, 999);

    // Get tomorrow's start for upcoming classes
    const startOfTomorrow = new Date(startOfToday);
    startOfTomorrow.setDate(startOfTomorrow.getDate() + 1);

    // Get classes for today
    const todaysClasses = await Class.find({
      studentId: studentId,
      scheduledDate: {
        $gte: startOfToday,
        $lte: endOfToday
      },
      isDeleted: false
    })
    .populate('teacherId', 'name email')
    .sort({ scheduledDate: 1 });

    // Get upcoming classes (next 7 days excluding today)
    const next7Days = new Date(startOfTomorrow);
    next7Days.setDate(next7Days.getDate() + 7);

    const upcomingClasses = await Class.find({
      studentId: studentId,
      scheduledDate: {
        $gte: startOfTomorrow,
        $lt: next7Days
      },
      status: { $in: ['scheduled'] },
      isDeleted: false
    })
    .populate('teacherId', 'name email')
    .sort({ scheduledDate: 1 })
    .limit(10);

    // Get current month stats
    const startOfMonth = new Date();
    startOfMonth.setDate(1);
    startOfMonth.setHours(0, 0, 0, 0);

    const endOfMonth = new Date();
    endOfMonth.setMonth(endOfMonth.getMonth() + 1);
    endOfMonth.setDate(0);
    endOfMonth.setHours(23, 59, 59, 999);

    const monthlyClasses = await Class.find({
      studentId: studentId,
      scheduledDate: {
        $gte: startOfMonth,
        $lte: endOfMonth
      },
      isDeleted: false
    });

    // Calculate attendance rate for the month
    const completedClasses = monthlyClasses.filter(c => c.status === 'completed');
    const attendedClasses = completedClasses.filter(c => c.attendance?.studentJoined || c.attendance?.present);
    const attendanceRate = completedClasses.length > 0
      ? Math.round((attendedClasses.length / completedClasses.length) * 100)
      : 0;

    // ONLY get teachers from actual classes (scheduled, ongoing, or completed)
    // This ensures we only show teachers who are actively teaching the student
    const classTeachers = await Class.find({
      studentId: studentId,
      status: { $in: ['scheduled', 'ongoing', 'completed'] },
      isDeleted: false
    })
    .populate('teacherId', 'name')
    .select('teacherId subject');

    // Deduplicate teachers by teacher ID and collect their subjects
    const teacherMap = new Map();
    const teacherSubjectsMap = new Map();
    const allSubjectsSet = new Set(); // Track all unique subjects across all classes

    classTeachers.forEach(classDoc => {
      if (classDoc.teacherId) {
        const teacherId = classDoc.teacherId._id.toString();

        // Track subjects for this teacher
        if (!teacherSubjectsMap.has(teacherId)) {
          teacherSubjectsMap.set(teacherId, new Set());
        }
        teacherSubjectsMap.get(teacherId).add(classDoc.subject);

        // Track all unique subjects
        if (classDoc.subject) {
          allSubjectsSet.add(classDoc.subject);
        }

        // Only set teacher data once
        if (!teacherMap.has(teacherId)) {
          teacherMap.set(teacherId, {
            _id: classDoc.teacherId._id,
            name: classDoc.teacherId.name
          });
        }
      }
    });

    // Get all unique subjects being taught to this student
    const activeSubjects = Array.from(allSubjectsSet);

    // Format for frontend
    const formattedAssignedTeachers = Array.from(teacherMap.entries()).map(([teacherId, teacher]) => {
      const subjects = Array.from(teacherSubjectsMap.get(teacherId) || []);
      return {
        teacherId: {
          _id: teacher._id,
          name: teacher.name
        },
        subjects: subjects,
        subject: subjects.join(', ') || 'N/A'
      };
    });

    const totalTeachers = formattedAssignedTeachers.length;

    console.log('📚 Teachers from classes:', formattedAssignedTeachers);
    console.log('📚 Total teachers:', totalTeachers);
    console.log('📚 Active subjects being taught:', activeSubjects);
    console.log('🕐 Today\'s classes count:', todaysClasses.length);
    console.log('🕐 Upcoming classes count:', upcomingClasses.length);
    if (todaysClasses.length > 0) {
      console.log('🕐 First today class timezone:', todaysClasses[0].timezone);
    }
    if (upcomingClasses.length > 0) {
      console.log('🕐 First upcoming class timezone:', upcomingClasses[0].timezone);
    }

    // Count pending homework (assigned or overdue, not submitted)
    const pendingHomework = await Homework.countDocuments({
      studentId: studentId,
      status: { $in: ['assigned', 'overdue'] }
    });
    console.log('📚 Pending homework count:', pendingHomework);

    // Count available quick tests (active tests that student hasn't submitted)
    const availableTests = await QuickTest.countDocuments({
      studentId: studentId,
      status: 'active',
      'submissions.studentId': { $ne: studentId }
    });
    console.log('📝 Available quick tests count:', availableTests);

    // Get total invoices count (only enabled invoices)
    let totalInvoices = 0;
    try {
      const InvoiceModel = (await import('../../models/admin/Invoice.js')).default;
      totalInvoices = await InvoiceModel.countDocuments({ studentId: studentId, enabled: true });
    } catch (error) {
      console.log('Invoice model not available:', error.message);
    }

    res.json({
      student: {
        name: student.studentName,
        email: student.email,
        class: student.class,
        subjects: activeSubjects.length > 0 ? activeSubjects : (student.subjects || []),
        timezone: student.timezone,
        country: student.country,
        assignedTeachers: formattedAssignedTeachers
      },
      todaysClasses: todaysClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        startTime: cls.startTime,
        endTime: cls.endTime,
        duration: cls.duration,
        timezone: cls.timezone,
        teacherId: cls.teacherId,
        teacher: cls.teacherId,
        status: cls.status,
        googleMeetLink: cls.googleMeetLink,
        room: cls.room
      })),
      upcomingClasses: upcomingClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        startTime: cls.startTime,
        endTime: cls.endTime,
        duration: cls.duration,
        timezone: cls.timezone,
        teacherId: cls.teacherId,
        teacher: cls.teacherId,
        status: cls.status
      })),
      stats: {
        monthlyClasses: monthlyClasses.length,
        completedClasses: completedClasses.length,
        attendanceRate: attendanceRate,
        totalTeachers: totalTeachers,
        pendingHomework: pendingHomework,
        availableTests: availableTests,
        totalInvoices: totalInvoices
      }
    });
  } catch (error) {
    console.error('Error fetching student dashboard:', error);
    res.status(500).json({ message: 'Internal server error', error: error.message });
  }
});

router.get('/dashboard-old', studentAuth, async (req, res) => {
  try {
    // Use the student data already available in req.user from middleware
    const student = req.user;

    if (!student || !student.studentName) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const today = new Date();
    const startOfDay = new Date(today.setHours(0, 0, 0, 0));
    const endOfDay = new Date(today.setHours(23, 59, 59, 999));

    const todaysClasses = await Class.find({
      studentId: student._id,
      scheduledDate: { $gte: startOfDay, $lte: endOfDay },
      status: { $in: ['scheduled', 'ongoing'] }
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    const upcomingClasses = await Class.find({
      studentId: student._id,
      scheduledDate: { $gt: endOfDay },
      status: 'scheduled'
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 })
    .limit(5);

    const currentMonth = new Date();
    const startOfMonth = new Date(currentMonth.getFullYear(), currentMonth.getMonth(), 1);
    const endOfMonth = new Date(currentMonth.getFullYear(), currentMonth.getMonth() + 1, 0);

    const monthlyClasses = await Class.countDocuments({
      studentId: student._id,
      scheduledDate: { $gte: startOfMonth, $lte: endOfMonth },
      status: 'completed'
    });

    const pendingInvoices = await Invoice.countDocuments({
      studentId: student._id,
      status: { $in: ['sent', 'overdue'] },
      enabled: true
    });

    // Get unique teachers from classes
    const uniqueTeachers = await Class.aggregate([
      { $match: { studentId: student._id, status: { $in: ['scheduled', 'ongoing', 'completed'] } } },
      { $group: { _id: '$teacherId' } },
      { $count: 'total' }
    ]);

    const totalTeachers = uniqueTeachers.length > 0 ? uniqueTeachers[0].total : 0;

    res.json({
      student: {
        studentName: student.studentName,
        email: student.email,
        grade: student.class,
        subjects: student.subjects,
        assignedTeachersCount: totalTeachers
      },
      todaysClasses: todaysClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        duration: cls.duration,
        teacher: cls.teacherId,
        status: cls.status
      })),
      upcomingClasses: upcomingClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        duration: cls.duration,
        teacher: cls.teacherId
      })),
      monthlyStats: {
        classes: monthlyClasses,
        attendance: monthlyClasses > 0 ? Math.round((monthlyClasses / (monthlyClasses + 1)) * 100) : 100,
        assignments: await Class.countDocuments({
          studentId: student._id,
          homeworkAssigned: { $ne: null, $ne: '' },
          scheduledDate: { $gte: startOfMonth, $lte: endOfMonth }
        }),
        pendingInvoices
      },
      recentAssignments: (await Class.find({
        studentId: student._id,
        homeworkAssigned: { $ne: null, $ne: '' },
        scheduledDate: { $gte: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000) }
      })
      .populate('teacherId', 'name')
      .sort({ scheduledDate: -1 })
      .limit(5)
      .select('subject homeworkAssigned scheduledDate teacherId'))
      .map(cls => ({
        title: cls.homeworkAssigned.substring(0, 50) + (cls.homeworkAssigned.length > 50 ? '...' : ''),
        subject: cls.subject,
        dueDate: new Date(cls.scheduledDate.getTime() + 7 * 24 * 60 * 60 * 1000), // Due 7 days after class
        status: 'pending',
        teacherName: cls.teacherId?.name
      }))
    });
  } catch (error) {
    console.error('Student dashboard error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/classes', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { page = 1, limit = 10, status = 'all', subject = 'all', date } = req.query;
    
    let query = { studentId: student._id };
    
    if (status !== 'all') {
      query.status = status;
    }
    
    if (subject !== 'all') {
      query.subject = subject;
    }
    
    if (date) {
      const startDate = new Date(date);
      const endDate = new Date(date);
      endDate.setDate(endDate.getDate() + 1);
      query.scheduledDate = { $gte: startDate, $lt: endDate };
    }

    const classes = await Class.find(query)
      .populate('teacherId', 'name')
      .sort({ scheduledDate: -1 });

    const total = await Class.countDocuments(query);

    // Convert times to student's timezone
    const studentTimezone = student.timezone || 'UTC';
    const classesWithStudentTimezone = classes.map(cls => {
      const classObj = cls.toObject();

      // Convert scheduledDate to student's timezone
      if (classObj.scheduledDate) {
        const originalDate = moment.tz(classObj.scheduledDate, classObj.timezone || 'UTC');
        classObj.scheduledDateStudentTZ = originalDate.clone().tz(studentTimezone).format();
      }

      // Convert startTime and endTime to student's timezone
      if (classObj.startTime && classObj.scheduledDate) {
        const [hours, minutes] = classObj.startTime.split(':');
        const originalStartTime = moment.tz(classObj.scheduledDate, classObj.timezone || 'UTC')
          .hours(parseInt(hours))
          .minutes(parseInt(minutes));
        classObj.startTimeStudentTZ = originalStartTime.clone().tz(studentTimezone).format('HH:mm');
      }

      if (classObj.endTime && classObj.scheduledDate) {
        const [hours, minutes] = classObj.endTime.split(':');
        const originalEndTime = moment.tz(classObj.scheduledDate, classObj.timezone || 'UTC')
          .hours(parseInt(hours))
          .minutes(parseInt(minutes));
        classObj.endTimeStudentTZ = originalEndTime.clone().tz(studentTimezone).format('HH:mm');
      }

      return classObj;
    });

    res.json({
      classes: classesWithStudentTimezone,
      totalPages: 1,
      currentPage: 1,
      total
    });
  } catch (error) {
    console.error('Get student classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/teachers', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    // ONLY get teachers from scheduled classes (not from assignedTeachers array)
    // This ensures we only show teachers who actually have classes with the student
    const classTeachers = await Class.find({
      studentId: student._id,
      status: { $in: ['scheduled', 'ongoing', 'completed'] },
      isDeleted: false
    })
    .populate('teacherId', 'name email subjects availableTime availableDays bio profilePicture')
    .select('teacherId subject');

    // Deduplicate teachers by teacher ID only
    const teacherMap = new Map();
    const teacherSubjectsMap = new Map(); // Track all subjects per teacher

    // Add teachers from classes
    classTeachers.forEach(classDoc => {
      if (classDoc.teacherId) {
        const teacherId = classDoc.teacherId._id.toString();

        // Track subjects for this teacher
        if (!teacherSubjectsMap.has(teacherId)) {
          teacherSubjectsMap.set(teacherId, new Set());
        }
        teacherSubjectsMap.get(teacherId).add(classDoc.subject);

        // Only set teacher data once
        if (!teacherMap.has(teacherId)) {
          teacherMap.set(teacherId, {
            teacher: classDoc.teacherId,
            canChat: true,
            source: 'class'
          });
        }
      }
    });

    // Merge teacher data with their subjects
    const teachersWithSubjects = Array.from(teacherMap.entries()).map(([teacherId, teacherData]) => {
      const subjects = Array.from(teacherSubjectsMap.get(teacherId) || []);
      return {
        ...teacherData,
        subject: subjects.join(', ') || 'N/A', // All subjects as comma-separated string
        subjects: subjects // All subjects as array
      };
    });

    // Get completed class counts for each teacher
    const teacherIds = teachersWithSubjects.map(t => t.teacher._id);
    const completedClassCounts = await Class.aggregate([
      {
        $match: {
          studentId: student._id,
          teacherId: { $in: teacherIds },
          status: 'completed',
          isDeleted: false
        }
      },
      {
        $group: {
          _id: '$teacherId',
          totalCompletedClasses: { $sum: 1 }
        }
      }
    ]);

    // Create a map for quick lookup of completed class counts
    const completedClassMap = new Map();
    completedClassCounts.forEach(count => {
      completedClassMap.set(count._id.toString(), count.totalCompletedClasses);
    });

    // Add completed class count to each teacher
    const teachersWithCompletedClasses = teachersWithSubjects.map(teacherData => ({
      ...teacherData,
      teacher: {
        ...teacherData.teacher.toObject(),
        totalCompletedClasses: completedClassMap.get(teacherData.teacher._id.toString()) || 0
      }
    }));

    res.json({ teachers: teachersWithCompletedClasses });
  } catch (error) {
    console.error('Get student teachers error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/invoices', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { page = 1, limit = 100, status = 'all' } = req.query;

    let query = { studentId: student._id, enabled: true };
    if (status !== 'all') {
      query.status = status;
    }

    const invoices = await Invoice.find(query)
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ createdAt: -1 });

    const total = await Invoice.countDocuments(query);

    // Transform invoices to match frontend expectations
    const transformedInvoices = invoices.map(invoice => {
      // Check if invoice is overdue
      const now = new Date();
      const isOverdue = invoice.status === 'sent' && new Date(invoice.dueDate) < now;

      // Group classes by subject for items
      const subjectGroups = {};
      invoice.classes.forEach(cls => {
        if (!subjectGroups[cls.subject]) {
          subjectGroups[cls.subject] = {
            description: `${cls.subject} Classes`,
            quantity: 0,
            rate: cls.rate,
            hours: 0,
            amount: 0
          };
        }
        subjectGroups[cls.subject].quantity += 1;
        subjectGroups[cls.subject].hours += cls.duration / 60;
        subjectGroups[cls.subject].amount += cls.amount;
      });

      const items = Object.values(subjectGroups);
      const subtotal = parseFloat(invoice.totalAmount);
      const discount = 0;
      const tax = 0;
      const total = subtotal;

      return {
        _id: invoice._id,
        invoiceNumber: invoice.invoiceNumber,
        date: invoice.createdAt,
        dueDate: invoice.dueDate,
        status: isOverdue ? 'overdue' : invoice.status,
        period: {
          start: moment(`${invoice.invoiceYear}-${invoice.invoiceMonth}-01`).startOf('month').toISOString(),
          end: moment(`${invoice.invoiceYear}-${invoice.invoiceMonth}-01`).endOf('month').toISOString()
        },
        items,
        subtotal,
        discount,
        tax,
        total,
        currency: invoice.currency || 'USD',
        paidDate: invoice.paidDate,
        paymentMethod: invoice.status === 'paid' ? 'Bank Transfer' : null,
        notes: isOverdue ? 'This invoice is overdue. Please make payment as soon as possible.' : ''
      };
    });

    res.json({
      invoices: transformedInvoices,
      totalPages: Math.ceil(total / limit),
      currentPage: parseInt(page),
      total
    });
  } catch (error) {
    console.error('Get student invoices error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/invoices/:id', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const invoice = await Invoice.findOne({
      _id: req.params.id,
      studentId: student._id,
      enabled: true
    })
    .populate('studentId', 'studentName parentName email phoneNumber country class');

    if (!invoice) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    // Check if invoice is overdue
    const now = new Date();
    const isOverdue = invoice.status === 'sent' && new Date(invoice.dueDate) < now;

    // Return full invoice object with all details needed for PDF generation
    const fullInvoice = {
      _id: invoice._id,
      invoiceNumber: invoice.invoiceNumber,
      date: invoice.createdAt,
      dueDate: invoice.dueDate,
      status: isOverdue ? 'overdue' : invoice.status,
      invoiceYear: invoice.invoiceYear,
      invoiceMonth: invoice.invoiceMonth,
      studentId: invoice.studentId,
      classes: invoice.classes || [],
      subtotal: invoice.subtotal,
      discount: invoice.discount || 0,
      totalAmount: invoice.totalAmount,
      currency: invoice.currency || 'USD',
      paidDate: invoice.paidDate,
      paymentMethod: invoice.paymentMethod,
      companyDetails: invoice.companyDetails || {
        companyName: 'EduMetrix UK',
        address: '123 Education Street, London, EC1A 1BB, United Kingdom',
        email: 'contact@edumetrix.uk',
        phone: '+44 20 1234 5678'
      },
      bankAccountDetails: invoice.bankAccountDetails || null,
      paymentLink: invoice.paymentLink || null,
      paymentTerms: invoice.paymentTerms || `1. Payment must be made by ${moment(invoice.dueDate).format('MMMM DD, YYYY')}
2. All invoices must be cleared on or before the 1st of the month
3. Late payments will incur a 5% late fee
4. Classes may be suspended if payment is not received by the due date
5. Refunds are only processed for classes cancelled by EduMetrix with 24 hours notice
6. Payment confirmation receipt must be submitted immediately after payment`,
      notes: isOverdue ? 'This invoice is overdue. Please make payment as soon as possible.' : (invoice.notes || '')
    };

    res.json({ invoice: fullInvoice });
  } catch (error) {
    console.error('Get student invoice error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/profile', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        student: {
          _id: 'dev-student-profile-123',
          studentName: 'Development Student',
          parentName: 'Development Parent',
          email: 'student@edumetrix.uk',
          phoneNumber: {
            countryCode: '+971',
            number: '501234567'
          },
          profilePicture: null,
          hourlyRate: 50,
          currency: 'AED',
          country: 'UAE',
          syllabus: 'CBSE',
          class: 'Class 10',
          subjects: ['Mathematics', 'Physics', 'Chemistry', 'Biology'],
          googleMeetLink: 'https://meet.google.com/dev-link',
          assignedTeachers: [
            {
              teacherId: {
                _id: 'dev-teacher-123',
                name: 'Mr. Smith'
              },
              subject: 'Mathematics'
            },
            {
              teacherId: {
                _id: 'dev-teacher-456',
                name: 'Ms. Johnson'
              },
              subject: 'Physics'
            }
          ],
          userId: {
            username: 'student@edumetrix.uk'
          }
        }
      });
    }

    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    res.json({ student });
  } catch (error) {
    console.error('Get student profile error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Homework route removed - now handled by dedicated homework.js route file

router.get('/join-class/:classId', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const classDoc = await Class.findOne({
      _id: req.params.classId,
      studentId: student._id
    })
    .populate('teacherId', 'name');

    if (!classDoc) {
      return res.status(404).json({ message: 'Class not found' });
    }

    const now = new Date();
    const classTime = new Date(classDoc.scheduledDate);
    const timeDiff = Math.abs(now - classTime) / 60000; // difference in minutes

    if (timeDiff > 15) {
      return res.status(400).json({ 
        message: 'Class can only be joined 15 minutes before or after scheduled time' 
      });
    }

    res.json({
      class: {
        id: classDoc._id,
        subject: classDoc.subject,
        teacher: classDoc.teacherId.name,
        scheduledDate: classDoc.scheduledDate,
        meetingLink: classDoc.meetingLink || student.googleMeetLink,
        duration: classDoc.duration
      }
    });
  } catch (error) {
    console.error('Join class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/attendance-report', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        month: '8/2025',
        summary: {
          totalClasses: 20,
          attendedClasses: 18,
          missedClasses: 1,
          pendingClasses: 1,
          attendancePercentage: '90.00'
        },
        classes: [
          {
            date: '2025-08-01',
            subject: 'Mathematics',
            teacher: 'Mr. Smith',
            status: 'completed',
            attended: true,
            notes: 'Good progress on algebra',
            homework: 'Practice problems 1-20'
          },
          {
            date: '2025-08-03',
            subject: 'Physics',
            teacher: 'Ms. Johnson',
            status: 'completed',
            attended: false,
            notes: null,
            homework: null
          }
        ]
      });
    }

    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { month, year } = req.query;
    
    const currentDate = new Date();
    const reportMonth = month || (currentDate.getMonth() + 1);
    const reportYear = year || currentDate.getFullYear();

    const startDate = new Date(reportYear, reportMonth - 1, 1);
    const endDate = new Date(reportYear, reportMonth, 0);

    const classes = await Class.find({
      studentId: student._id,
      scheduledDate: { $gte: startDate, $lte: endDate }
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    const totalClasses = classes.length;
    const attendedClasses = classes.filter(cls => cls.attendanceStatus?.studentAttended === true).length;
    const missedClasses = classes.filter(cls => cls.attendanceStatus?.studentAttended === false).length;
    const pendingClasses = classes.filter(cls => cls.status === 'scheduled').length;

    const attendancePercentage = totalClasses > 0 ? ((attendedClasses / totalClasses) * 100).toFixed(2) : 0;

    const classDetails = classes.map(cls => ({
      date: cls.scheduledDate.toISOString().split('T')[0],
      subject: cls.subject,
      teacher: cls.teacherId.name,
      status: cls.status,
      attended: cls.attendanceStatus?.studentAttended,
      notes: cls.classNotes,
      homework: cls.homeworkAssigned
    }));

    res.json({
      month: `${reportMonth}/${reportYear}`,
      summary: {
        totalClasses,
        attendedClasses,
        missedClasses,
        pendingClasses,
        attendancePercentage
      },
      classes: classDetails
    });
  } catch (error) {
    console.error('Get student attendance report error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/calendar', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    // Always use student's timezone from their profile
    const studentTimezone = student.timezone || 'UTC';
    const { view = 'month', date = new Date() } = req.query;

    let startDate, endDate;
    const currentDate = moment.tz(date, studentTimezone);

    switch (view) {
      case 'day':
        startDate = currentDate.clone().startOf('day');
        endDate = currentDate.clone().endOf('day');
        break;
      case 'week':
        startDate = currentDate.clone().startOf('week');
        endDate = currentDate.clone().endOf('week');
        break;
      case 'month':
        startDate = currentDate.clone().startOf('month');
        endDate = currentDate.clone().endOf('month');
        break;
      default:
        startDate = currentDate.clone().startOf('month');
        endDate = currentDate.clone().endOf('month');
    }

    const classes = await Class.find({
      studentId: student._id,
      scheduledDate: {
        $gte: startDate.toDate(),
        $lte: endDate.toDate()
      },
      status: { $in: ['scheduled', 'ongoing', 'completed'] }
    })
    .populate('teacherId', 'name')
    .sort({ scheduledDate: 1 });

    // Convert times to student's timezone
    const formattedClasses = classes.map(cls => {
      const classObj = cls.toObject();

      // Convert scheduledDate to student's timezone
      if (classObj.scheduledDate) {
        const originalDate = moment.tz(classObj.scheduledDate, classObj.timezone || 'UTC');
        classObj.scheduledDateStudentTZ = originalDate.clone().tz(studentTimezone).format();
      }

      // Convert startTime and endTime to student's timezone
      if (classObj.startTime && classObj.scheduledDate) {
        const [hours, minutes] = classObj.startTime.split(':');
        const originalStartTime = moment.tz(classObj.scheduledDate, classObj.timezone || 'UTC')
          .hours(parseInt(hours))
          .minutes(parseInt(minutes));
        classObj.startTimeStudentTZ = originalStartTime.clone().tz(studentTimezone).format('HH:mm');
      }

      if (classObj.endTime && classObj.scheduledDate) {
        const [hours, minutes] = classObj.endTime.split(':');
        const originalEndTime = moment.tz(classObj.scheduledDate, classObj.timezone || 'UTC')
          .hours(parseInt(hours))
          .minutes(parseInt(minutes));
        classObj.endTimeStudentTZ = originalEndTime.clone().tz(studentTimezone).format('HH:mm');
      }

      classObj.formattedDate = moment.tz(classObj.scheduledDate, studentTimezone).format('YYYY-MM-DD HH:mm');
      classObj.localTime = moment.tz(classObj.scheduledDate, studentTimezone).format('HH:mm');
      classObj.meetingLink = classObj.meetingLink || student.googleMeetLink;

      return classObj;
    });

    res.json({
      classes: formattedClasses,
      view,
      startDate: startDate.format('YYYY-MM-DD'),
      endDate: endDate.format('YYYY-MM-DD')
    });
  } catch (error) {
    console.error('Get student calendar error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/chat/users', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        users: [
          {
            _id: 'dev-teacher-123',
            username: 'teacher@edumetrix.uk',
            profile: {
              name: 'Mr. Smith',
              subject: 'Mathematics'
            },
            isOnline: true,
            unreadCount: 2
          },
          {
            _id: 'dev-teacher-456',
            username: 'teacher2@edumetrix.uk',
            profile: {
              name: 'Ms. Johnson',
              subject: 'Physics'
            },
            isOnline: false,
            unreadCount: 0
          }
        ]
      });
    }

    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    // Find all teachers assigned to this student
    const populatedStudent = await Student.findById(student._id)
      .populate({
        path: 'assignedTeachers.teacherId',
        select: 'name email subjects userId',
        populate: {
          path: 'userId',
          select: 'username isActive'
        }
      });

    if (!populatedStudent) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const teachers = populatedStudent.assignedTeachers
      .filter(assignment => assignment.teacherId && assignment.teacherId.userId && assignment.teacherId.userId.isActive)
      .map(assignment => ({
        _id: assignment.teacherId._id,
        username: assignment.teacherId.userId.username,
        profile: {
          name: assignment.teacherId.name,
          subject: assignment.subject
        },
        isOnline: true, // You can implement real-time status later
        unreadCount: 0  // You can implement unread message counts later
      }));

    res.json({ users: teachers });
  } catch (error) {
    console.error('Get student chat users error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/chat/:chatId/messages', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock data
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        messages: [
          {
            _id: 'msg1',
            senderId: 'dev-teacher-123',
            content: 'Hello! How are you doing with your math homework?',
            timestamp: new Date(Date.now() - 2 * 60 * 60 * 1000),
            type: 'text'
          },
          {
            _id: 'msg2',
            senderId: 'dev-student-123',
            content: 'Hi! I\'m struggling a bit with quadratic equations. Could you help?',
            timestamp: new Date(Date.now() - 1 * 60 * 60 * 1000),
            type: 'text'
          },
          {
            _id: 'msg3',
            senderId: 'dev-teacher-123',
            content: 'Of course! Let\'s schedule a quick session to go through them together.',
            timestamp: new Date(Date.now() - 30 * 60 * 1000),
            type: 'text'
          }
        ]
      });
    }

    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { chatId } = req.params;

    // Find chat where student is a participant and chatId matches teacher ID
    const chat = await Chat.findOne({
      $and: [
        { 'participants.userId': student.userId },
        { 'participants.userId': chatId }
      ]
    }).populate('messages.senderId', 'username');

    if (!chat) {
      return res.json({ messages: [] });
    }

    res.json({ messages: chat.messages });
  } catch (error) {
    console.error('Get student chat messages error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/chat/:chatId/messages', studentAuth, async (req, res) => {
  try {
    // Development mode: Return mock response
    if (req.user._id.toString().startsWith('dev-')) {
      return res.json({
        message: {
          _id: 'new-msg-' + Date.now(),
          senderId: 'dev-student-123',
          content: req.body.message,
          timestamp: new Date(),
          type: req.body.type || 'text'
        }
      });
    }

    const studentId = req.user._id;

    // Get student info
    const student = await Student.findById(studentId);

    if (!student) {
      return res.status(404).json({ message: 'Student profile not found' });
    }

    const { chatId } = req.params;
    const { message, type = 'text' } = req.body;

    if (!message || !message.trim()) {
      return res.status(400).json({ message: 'Message content is required' });
    }

    // Find or create chat between student and teacher
    let chat = await Chat.findOne({
      $and: [
        { 'participants.userId': student.userId },
        { 'participants.userId': chatId }
      ]
    });

    if (!chat) {
      // Create new chat
      chat = new Chat({
        participants: [
          { userId: student.userId, userType: 'student' },
          { userId: chatId, userType: 'teacher' }
        ],
        chatType: 'student-teacher',
        subject: null, // Allow null subject for general chat
        messages: []
      });
    }

    // Add message
    const newMessage = {
      senderId: student.userId,
      senderType: 'student',
      message: message.trim(),
      timestamp: new Date(),
      readBy: [{ userId: student.userId, readAt: new Date() }]
    };

    chat.messages.push(newMessage);
    chat.lastActivity = new Date();
    await chat.save();

    res.json({ 
      message: {
        ...newMessage,
        _id: newMessage._id,
        content: newMessage.message
      }
    });
  } catch (error) {
    console.error('Send student chat message error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Reschedule request routes

router.post('/reschedule-requests', studentAuth, async (req, res) => {
  try {
    const { classId, reason, classDetails } = req.body;

    if (!classId || !reason || !classDetails) {
      return res.status(400).json({ message: 'Missing required fields' });
    }

    const rescheduleRequest = new RescheduleRequest({
      studentId: req.user._id,
      classId,
      reason: reason.trim(),
      classDetails: {
        subject: classDetails.subject,
        teacherName: classDetails.teacherName,
        scheduledDate: new Date(classDetails.scheduledDate),
        duration: classDetails.duration || 60
      }
    });

    await rescheduleRequest.save();

    // Send notification to admin
    const io = req.app.get('io');
    const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');

    // Find admin user
    const Admin = (await import('../../models/admin/User.js')).default;
    const admin = await Admin.findOne({ userType: 'admin', isActive: true });

    if (admin && io) {
      const student = await Student.findById(req.user._id);
      const notificationData = NotificationTemplates.rescheduleRequest(
        classDetails.subject,
        new Date(classDetails.scheduledDate).toLocaleDateString()
      );

      await createNotification({
        userId: admin._id,
        userType: 'admin',
        type: notificationData.type,
        title: notificationData.title,
        message: `${student?.studentName || 'A student'} requested to reschedule ${classDetails.subject} class. Reason: ${reason}`,
        relatedId: rescheduleRequest._id,
        relatedModel: 'RescheduleRequest',
        priority: notificationData.priority,
        actionUrl: `/admin/reschedule-requests`,
        io
      });
    }

    res.status(201).json({
      message: 'Reschedule request submitted successfully',
      request: rescheduleRequest
    });
  } catch (error) {
    console.error('Create reschedule request error:', error);
    res.status(500).json({ message: 'Failed to submit reschedule request' });
  }
});

router.get('/reschedule-requests', studentAuth, async (req, res) => {
  try {
    const { status, page = 1, limit = 10 } = req.query;
    const query = { studentId: req.user._id };
    
    if (status) {
      query.status = status;
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);
    
    const requests = await RescheduleRequest
      .find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .populate('processedBy', 'name email');

    const total = await RescheduleRequest.countDocuments(query);

    res.json({
      requests,
      pagination: {
        total,
        page: parseInt(page),
        limit: parseInt(limit),
        pages: Math.ceil(total / parseInt(limit))
      }
    });
  } catch (error) {
    console.error('Get reschedule requests error:', error);
    res.status(500).json({ message: 'Failed to fetch reschedule requests' });
  }
});

export default router;