import express from 'express';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import User from '../../models/admin/User.js';
import Student from '../../models/student/Student.js';
import Teacher from '../../models/teacher/Teacher.js';
import { auth } from '../../middleware/auth.js';
import { sendOTP, verifyOTP } from '../../services/emailService.js';

const router = express.Router();

// Admin OTP request endpoint
router.post('/admin/request-otp', async (req, res) => {
  try {
    const { email } = req.body;

    if (!email) {
      return res.status(400).json({ message: 'Email is required' });
    }

    // Only allow edumetrixuk@gmail.com for admin access
    const allowedAdmins = ['edumetrixuk@gmail.com', 'dilkashpeshimam@gmail.com'];

    if (!allowedAdmins.includes(email)) {
      return res.status(403).json({ message: 'Unauthorized email address for admin access' });
    }

    const result = await sendOTP(email);
    res.json(result);
  } catch (error) {
    console.error('OTP request error:', error);
    res.status(500).json({ message: error.message || 'Failed to send OTP' });
  }
});

// Admin OTP verification and login endpoint
router.post('/admin/verify-otp', async (req, res) => {
  try {
    const { email, otp } = req.body;

    if (!email || !otp) {
      return res.status(400).json({ message: 'Email and OTP are required' });
    }

    // Only allow edumetrixuk@gmail.com for admin access
    const allowedAdmins = ['edumetrixuk@gmail.com', 'dilkashpeshimam@gmail.com'];

    if (!allowedAdmins.includes(email)) {
      return res.status(403).json({ message: 'Unauthorized email address for admin access' });
    }

    const verification = verifyOTP(email, otp);

    if (!verification.success) {
      return res.status(400).json({ message: verification.message });
    }

    // Find admin user by email
    const adminUser = await User.findOne({
      'profile.email': email,
      userType: 'admin',
      isActive: true,
      isDeleted: false,
    });

    if (!adminUser) {
      return res.status(404).json({ message: 'Admin user not found' });
    }

    // Generate JWT token
    const token = jwt.sign(
      {
        userId: adminUser._id.toString(),
        userType: adminUser.userType,
      },
      process.env.JWT_SECRET,
      { expiresIn: '24h' }
    );

    res.json({
        success: true,

      token,
      user: {
        id: adminUser._id.toString(),
        username: adminUser.username,
        userType: adminUser.userType,
        profile: adminUser.profile,
      },
    });
  } catch (error) {
    console.error('OTP verification error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/login', async (req, res) => {
  try {
    const { username, password } = req.body;

    if (!username || !password) {
      return res.status(400).json({ message: 'Please provide username and password' });
    }

    const user = await User.findOne({ username, isActive: true, isDeleted: false });
    if (!user) {
      return res.status(400).json({ message: 'Invalid credentials' });
    }

    // SECURITY: Block admin users from using username/password login
    // Admins must use OTP login via /admin/verify-otp endpoint for enhanced security
    if (user.userType === 'admin') {
      return res.status(403).json({
        message: 'Admin login not allowed via this endpoint. Please use email OTP verification via /admin/request-otp',
      });
    }

    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch) {
      return res.status(400).json({ message: 'Invalid credentials' });
    }

    const token = jwt.sign(
      {
        userId: user._id,
        userType: user.userType,
      },
      process.env.JWT_SECRET,
      { expiresIn: '24h' }
    );

    let profileData = {};

    if (user.userType === 'student') {
      profileData = await Student.findOne({ userId: user._id });
    } else if (user.userType === 'teacher') {
      profileData = await Teacher.findOne({ userId: user._id });
    } else if (user.userType === 'admin') {
      profileData = user.profile;
    }

    res.json({
      token,
      user: {
        id: user._id,
        username: user.username,
        userType: user.userType,
        profile: profileData,
      },
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/me', auth, async (req, res) => {
  try {
    let profileData = {};

    if (req.user.userType === 'student') {
      profileData = await Student.findOne({ userId: req.user._id }).populate('assignedTeachers.teacherId', 'name');
    } else if (req.user.userType === 'teacher') {
      profileData = await Teacher.findOne({ userId: req.user._id }).populate(
        'assignedStudents.studentId',
        'studentName'
      );
    } else if (req.user.userType === 'admin') {
      profileData = req.user.profile;
    }

    res.json({
      user: {
        id: req.user._id,
        username: req.user.username,
        userType: req.user.userType,
        profile: profileData,
      },
    });
  } catch (error) {
    console.error('Get user error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/change-password', auth, async (req, res) => {
  try {
    const { currentPassword, newPassword } = req.body;

    if (!currentPassword || !newPassword) {
      return res.status(400).json({ message: 'Please provide current and new password' });
    }

    const user = await User.findById(req.user._id);
    const isMatch = await bcrypt.compare(currentPassword, user.password);

    if (!isMatch) {
      return res.status(400).json({ message: 'Current password is incorrect' });
    }

    const salt = await bcrypt.genSalt(10);
    user.password = await bcrypt.hash(newPassword, salt);
    await user.save();

    res.json({ message: 'Password updated successfully' });
  } catch (error) {
    console.error('Change password error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

export default router;
