import mongoose from 'mongoose';

const teacherSettingsSchema = new mongoose.Schema({
  teacherId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Teacher',
    required: true,
    unique: true
  },

  // Appearance Settings
  appearance: {
    colorTheme: {
      type: String,
      enum: ['blue', 'purple', 'green', 'orange', 'pink', 'teal', 'cyan', 'indigo'],
      default: 'blue'
    },
    darkMode: {
      type: Boolean,
      default: false
    }
  },

  // Notification Settings
  notifications: {
    assignments: {
      type: Boolean,
      default: true
    },
    grades: {
      type: Boolean,
      default: true
    },
    exams: {
      type: Boolean,
      default: true
    },
    announcements: {
      type: Boolean,
      default: true
    },
    messages: {
      type: Boolean,
      default: true
    },
    classReminders: {
      type: Boolean,
      default: true
    }
  },

  // Profile Settings
  profile: {
    name: {
      type: String,
      default: ''
    },
    email: {
      type: String,
      default: ''
    },
    phone: {
      type: String,
      default: ''
    },
    timezone: {
      type: String,
      default: 'UTC'
    },
    bio: {
      type: String,
      default: ''
    }
  },

  // Privacy Settings
  privacy: {
    dataAnalytics: {
      type: Boolean,
      default: true
    },
    promotionalEmails: {
      type: Boolean,
      default: true
    },
    profileVisibility: {
      type: String,
      enum: ['public', 'students', 'private'],
      default: 'students'
    }
  },

  // Language Settings
  language: {
    type: String,
    enum: ['en', 'es', 'fr', 'de', 'it', 'pt', 'zh', 'ja'],
    default: 'en'
  },

  // Teaching Preferences
  teachingPreferences: {
    autoGrading: {
      type: Boolean,
      default: false
    },
    reminderEmails: {
      type: Boolean,
      default: true
    },
    calendarSync: {
      type: Boolean,
      default: false
    }
  }
}, {
  timestamps: true
});

// Index for faster queries
teacherSettingsSchema.index({ teacherId: 1 });

// Static method to get default settings for a teacher
teacherSettingsSchema.statics.getDefaultSettings = function(teacherId) {
  return {
    teacherId,
    appearance: {
      colorTheme: 'blue',
      darkMode: false
    },
    notifications: {
      assignments: true,
      grades: true,
      exams: true,
      announcements: true,
      messages: true,
      classReminders: true
    },
    profile: {
      name: '',
      email: '',
      phone: '',
      timezone: 'UTC',
      bio: ''
    },
    privacy: {
      dataAnalytics: true,
      promotionalEmails: true,
      profileVisibility: 'students'
    },
    language: 'en',
    teachingPreferences: {
      autoGrading: false,
      reminderEmails: true,
      calendarSync: false
    }
  };
};

// Method to update specific setting category
teacherSettingsSchema.methods.updateCategory = function(category, updates) {
  if (this[category]) {
    Object.assign(this[category], updates);
  }
  return this.save();
};

export default mongoose.model('TeacherSettings', teacherSettingsSchema);