import mongoose from 'mongoose';

const messageSchema = new mongoose.Schema({
  conversationId: {
    type: String,
    ref: 'Conversation',

    required: function () {
      return !this.isGroupMessage; // Only required if it's NOT a group message
    },
  },
  senderId: {
    type: mongoose.Schema.Types.ObjectId,
    required: true,
    refPath: 'senderType',
  },
  senderType: {
    type: String,
    required: true,
    enum: ['Admin', 'Teacher', 'Student'],
  },
  receiverId: {
    type: mongoose.Schema.Types.ObjectId,
    refPath: 'receiverType',
  },
  receiverType: {
    type: String,
    enum: ['Admin', 'Teacher', 'Student'],
  },
  isGroupMessage: {
    type: Boolean,
    default: false,
  },
  content: {
    type: String,
    required: function() {
      return !this.attachment; // Content is required only if there's no attachment
    },
    default: ''
  },
  attachment: {
    fileName: String,
    fileUrl: String,
    fileType: String,
    fileSize: Number,
  },
  timestamp: {
    type: Date,
    required: true,
    default: Date.now,
  },
  isRead: {
    type: Boolean,
    default: false,
  },
  readAt: {
    type: Date,
  },
});

messageSchema.index({ conversationId: 1 });
messageSchema.index({ senderId: 1 });
messageSchema.index({ receiverId: 1 });

// Middleware to update conversation timestamp when a message is created
messageSchema.post('save', async function(doc) {
  if (doc.conversationId && !doc.isGroupMessage) {
    try {
      const Conversation = mongoose.model('Conversation');
      await Conversation.findByIdAndUpdate(
        doc.conversationId,
        { updatedAt: doc.timestamp || new Date() },
        { new: true }
      );
      console.log(`✅ Updated conversation ${doc.conversationId} timestamp to ${doc.timestamp}`);
    } catch (error) {
      console.error('Error updating conversation timestamp:', error);
    }
  }
});

const Message = mongoose.model('Message', messageSchema);

export default Message;
