import mongoose from 'mongoose';

const adminSettingsSchema = new mongoose.Schema({
  adminId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true,
    unique: true
  },

  // Appearance Settings
  appearance: {
    colorTheme: {
      type: String,
      enum: ['blue', 'purple', 'green', 'orange', 'pink', 'teal', 'cyan', 'indigo'],
      default: 'blue'
    },
    darkMode: {
      type: Boolean,
      default: false
    }
  },

  // Notification Settings
  notifications: {
    userRegistrations: {
      type: Boolean,
      default: true
    },
    systemAlerts: {
      type: Boolean,
      default: true
    },
    financialReports: {
      type: Boolean,
      default: true
    },
    classSchedules: {
      type: Boolean,
      default: true
    },
    trialRequests: {
      type: Boolean,
      default: true
    },
    messages: {
      type: Boolean,
      default: true
    },
    systemMaintenance: {
      type: Boolean,
      default: true
    }
  },

  // Profile Settings
  profile: {
    name: {
      type: String,
      default: ''
    },
    email: {
      type: String,
      default: ''
    },
    phone: {
      type: String,
      default: ''
    },
    timezone: {
      type: String,
      default: 'UTC'
    },
    bio: {
      type: String,
      default: ''
    },
    jobTitle: {
      type: String,
      default: 'Administrator'
    }
  },

  // Privacy Settings
  privacy: {
    dataAnalytics: {
      type: Boolean,
      default: true
    },
    promotionalEmails: {
      type: Boolean,
      default: true
    },
    profileVisibility: {
      type: String,
      enum: ['public', 'staff', 'private'],
      default: 'staff'
    },
    auditLogs: {
      type: Boolean,
      default: true
    }
  },

  // Language Settings
  language: {
    type: String,
    enum: ['en', 'es', 'fr', 'de', 'it', 'pt', 'zh', 'ja'],
    default: 'en'
  },

  // Admin Preferences
  adminPreferences: {
    autoApproveTrials: {
      type: Boolean,
      default: false
    },
    defaultClassDuration: {
      type: Number,
      default: 60
    },
    emailDigests: {
      type: Boolean,
      default: true
    },
    systemBackups: {
      type: Boolean,
      default: true
    },
    dashboardLayout: {
      type: String,
      enum: ['compact', 'detailed', 'overview'],
      default: 'detailed'
    },
    multiFactorAuth: {
      type: Boolean,
      default: false
    }
  },

  // System Settings
  systemSettings: {
    maintenanceMode: {
      type: Boolean,
      default: false
    },
    maxStudentsPerClass: {
      type: Number,
      default: 30
    },
    allowSelfRegistration: {
      type: Boolean,
      default: true
    },
    sessionTimeout: {
      type: Number,
      default: 120
    }
  }
}, {
  timestamps: true
});

// Index for faster queries
adminSettingsSchema.index({ adminId: 1 });

// Static method to get default settings for an admin
adminSettingsSchema.statics.getDefaultSettings = function(adminId) {
  return {
    adminId,
    appearance: {
      colorTheme: 'blue',
      darkMode: false
    },
    notifications: {
      userRegistrations: true,
      systemAlerts: true,
      financialReports: true,
      classSchedules: true,
      trialRequests: true,
      messages: true,
      systemMaintenance: true
    },
    profile: {
      name: '',
      email: '',
      phone: '',
      timezone: 'UTC',
      bio: '',
      jobTitle: 'Administrator'
    },
    privacy: {
      dataAnalytics: true,
      promotionalEmails: true,
      profileVisibility: 'staff',
      auditLogs: true
    },
    language: 'en',
    adminPreferences: {
      autoApproveTrials: false,
      defaultClassDuration: 60,
      emailDigests: true,
      systemBackups: true,
      dashboardLayout: 'detailed',
      multiFactorAuth: false
    },
    systemSettings: {
      maintenanceMode: false,
      maxStudentsPerClass: 30,
      allowSelfRegistration: true,
      sessionTimeout: 120
    }
  };
};

// Method to update specific setting category
adminSettingsSchema.methods.updateCategory = function(category, updates) {
  if (this[category]) {
    Object.assign(this[category], updates);
  }
  return this.save();
};

export default mongoose.model('AdminSettings', adminSettingsSchema);