const jwt = require('jsonwebtoken');
const crypto = require('crypto');
const User = require('../models/admin/User');
const SecurityConfig = require('../config/security');
const auditLogger = require('../utils/auditLogger');
const rateLimiter = require('../middleware/rateLimiter');

class EnterpriseAuth {
  constructor() {
    this.blacklistedTokens = new Set();
    this.refreshTokens = new Map(); // In production, use Redis
  }

  // Generate secure JWT tokens
  generateTokens(user) {
    const payload = {
      userId: user._id,
      userType: user.userType,
      email: user.username,
      sessionId: crypto.randomUUID(),
      iat: Math.floor(Date.now() / 1000),
      iss: SecurityConfig.jwt.issuer,
      aud: SecurityConfig.jwt.audience
    };

    const accessToken = jwt.sign(
      payload,
      process.env.JWT_SECRET,
      {
        expiresIn: SecurityConfig.jwt.accessTokenExpiry,
        algorithm: SecurityConfig.jwt.algorithm
      }
    );

    const refreshToken = jwt.sign(
      { ...payload, type: 'refresh' },
      process.env.JWT_REFRESH_SECRET || process.env.JWT_SECRET,
      {
        expiresIn: SecurityConfig.jwt.refreshTokenExpiry,
        algorithm: SecurityConfig.jwt.algorithm
      }
    );

    // Store refresh token securely
    this.refreshTokens.set(refreshToken, {
      userId: user._id,
      sessionId: payload.sessionId,
      createdAt: new Date(),
      lastUsed: new Date()
    });

    return { accessToken, refreshToken, sessionId: payload.sessionId };
  }

  // Enhanced authentication middleware
  authenticate = async (req, res, next) => {
    try {
      const authHeader = req.header('Authorization');
      
      if (!authHeader || !authHeader.startsWith('Bearer ')) {
        return this.unauthorizedResponse(res, 'No valid token provided');
      }

      const token = authHeader.replace('Bearer ', '');

      // Check if token is blacklisted
      if (this.blacklistedTokens.has(token)) {
        auditLogger.warn('Blacklisted token used', {
          ip: req.ip,
          userAgent: req.get('User-Agent'),
          token: token.substring(0, 10) + '...'
        });
        return this.unauthorizedResponse(res, 'Token has been revoked');
      }

      // Development mode bypass
      if (token.startsWith('dev-')) {
        return this.handleDevelopmentToken(token, req, res, next);
      }

      // Verify JWT token
      const decoded = jwt.verify(token, process.env.JWT_SECRET, {
        issuer: SecurityConfig.jwt.issuer,
        audience: SecurityConfig.jwt.audience,
        algorithms: [SecurityConfig.jwt.algorithm]
      });

      // Additional security checks
      if (!decoded.userId || !decoded.userType || !decoded.sessionId) {
        throw new Error('Invalid token payload');
      }

      // Fetch user and validate
      const user = await User.findById(decoded.userId)
        .select('-password -refreshTokens')
        .lean();

      if (!user || !user.isActive || user.isDeleted) {
        auditLogger.warn('Invalid user attempted access', {
          userId: decoded.userId,
          ip: req.ip
        });
        return this.unauthorizedResponse(res, 'User not found or inactive');
      }

      // Check for suspicious activity
      if (this.detectSuspiciousActivity(req, user)) {
        auditLogger.error('Suspicious activity detected', {
          userId: user._id,
          ip: req.ip,
          userAgent: req.get('User-Agent')
        });
        return this.unauthorizedResponse(res, 'Suspicious activity detected');
      }

      // Add user info to request
      req.user = {
        ...user,
        sessionId: decoded.sessionId,
        tokenIssuedAt: decoded.iat
      };

      // Log successful authentication
      auditLogger.info('User authenticated', {
        userId: user._id,
        userType: user.userType,
        ip: req.ip,
        endpoint: req.path
      });

      next();
    } catch (error) {
      if (error.name === 'JsonWebTokenError') {
        auditLogger.warn('Invalid JWT token', {
          error: error.message,
          ip: req.ip
        });
        return this.unauthorizedResponse(res, 'Invalid token');
      } else if (error.name === 'TokenExpiredError') {
        return this.unauthorizedResponse(res, 'Token expired', 'TOKEN_EXPIRED');
      } else {
        auditLogger.error('Authentication error', {
          error: error.message,
          stack: error.stack,
          ip: req.ip
        });
        return res.status(500).json({ message: 'Authentication failed' });
      }
    }
  };

  // Refresh token endpoint
  refreshAccessToken = async (req, res) => {
    try {
      const { refreshToken } = req.body;

      if (!refreshToken) {
        return res.status(400).json({ message: 'Refresh token required' });
      }

      // Verify refresh token
      const decoded = jwt.verify(
        refreshToken,
        process.env.JWT_REFRESH_SECRET || process.env.JWT_SECRET
      );

      if (decoded.type !== 'refresh') {
        throw new Error('Invalid refresh token type');
      }

      // Check if refresh token exists in store
      const tokenData = this.refreshTokens.get(refreshToken);
      if (!tokenData || tokenData.userId !== decoded.userId) {
        auditLogger.warn('Invalid refresh token used', {
          userId: decoded.userId,
          ip: req.ip
        });
        return this.unauthorizedResponse(res, 'Invalid refresh token');
      }

      // Fetch user
      const user = await User.findById(decoded.userId)
        .select('-password')
        .lean();

      if (!user || !user.isActive || user.isDeleted) {
        return this.unauthorizedResponse(res, 'User not found or inactive');
      }

      // Generate new tokens
      const tokens = this.generateTokens(user);

      // Remove old refresh token and add new one
      this.refreshTokens.delete(refreshToken);

      auditLogger.info('Tokens refreshed', {
        userId: user._id,
        ip: req.ip
      });

      res.json({
        accessToken: tokens.accessToken,
        refreshToken: tokens.refreshToken,
        expiresIn: SecurityConfig.jwt.accessTokenExpiry
      });
    } catch (error) {
      auditLogger.error('Token refresh failed', {
        error: error.message,
        ip: req.ip
      });
      return this.unauthorizedResponse(res, 'Token refresh failed');
    }
  };

  // Logout and token revocation
  logout = async (req, res) => {
    try {
      const authHeader = req.header('Authorization');
      if (authHeader && authHeader.startsWith('Bearer ')) {
        const token = authHeader.replace('Bearer ', '');
        
        // Add to blacklist
        this.blacklistedTokens.add(token);
        
        // Remove refresh tokens for this session
        const sessionId = req.user?.sessionId;
        if (sessionId) {
          for (const [refreshToken, data] of this.refreshTokens.entries()) {
            if (data.sessionId === sessionId) {
              this.refreshTokens.delete(refreshToken);
            }
          }
        }
      }

      auditLogger.info('User logged out', {
        userId: req.user?._id,
        ip: req.ip
      });

      res.json({ message: 'Logged out successfully' });
    } catch (error) {
      auditLogger.error('Logout error', {
        error: error.message,
        userId: req.user?._id
      });
      res.status(500).json({ message: 'Logout failed' });
    }
  };

  // Role-based authorization
  authorize = (requiredRoles = []) => {
    return (req, res, next) => {
      if (!req.user) {
        return this.unauthorizedResponse(res, 'Authentication required');
      }

      const userRoles = Array.isArray(req.user.userType) 
        ? req.user.userType 
        : [req.user.userType];

      const hasRequiredRole = requiredRoles.some(role => 
        userRoles.includes(role) || userRoles.includes('admin')
      );

      if (requiredRoles.length > 0 && !hasRequiredRole) {
        auditLogger.warn('Access denied - insufficient permissions', {
          userId: req.user._id,
          userRoles: userRoles,
          requiredRoles: requiredRoles,
          endpoint: req.path,
          ip: req.ip
        });
        return res.status(403).json({ 
          message: 'Access denied - insufficient permissions' 
        });
      }

      next();
    };
  };

  // Detect suspicious activity
  detectSuspiciousActivity(req, user) {
    // Check for multiple rapid requests
    const userAgent = req.get('User-Agent');
    const ip = req.ip;

    // Simple heuristics - in production, use more sophisticated detection
    if (!userAgent || userAgent.length < 10) {
      return true; // Suspicious user agent
    }

    // Add more detection logic as needed
    return false;
  }

  // Handle development tokens
  handleDevelopmentToken(token, req, res, next) {
    if (process.env.NODE_ENV === 'production') {
      return this.unauthorizedResponse(res, 'Development tokens not allowed in production');
    }

    let userData = {
      _id: 'dev-admin-123',
      username: 'admin@edumetrix.uk',
      userType: 'admin',
      isActive: true,
      isDeleted: false
    };

    if (token.includes('teacher')) {
      userData = {
        _id: 'dev-teacher-123',
        username: 'teacher@edumetrix.uk',
        userType: 'teacher',
        isActive: true,
        isDeleted: false
      };
    } else if (token.includes('student')) {
      userData = {
        _id: 'dev-student-123',
        username: 'student@edumetrix.uk',
        userType: 'student',
        isActive: true,
        isDeleted: false
      };
    }

    req.user = userData;
    next();
  }

  // Standardized unauthorized response
  unauthorizedResponse(res, message, code = 'UNAUTHORIZED') {
    return res.status(401).json({
      message,
      code,
      timestamp: new Date().toISOString()
    });
  }

  // Cleanup expired tokens periodically
  cleanupExpiredTokens() {
    const now = new Date();
    const refreshTokenExpiry = 7 * 24 * 60 * 60 * 1000; // 7 days

    for (const [token, data] of this.refreshTokens.entries()) {
      if (now - data.createdAt > refreshTokenExpiry) {
        this.refreshTokens.delete(token);
      }
    }

    // Cleanup blacklisted tokens (basic implementation)
    if (this.blacklistedTokens.size > 10000) {
      this.blacklistedTokens.clear(); // In production, implement smarter cleanup
    }
  }
}

// Create singleton instance
const enterpriseAuth = new EnterpriseAuth();

// Cleanup expired tokens every hour
setInterval(() => {
  enterpriseAuth.cleanupExpiredTokens();
}, 60 * 60 * 1000);

module.exports = enterpriseAuth;