import jwt from 'jsonwebtoken';
import mongoose from 'mongoose';
import User from '../models/admin/User.js';
import Student from '../models/student/Student.js';
import Teacher from '../models/teacher/Teacher.js';
import { securityLogger } from './security.js';
import crypto from 'crypto';

// Token blacklist (in production, use Redis)
const tokenBlacklist = new Set();

// Session tracking
const activeSessions = new Map();

// Grace period for token transitions (3 minutes)
const TOKEN_GRACE_PERIOD = 3 * 60 * 1000; // 3 minutes in milliseconds

// Generate secure session ID
const generateSessionId = () => {
  return crypto.randomBytes(32).toString('hex');
};

// Enhanced JWT verification
const verifyJWT = (token) => {
  return new Promise((resolve, reject) => {
        console.log("JWT_SECRET in verify route:", process.env.JWT_SECRET);

    jwt.verify(token, process.env.JWT_SECRET, (err, decoded) => {
      if (err) {
        reject(err);
      } else {
        console.log('getting decoded data>>',decoded)
        resolve(decoded);
      }
    });
  });
};


// Admin-only middleware
const requireAdmin = (req, res, next) => {
  if (!req.user || req.user.userType !== 'admin') {
    securityLogger.warn('Unauthorized admin access attempt', {
      userId: req.user?.id || 'anonymous',
      userType: req.user?.userType || 'none',
      ip: req.ip,
      url: req.originalUrl,
      timestamp: new Date()
    });
    return res.status(403).json({ error: 'Admin access required.' });
  }
  next();
};

// Teacher-only middleware
const requireTeacher = (req, res, next) => {
  if (!req.user || req.user.userType !== 'teacher') {
    securityLogger.warn('Unauthorized teacher access attempt', {
      userId: req.user?.id || 'anonymous',
      userType: req.user?.userType || 'none',
      ip: req.ip,
      url: req.originalUrl,
      timestamp: new Date()
    });
    return res.status(403).json({ error: 'Teacher access required.' });
  }
  next();
};

// Student-only middleware
const requireStudent = (req, res, next) => {
  if (!req.user || req.user.userType !== 'student') {
    securityLogger.warn('Unauthorized student access attempt', {
      userId: req.user?.id || 'anonymous',
      userType: req.user?.userType || 'none',
      ip: req.ip,
      url: req.originalUrl,
      timestamp: new Date()
    });
    return res.status(403).json({ error: 'Student access required.' });
  }
  next();
};

// Multi-role middleware
const requireRole = (...roles) => {
  return (req, res, next) => {
    if (!req.user || !roles.includes(req.user.userType)) {
      securityLogger.warn('Unauthorized role access attempt', {
        userId: req.user?.id || 'anonymous',
        userType: req.user?.userType || 'none',
        requiredRoles: roles,
        ip: req.ip,
        url: req.originalUrl,
        timestamp: new Date()
      });
      return res.status(403).json({ 
        error: `Access denied. Required roles: ${roles.join(', ')}` 
      });
    }
    next();
  };
};

// Token blacklisting
const blacklistToken = (token) => {
  tokenBlacklist.add(token);
  securityLogger.info('Token blacklisted', { timestamp: new Date() });
};

// Main authentication middleware
const advancedAuth = async (req, res, next) => {
  try {
    const authHeader = req.header('Authorization');

    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return res.status(401).json({ error: 'No token provided. Please login again.' });
    }

    const token = authHeader.replace('Bearer ', '');

    // Check if token is blacklisted
    if (tokenBlacklist.has(token)) {
      securityLogger.warn('Blacklisted token used', {
        ip: req.ip,
        timestamp: new Date()
      });
      return res.status(401).json({ error: 'Token has been revoked. Please login again.' });
    }

    // Verify JWT token
    const decoded = await verifyJWT(token);

    if (!decoded.userId || !decoded.userType) {
      throw new Error('Invalid token payload');
    }

    // Fetch user based on userType
    let user;
    let profileId;

    if (decoded.userType === 'admin') {
      user = await User.findById(decoded.userId).select('-password').lean();
      profileId = user?._id;
    } else if (decoded.userType === 'student') {
      const studentProfile = await Student.findOne({ userId: decoded.userId }).lean();
      if (studentProfile) {
        user = await User.findById(decoded.userId).select('-password').lean();
        if (user) {
          user.profile = studentProfile;
          profileId = studentProfile._id; // Use Student profile ID
        }
      }
    } else if (decoded.userType === 'teacher') {
      const teacherProfile = await Teacher.findOne({ userId: decoded.userId }).lean();
      if (teacherProfile) {
        user = await User.findById(decoded.userId).select('-password').lean();
        if (user) {
          user.profile = teacherProfile;
          profileId = teacherProfile._id; // Use Teacher profile ID
        }
      }
    }

    if (!user || !user.isActive || user.isDeleted) {
      securityLogger.warn('Invalid user attempted access', {
        userId: decoded.userId,
        userType: decoded.userType,
        ip: req.ip,
        timestamp: new Date()
      });
      return res.status(401).json({ error: 'User not found or inactive. Please login again.' });
    }

    // Add user info to request
    // IMPORTANT: For students and teachers, _id should be the profile ID, not the User ID
    req.user = {
      _id: profileId, // Student._id or Teacher._id or User._id (for admin)
      id: profileId.toString(),
      userId: user._id, // Original User model ID
      username: user.username,
      userType: user.userType,
      profile: user.profile,
      isActive: user.isActive
    };

    req.token = token;

    // Update session activity
    const sessionKey = `${user._id}_${req.ip}_${req.get('User-Agent') || 'unknown'}`;
    const hashedSessionKey = crypto.createHash('sha256').update(sessionKey).digest('hex');

    activeSessions.set(hashedSessionKey, {
      userId: user._id,
      sessionId: decoded.sessionId || generateSessionId(),
      lastActivity: Date.now()
    });

    // Log successful authentication (only for sensitive endpoints)
    if (req.originalUrl.includes('/admin/') || req.method === 'DELETE') {
      securityLogger.info('User authenticated', {
        userId: user._id,
        userType: user.userType,
        ip: req.ip,
        endpoint: req.path,
        timestamp: new Date()
      });
    }

    next();
  } catch (error) {
    if (error.name === 'JsonWebTokenError') {
      securityLogger.warn('Invalid JWT token', {
        error: error.message,
        ip: req.ip,
        timestamp: new Date()
      });
      return res.status(401).json({ error: 'Invalid token. Please login again.' });
    } else if (error.name === 'TokenExpiredError') {
      return res.status(401).json({ error: 'Token expired. Please login again.', code: 'TOKEN_EXPIRED' });
    } else {
      securityLogger.error('Authentication error', {
        error: error.message,
        stack: error.stack,
        ip: req.ip,
        timestamp: new Date()
      });
      return res.status(500).json({ error: 'Authentication failed. Please try again.' });
    }
  }
};

// Session cleanup
const cleanupSessions = () => {
  const now = Date.now();
  const sessionTimeout = 30 * 60 * 1000; // 30 minutes

  for (const [key, session] of activeSessions.entries()) {
    if (now - session.lastActivity > sessionTimeout) {
      activeSessions.delete(key);
      securityLogger.info('Session cleaned up due to inactivity', {
        userId: session.userId,
        sessionId: session.sessionId,
        timestamp: new Date()
      });
    }
  }
};

// Run session cleanup every 5 minutes
setInterval(cleanupSessions, 5 * 60 * 1000);

// Enhanced logout
const secureLogout = (req, res) => {
  if (req.token) {
    blacklistToken(req.token);
  }
  
  if (req.sessionId) {
    const sessionKey = `${req.user.id}_${req.ip}_${req.get('User-Agent')}`;
    const hashedSessionKey = crypto.createHash('sha256').update(sessionKey).digest('hex');
    activeSessions.delete(hashedSessionKey);
  }

  securityLogger.info('User logged out', {
    userId: req.user?.id,
    ip: req.ip,
    timestamp: new Date()
  });

  res.json({ message: 'Logged out successfully' });
};

export {
  advancedAuth as auth,
  requireAdmin,
  requireTeacher,
  requireStudent,
  requireRole,
  blacklistToken,
  secureLogout,
  cleanupSessions
};