const User = require('../../../models/admin/User');
const mongoose = require('mongoose');

describe('User Model Test', () => {
  beforeAll(async () => {
    await mongoose.connect('mongodb://localhost:27017/edumetrix_test', {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });
  });

  afterAll(async () => {
    await mongoose.connection.close();
  });

  afterEach(async () => {
    await User.deleteMany({});
  });

  test('should create a new user', async () => {
    const userData = {
      username: 'test@example.com',
      password: 'hashedPassword123',
      userType: 'student',
      profile: {
        name: 'Test User',
        email: 'test@example.com'
      }
    };

    const user = new User(userData);
    await user.save();

    expect(user.username).toBe(userData.username);
    expect(user.userType).toBe(userData.userType);
    expect(user.isActive).toBe(true);
    expect(user.isDeleted).toBe(false);
  });

  test('should not create user without required fields', async () => {
    const user = new User({
      username: 'test@example.com'
      // Missing password and userType
    });

    let err;
    try {
      await user.save();
    } catch (error) {
      err = error;
    }

    expect(err).toBeInstanceOf(mongoose.Error.ValidationError);
    expect(err.errors.password).toBeDefined();
    expect(err.errors.userType).toBeDefined();
  });

  test('should enforce unique username', async () => {
    const userData = {
      username: 'duplicate@example.com',
      password: 'hashedPassword123',
      userType: 'student'
    };

    await new User(userData).save();

    let err;
    try {
      await new User(userData).save();
    } catch (error) {
      err = error;
    }

    expect(err.code).toBe(11000); // MongoDB duplicate key error
  });

  test('should have correct default values', async () => {
    const user = new User({
      username: 'test@example.com',
      password: 'hashedPassword123',
      userType: 'admin'
    });

    await user.save();

    expect(user.profile.position).toBe('EduMetrix');
    expect(user.profile.department).toBe('');
    expect(user.isActive).toBe(true);
    expect(user.isDeleted).toBe(false);
  });
});