#!/bin/bash

# EduMetrix UK Platform - Production Startup Script
# ==================================================

echo "🎓 Starting EduMetrix UK Platform - Production Mode"
echo "=================================================="

# Check if .env file exists
if [ ! -f .env ]; then
    echo "❌ Error: .env file not found!"
    echo "Please run: node setup-production-admin.js first"
    exit 1
fi

# Function to check if port is in use
check_port() {
    if lsof -Pi :$1 -sTCP:LISTEN -t >/dev/null ; then
        echo "❌ Port $1 is already in use. Please stop the service using that port."
        exit 1
    fi
}

# Check if required ports are available
echo "🔍 Checking port availability..."
check_port 5001
check_port 3001
check_port 3002
check_port 3003

# Kill any existing processes (cleanup)
echo "🧹 Cleaning up any existing processes..."
pkill -f "PORT=300" 2>/dev/null || true
pkill -f "node server.js" 2>/dev/null || true

# Start backend server
echo "🔧 Starting Backend Server (Port 5001)..."
NODE_ENV=production node server.js > server.log 2>&1 &
BACKEND_PID=$!

# Wait for backend to start
echo "⏳ Waiting for backend to initialize..."
sleep 5

# Check if backend started successfully
if ! ps -p $BACKEND_PID > /dev/null; then
    echo "❌ Backend failed to start. Check server.log for details."
    exit 1
fi

# Start frontend portals
echo "🔵 Starting Admin Portal (Port 3001)..."
cd client
REACT_APP_PORTAL_TYPE=admin PORT=3001 BROWSER=none npm start > ../admin-portal.log 2>&1 &
ADMIN_PID=$!

echo "🟢 Starting Teacher Portal (Port 3002)..."
REACT_APP_PORTAL_TYPE=teacher PORT=3002 BROWSER=none npm start > ../teacher-portal.log 2>&1 &
TEACHER_PID=$!

echo "🟣 Starting Student Portal (Port 3003)..."
REACT_APP_PORTAL_TYPE=student PORT=3003 BROWSER=none npm start > ../student-portal.log 2>&1 &
STUDENT_PID=$!

cd ..

# Wait for all services to start
echo "⏳ Waiting for all services to start..."
sleep 15

echo ""
echo "✅ EduMetrix UK Platform is now running in PRODUCTION mode!"
echo ""
echo "📱 Portal URLs:"
echo "🔵 Admin Portal:    http://localhost:3001"
echo "🟢 Teacher Portal:  http://localhost:3002"
echo "🟣 Student Portal:  http://localhost:3003"
echo "🔧 Backend API:     https://backend.edumetrix.uk"
echo ""
echo "🔐 Authentication Required:"
echo "   Login required for all portals"
echo "   Use credentials created with setup-production-admin.js"
echo ""
echo "📊 Log Files:"
echo "   Backend:  server.log"
echo "   Admin:    admin-portal.log"
echo "   Teacher:  teacher-portal.log"
echo "   Student:  student-portal.log"
echo ""
echo "🛑 To stop all services:"
echo "   Press Ctrl+C or run: pkill -f 'PORT=300' && pkill -f 'node server.js'"
echo ""

# Keep script running and handle Ctrl+C
trap 'echo ""; echo "🛑 Stopping all services..."; pkill -f "PORT=300"; pkill -f "node server.js"; echo "✅ All services stopped."; exit 0' INT

# Wait indefinitely
while true; do
    sleep 1
done