#!/bin/bash

echo "🎓 EduMetrix UK Platform - Development Mode"
echo "==========================================="
echo ""

# Function to check if port is in use
port_in_use() {
    lsof -Pi :$1 -sTCP:LISTEN -t >/dev/null
}

# Function to open URLs
open_urls() {
    sleep 10
    echo ""
    echo "🌐 Opening portals in your default browser..."
    
    if command -v xdg-open > /dev/null; then
        # Linux
        xdg-open http://localhost:3001 >/dev/null 2>&1 &
        sleep 1
        xdg-open http://localhost:3002 >/dev/null 2>&1 &
        sleep 1
        xdg-open http://localhost:3003 >/dev/null 2>&1 &
    elif command -v open > /dev/null; then
        # macOS
        open http://localhost:3001 >/dev/null 2>&1 &
        sleep 1
        open http://localhost:3002 >/dev/null 2>&1 &
        sleep 1
        open http://localhost:3003 >/dev/null 2>&1 &
    elif command -v start > /dev/null; then
        # Windows
        start http://localhost:3001 >/dev/null 2>&1 &
        sleep 1
        start http://localhost:3002 >/dev/null 2>&1 &
        sleep 1
        start http://localhost:3003 >/dev/null 2>&1 &
    else
        echo "🔗 Please manually open these URLs in your browser:"
        echo "   Admin Portal:   http://localhost:3001"
        echo "   Teacher Portal: http://localhost:3002"
        echo "   Student Portal: http://localhost:3003"
    fi
}

# Check if backend port is in use
if port_in_use 5001; then
    echo "⚠️  Backend server appears to be running on port 5001"
else
    echo "🚀 Starting backend server..."
    node server.js > server.log 2>&1 &
    SERVER_PID=$!
    echo "✅ Backend server started (PID: $SERVER_PID)"
    sleep 3
fi

# Kill any existing frontend processes
echo "🧹 Cleaning up any existing frontend processes..."
lsof -ti:3001 | xargs kill -9 2>/dev/null || true
lsof -ti:3002 | xargs kill -9 2>/dev/null || true  
lsof -ti:3003 | xargs kill -9 2>/dev/null || true
sleep 2

echo ""
echo "🚀 Starting all frontend portals..."
echo ""

# Start Admin Portal
echo "🔵 Starting Admin Portal on port 3001..."
cd client
PORT=3001 REACT_APP_PORTAL_TYPE=admin npm start > ../admin-portal.log 2>&1 &
ADMIN_PID=$!
cd ..

sleep 3

# Start Teacher Portal  
echo "🟢 Starting Teacher Portal on port 3002..."
cd client
PORT=3002 REACT_APP_PORTAL_TYPE=teacher npm start > ../teacher-portal.log 2>&1 &
TEACHER_PID=$!
cd ..

sleep 3

# Start Student Portal
echo "🟣 Starting Student Portal on port 3003..."
cd client
PORT=3003 REACT_APP_PORTAL_TYPE=student npm start > ../student-portal.log 2>&1 &
STUDENT_PID=$!

echo ""
echo "⏳ Waiting for all portals to start up..."

# Open URLs in background
open_urls &

sleep 15

echo ""
echo "✅ All portals are now running!"
echo "==============================="
echo ""
echo "🔗 Portal URLs:"
echo "  🔵 Admin Portal:   http://localhost:3001"  
echo "  🟢 Teacher Portal: http://localhost:3002"
echo "  🟣 Student Portal: http://localhost:3003"
echo ""
echo "⚡ Development Features:"
echo "  ✓ Auto-login enabled (no credentials needed)"
echo "  ✓ Mock data for all features"
echo "  ✓ All pages functional"
echo "  ✓ Real-time updates"
echo ""
echo "📋 Development Users:"
echo "  🔵 Admin:   Development Admin"
echo "  🟢 Teacher: Development Teacher"
echo "  🟣 Student: Development Student"
echo ""
echo "📝 Process IDs:"
if [ ! -z "$SERVER_PID" ]; then
    echo "  Backend: $SERVER_PID"
fi
echo "  Admin:   $ADMIN_PID"
echo "  Teacher: $TEACHER_PID" 
echo "  Student: $STUDENT_PID"
echo ""
echo "🛑 To stop all services, press Ctrl+C"
echo ""

# Function to handle cleanup on script exit
cleanup() {
    echo ""
    echo "🛑 Shutting down all services..."
    if [ ! -z "$SERVER_PID" ]; then
        kill $SERVER_PID 2>/dev/null
        echo "   ✓ Backend server stopped"
    fi
    kill $ADMIN_PID $TEACHER_PID $STUDENT_PID 2>/dev/null
    echo "   ✓ All frontend portals stopped"
    echo ""
    echo "👋 Development session ended"
    exit 0
}

# Set up signal handlers
trap cleanup SIGINT SIGTERM

# Keep script running
echo "Press Ctrl+C to stop all services..."
while true; do
    sleep 1
done