#!/bin/bash

# EduMetrix UK Platform - Start All Portals
echo "🎓 Starting EduMetrix UK Platform - All Portals"
echo "=============================================="

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
PURPLE='\033[0;35m'
NC='\033[0m' # No Color

# Function to cleanup background processes
cleanup() {
    echo -e "\n${YELLOW}Shutting down all services...${NC}"
    pkill -f "PORT=300"
    pkill -f "node server.js"
    exit 0
}

# Set trap to cleanup on exit
trap cleanup INT TERM

# Check if we're in the right directory
if [ ! -f "package.json" ]; then
    echo -e "${RED}Error: Please run this script from the project root directory${NC}"
    exit 1
fi

# Install dependencies if needed
if [ ! -d "node_modules" ]; then
    echo -e "${BLUE}Installing backend dependencies...${NC}"
    npm install
fi

if [ ! -d "client/node_modules" ]; then
    echo -e "${BLUE}Installing frontend dependencies...${NC}"
    cd client && npm install && cd ..
fi

# Start the backend server
echo -e "${GREEN}🔧 Starting Backend Server (Port 5001)...${NC}"
node server.js > server.log 2>&1 &
BACKEND_PID=$!

# Wait a moment for backend to start
sleep 3

# Start Admin Portal
echo -e "${BLUE}🔵 Starting Admin Portal (Port 3001)...${NC}"
cd client
PORT=3001 REACT_APP_PORTAL_TYPE=admin npm start > ../admin-portal.log 2>&1 &
ADMIN_PID=$!

# Wait and start Teacher Portal
sleep 5
echo -e "${GREEN}🟢 Starting Teacher Portal (Port 3002)...${NC}"
PORT=3002 REACT_APP_PORTAL_TYPE=teacher npm start > ../teacher-portal.log 2>&1 &
TEACHER_PID=$!

# Wait and start Student Portal
sleep 5
echo -e "${PURPLE}🟣 Starting Student Portal (Port 3003)...${NC}"
PORT=3003 REACT_APP_PORTAL_TYPE=student npm start > ../student-portal.log 2>&1 &
STUDENT_PID=$!

cd ..

# Wait for all services to start
echo -e "${YELLOW}⏳ Waiting for all services to start...${NC}"
sleep 15

# Display status
echo -e "\n${GREEN}✅ All services are now running!${NC}"
echo -e "\n📱 ${BLUE}Portal URLs:${NC}"
echo -e "${BLUE}🔵 Admin Portal:    ${NC}http://localhost:3001"
echo -e "${GREEN}🟢 Teacher Portal:  ${NC}http://localhost:3002"
echo -e "${PURPLE}🟣 Student Portal:  ${NC}http://localhost:3003"
echo -e "${YELLOW}🔧 Backend API:     ${NC}https://backend.edumetrix.uk"

echo -e "\n${GREEN}🎯 Auto-Login Enabled:${NC}"
echo -e "${BLUE}🔵 Admin:    ${NC}Development Admin (admin@edumetrix.uk)"
echo -e "${GREEN}🟢 Teacher:  ${NC}Development Teacher (teacher@edumetrix.uk)"
echo -e "${PURPLE}🟣 Student:  ${NC}Development Student (student@edumetrix.uk)"

echo -e "\n${YELLOW}📊 Log Files:${NC}"
echo -e "Backend:  server.log"
echo -e "Admin:    admin-portal.log"
echo -e "Teacher:  teacher-portal.log"
echo -e "Student:  student-portal.log"

echo -e "\n${YELLOW}🌐 Opening portals in browser...${NC}"

# Open browsers (works on macOS, Linux, and Windows)
if command -v open &> /dev/null; then
    # macOS
    sleep 3
    open http://localhost:3001
    sleep 2
    open http://localhost:3002
    sleep 2
    open http://localhost:3003
elif command -v xdg-open &> /dev/null; then
    # Linux
    sleep 3
    xdg-open http://localhost:3001 &
    sleep 2
    xdg-open http://localhost:3002 &
    sleep 2
    xdg-open http://localhost:3003 &
elif command -v start &> /dev/null; then
    # Windows
    sleep 3
    start http://localhost:3001
    sleep 2
    start http://localhost:3002
    sleep 2
    start http://localhost:3003
fi

echo -e "\n${GREEN}🚀 Platform is ready! Press Ctrl+C to stop all services.${NC}"
echo -e "\n${BLUE}💡 Each portal will automatically log you in and redirect to the dashboard.${NC}"

# Keep script running and wait for user to stop
wait