#!/bin/bash

echo "🚀 Setting up EduMetrix UK Platform..."

# Check if Node.js is installed
if ! command -v node &> /dev/null; then
    echo "❌ Node.js is not installed. Please install Node.js 14+ and try again."
    exit 1
fi

# Check if MongoDB is running
if ! pgrep -x "mongod" > /dev/null; then
    echo "⚠️  MongoDB is not running. Please start MongoDB before continuing."
    echo "   You can start MongoDB with: mongod"
    echo "   Or if using Homebrew: brew services start mongodb-community"
fi

echo "📦 Installing backend dependencies..."
npm install

echo "📦 Installing frontend dependencies..."
cd client
npm install
cd ..

echo "🗂️  Creating uploads directory..."
mkdir -p uploads

echo "📄 Creating default admin user in MongoDB..."
# Create a script to insert admin user
cat > create_admin.js << 'EOF'
const bcrypt = require('bcryptjs');
const { MongoClient } = require('mongodb');

async function createAdmin() {
  const client = new MongoClient('mongodb://localhost:27017');
  
  try {
    await client.connect();
    const db = client.db('edumetrix');
    
    // Check if admin already exists
    const existingAdmin = await db.collection('users').findOne({ username: 'admin@edumetrix.uk' });
    
    if (existingAdmin) {
      console.log('✅ Admin user already exists');
      return;
    }
    
    // Create admin user
    const hashedPassword = await bcrypt.hash('admin123', 10);
    
    const adminUser = {
      username: 'admin@edumetrix.uk',
      password: hashedPassword,
      userType: 'admin',
      isActive: true,
      isDeleted: false,
      createdAt: new Date(),
      updatedAt: new Date()
    };
    
    await db.collection('users').insertOne(adminUser);
    console.log('✅ Admin user created successfully');
    console.log('   Username: admin@edumetrix.uk');
    console.log('   Password: admin123');
    
  } catch (error) {
    console.error('❌ Error creating admin user:', error.message);
  } finally {
    await client.close();
  }
}

createAdmin();
EOF

# Run the admin creation script
node create_admin.js

# Clean up
rm create_admin.js

echo "🎉 Setup completed successfully!"
echo ""
echo "📋 Next steps:"
echo "   1. Make sure MongoDB is running"
echo "   2. Start the development server: npm run dev"
echo "   3. Open http://localhost:3000 in your browser"
echo "   4. Login with:"
echo "      Username: admin@edumetrix.uk"
echo "      Password: admin123"
echo ""
echo "🔧 Available commands:"
echo "   npm run dev     - Start both backend and frontend"
echo "   npm run server  - Start only backend server"
echo "   npm run client  - Start only React client"
echo "   npm start       - Start backend in production mode"
echo ""
echo "📚 Documentation:"
echo "   - Backend API runs on http://localhost:5000"
echo "   - Frontend runs on http://localhost:3000"
echo "   - MongoDB should be running on localhost:27017"
echo "   - Check README.md for detailed information"