import express from 'express';
import TeacherSettings from '../../models/teacher/TeacherSettings.js';
import Teacher from '../../models/teacher/Teacher.js';
import { teacherAuth } from '../../middleware/auth.js';
import { validationResult, body } from 'express-validator';

const router = express.Router();

// Get current teacher's settings
router.get('/', teacherAuth, async (req, res) => {
  try {
    console.log('🔍 Teacher Settings GET - req.user:', req.user);
    console.log('🔍 Teacher Settings GET - req.user type:', typeof req.user);
    console.log('🔍 Teacher Settings GET - req.user keys:', req.user ? Object.keys(req.user) : 'undefined');

    if (!req.user) {
      console.error('❌ Teacher Settings GET - req.user is undefined');
      return res.status(401).json({
        success: false,
        message: 'User not authenticated'
      });
    }

    // For teachers, req.user should have _id from the Teacher document
    const teacherId = req.user._id || req.user.id;

    let settings = await TeacherSettings.findOne({ teacherId });

    // If no settings exist, create default settings
    if (!settings) {
      const defaultSettings = TeacherSettings.getDefaultSettings(teacherId);
      settings = new TeacherSettings(defaultSettings);
      await settings.save();
    }

    res.json({
      success: true,
      settings
    });
  } catch (error) {
    console.error('Error fetching teacher settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch settings',
      error: error.message
    });
  }
});

// Update appearance settings
router.put('/appearance', [
  teacherAuth,
  body('colorTheme').optional().isIn(['blue', 'purple', 'green', 'orange', 'pink', 'teal', 'cyan', 'indigo']),
  body('darkMode').optional().isBoolean()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const teacherId = req.user._id || req.user.id;
    const { colorTheme, darkMode } = req.body;

    let settings = await TeacherSettings.findOne({ teacherId });

    if (!settings) {
      settings = new TeacherSettings(TeacherSettings.getDefaultSettings(teacherId));
    }

    // Update appearance settings
    if (colorTheme !== undefined) settings.appearance.colorTheme = colorTheme;
    if (darkMode !== undefined) settings.appearance.darkMode = darkMode;

    await settings.save();

    res.json({
      success: true,
      message: 'Appearance settings updated successfully',
      settings: settings.appearance
    });
  } catch (error) {
    console.error('Error updating appearance settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update appearance settings',
      error: error.message
    });
  }
});

// Update notification settings
router.put('/notifications', [
  teacherAuth,
  body('assignments').optional().isBoolean(),
  body('grades').optional().isBoolean(),
  body('exams').optional().isBoolean(),
  body('announcements').optional().isBoolean(),
  body('messages').optional().isBoolean(),
  body('classReminders').optional().isBoolean()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const teacherId = req.user._id || req.user.id;
    const { assignments, grades, exams, announcements, messages, classReminders } = req.body;

    let settings = await TeacherSettings.findOne({ teacherId });

    if (!settings) {
      settings = new TeacherSettings(TeacherSettings.getDefaultSettings(teacherId));
    }

    // Update notification settings
    if (assignments !== undefined) settings.notifications.assignments = assignments;
    if (grades !== undefined) settings.notifications.grades = grades;
    if (exams !== undefined) settings.notifications.exams = exams;
    if (announcements !== undefined) settings.notifications.announcements = announcements;
    if (messages !== undefined) settings.notifications.messages = messages;
    if (classReminders !== undefined) settings.notifications.classReminders = classReminders;

    await settings.save();

    res.json({
      success: true,
      message: 'Notification settings updated successfully',
      settings: settings.notifications
    });
  } catch (error) {
    console.error('Error updating notification settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update notification settings',
      error: error.message
    });
  }
});

// Update profile settings
router.put('/profile', [
  teacherAuth,
  body('name').optional({ values: 'falsy' }).trim().isLength({ min: 1, max: 100 }),
  body('email').optional({ values: 'falsy' }).custom((value) => {
    if (value === '' || value == null) return true;
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value);
  }),
  body('phone').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 20 }),
  body('timezone').optional({ values: 'falsy' }).isString(),
  body('bio').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 500 })
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const teacherId = req.user._id || req.user.id;
    const { name, email, phone, timezone, bio } = req.body;

    let settings = await TeacherSettings.findOne({ teacherId });

    if (!settings) {
      settings = new TeacherSettings(TeacherSettings.getDefaultSettings(teacherId));
    }

    // Update profile settings
    if (name !== undefined) settings.profile.name = name;
    if (email !== undefined) settings.profile.email = email;
    if (phone !== undefined) settings.profile.phone = phone;
    if (timezone !== undefined) settings.profile.timezone = timezone;
    if (bio !== undefined) settings.profile.bio = bio;

    await settings.save();

    // Also update the main teacher record if needed
    if (name || email || phone) {
      const updateData = {};
      if (name) updateData.teacherName = name;
      if (email) updateData.email = email;
      if (phone) {
        // Get current phoneNumber or create new one
        const currentTeacher = await Teacher.findById(teacherId);
        updateData.phoneNumber = {
          countryCode: currentTeacher?.phoneNumber?.countryCode || '+1',
          number: phone
        };
      }

      await Teacher.findByIdAndUpdate(teacherId, updateData);
    }

    res.json({
      success: true,
      message: 'Profile settings updated successfully',
      settings: settings.profile
    });
  } catch (error) {
    console.error('Error updating profile settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update profile settings',
      error: error.message
    });
  }
});

// Update privacy settings
router.put('/privacy', [
  teacherAuth,
  body('dataAnalytics').optional().isBoolean(),
  body('promotionalEmails').optional().isBoolean(),
  body('profileVisibility').optional().isIn(['public', 'students', 'private'])
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const teacherId = req.user._id || req.user.id;
    const { dataAnalytics, promotionalEmails, profileVisibility } = req.body;

    let settings = await TeacherSettings.findOne({ teacherId });

    if (!settings) {
      settings = new TeacherSettings(TeacherSettings.getDefaultSettings(teacherId));
    }

    // Update privacy settings
    if (dataAnalytics !== undefined) settings.privacy.dataAnalytics = dataAnalytics;
    if (promotionalEmails !== undefined) settings.privacy.promotionalEmails = promotionalEmails;
    if (profileVisibility !== undefined) settings.privacy.profileVisibility = profileVisibility;

    await settings.save();

    res.json({
      success: true,
      message: 'Privacy settings updated successfully',
      settings: settings.privacy
    });
  } catch (error) {
    console.error('Error updating privacy settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update privacy settings',
      error: error.message
    });
  }
});

// Update language settings
router.put('/language', [
  teacherAuth,
  body('language').isIn(['en', 'es', 'fr', 'de', 'it', 'pt', 'zh', 'ja'])
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const teacherId = req.user._id || req.user.id;
    const { language } = req.body;

    let settings = await TeacherSettings.findOne({ teacherId });

    if (!settings) {
      settings = new TeacherSettings(TeacherSettings.getDefaultSettings(teacherId));
    }

    settings.language = language;
    await settings.save();

    res.json({
      success: true,
      message: 'Language settings updated successfully',
      language: settings.language
    });
  } catch (error) {
    console.error('Error updating language settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update language settings',
      error: error.message
    });
  }
});

// Update teaching preferences
router.put('/teaching-preferences', [
  teacherAuth,
  body('autoGrading').optional().isBoolean(),
  body('reminderEmails').optional().isBoolean(),
  body('calendarSync').optional().isBoolean()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const teacherId = req.user._id || req.user.id;
    const { autoGrading, reminderEmails, calendarSync } = req.body;

    let settings = await TeacherSettings.findOne({ teacherId });

    if (!settings) {
      settings = new TeacherSettings(TeacherSettings.getDefaultSettings(teacherId));
    }

    // Update teaching preferences
    if (autoGrading !== undefined) settings.teachingPreferences.autoGrading = autoGrading;
    if (reminderEmails !== undefined) settings.teachingPreferences.reminderEmails = reminderEmails;
    if (calendarSync !== undefined) settings.teachingPreferences.calendarSync = calendarSync;

    await settings.save();

    res.json({
      success: true,
      message: 'Teaching preferences updated successfully',
      settings: settings.teachingPreferences
    });
  } catch (error) {
    console.error('Error updating teaching preferences:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update teaching preferences',
      error: error.message
    });
  }
});

// Update all settings at once
router.put('/', [
  teacherAuth,
  // Appearance validation
  body('appearance.colorTheme').optional().isIn(['blue', 'purple', 'green', 'orange', 'pink', 'teal', 'cyan', 'indigo']),
  body('appearance.darkMode').optional().isBoolean(),
  // Notification validation
  body('notifications.assignments').optional().isBoolean(),
  body('notifications.grades').optional().isBoolean(),
  body('notifications.exams').optional().isBoolean(),
  body('notifications.announcements').optional().isBoolean(),
  body('notifications.messages').optional().isBoolean(),
  body('notifications.classReminders').optional().isBoolean(),
  // Profile validation
  body('profile.name').optional({ values: 'falsy' }).trim().isLength({ min: 1, max: 100 }),
  body('profile.email').optional({ values: 'falsy' }).custom((value) => {
    if (value === '' || value == null) return true;
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value);
  }),
  body('profile.phone').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 20 }),
  body('profile.timezone').optional({ values: 'falsy' }).isString(),
  body('profile.bio').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 500 }),
  // Privacy validation
  body('privacy.dataAnalytics').optional().isBoolean(),
  body('privacy.promotionalEmails').optional().isBoolean(),
  body('privacy.profileVisibility').optional().isIn(['public', 'students', 'private']),
  // Language validation
  body('language').optional().isIn(['en', 'es', 'fr', 'de', 'it', 'pt', 'zh', 'ja']),
  // Teaching preferences validation
  body('teachingPreferences.autoGrading').optional().isBoolean(),
  body('teachingPreferences.reminderEmails').optional().isBoolean(),
  body('teachingPreferences.calendarSync').optional().isBoolean()
], async (req, res) => {
  try {
    console.log('💾 Teacher Settings PUT BULK - req.user:', req.user);
    console.log('💾 Teacher Settings PUT BULK - req.body:', req.body);

    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    if (!req.user) {
      console.error('❌ Teacher Settings PUT BULK - req.user is undefined');
      return res.status(401).json({
        success: false,
        message: 'User not authenticated'
      });
    }

    // For teachers, req.user should have _id from the Teacher document
    const teacherId = req.user._id || req.user.id;
    const { appearance, notifications, profile, privacy, language, teachingPreferences } = req.body;

    let settings = await TeacherSettings.findOne({ teacherId });

    if (!settings) {
      settings = new TeacherSettings(TeacherSettings.getDefaultSettings(teacherId));
    }

    // Update all categories
    if (appearance) {
      Object.assign(settings.appearance, appearance);
    }
    if (notifications) {
      Object.assign(settings.notifications, notifications);
    }
    if (profile) {
      Object.assign(settings.profile, profile);
    }
    if (privacy) {
      Object.assign(settings.privacy, privacy);
    }
    if (language) {
      settings.language = language;
    }
    if (teachingPreferences) {
      Object.assign(settings.teachingPreferences, teachingPreferences);
    }

    await settings.save();

    // Update main teacher record if profile data changed
    if (profile && (profile.name || profile.email || profile.phone)) {
      const updateData = {};
      if (profile.name) updateData.teacherName = profile.name;
      if (profile.email) updateData.email = profile.email;
      if (profile.phone) {
        const currentTeacher = await Teacher.findById(teacherId);
        updateData.phoneNumber = {
          countryCode: currentTeacher?.phoneNumber?.countryCode || '+1',
          number: profile.phone
        };
      }

      await Teacher.findByIdAndUpdate(teacherId, updateData);
    }

    res.json({
      success: true,
      message: 'Settings updated successfully',
      settings
    });
  } catch (error) {
    console.error('Error updating settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update settings',
      error: error.message
    });
  }
});

// Reset settings to default
router.post('/reset', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id || req.user.id;

    // Delete existing settings
    await TeacherSettings.findOneAndDelete({ teacherId });

    // Create new default settings
    const defaultSettings = TeacherSettings.getDefaultSettings(teacherId);
    const newSettings = new TeacherSettings(defaultSettings);
    await newSettings.save();

    res.json({
      success: true,
      message: 'Settings reset to default successfully',
      settings: newSettings
    });
  } catch (error) {
    console.error('Error resetting settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to reset settings',
      error: error.message
    });
  }
});

export default router;