import express from 'express';
import mongoose from 'mongoose';
import Exam from '../../models/teacher/Exam.js';
import Question from '../../models/teacher/Question.js';
import Submission from '../../models/student/Submission.js';
import { teacherAuth } from '../../middleware/auth.js';

const router = express.Router();

// Create a new exam
router.post('/', teacherAuth, async (req, res) => {
  try {
    const { title, description, teacherId, classId, duration, passingMarks } = req.body;
    
    const exam = new Exam({
      title,
      description,
      teacherId,
      classId,
      duration,
      passingMarks
    });
    
    await exam.save();
    res.status(201).json({ success: true, exam });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Get exams for a teacher
router.get('/', async (req, res) => {
  try {
    const { teacherId, classId } = req.query;
    const filter = {};
    
    if (teacherId) filter.teacherId = teacherId;
    if (classId) filter.classId = classId;
    
    const exams = await Exam.find(filter).sort({ createdAt: -1 });
    res.json({ success: true, exams });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get exam by ID with questions
router.get('/:id', async (req, res) => {
  try {
    const exam = await Exam.findById(req.params.id);
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    const questions = await Question.find({ examId: exam._id }).sort({ order: 1 });
    
    res.json({ success: true, exam, questions });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Update exam
router.put('/:id', async (req, res) => {
  try {
    const exam = await Exam.findByIdAndUpdate(
      req.params.id,
      req.body,
      { new: true, runValidators: true }
    );
    
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    res.json({ success: true, exam });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Delete exam
router.delete('/:id', async (req, res) => {
  try {
    const exam = await Exam.findByIdAndDelete(req.params.id);
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    await Question.deleteMany({ examId: exam._id });
    await Submission.deleteMany({ examId: exam._id });
    
    res.json({ success: true, message: 'Exam deleted successfully' });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Update exam status (ready, live, completed)
router.put('/:id/status', async (req, res) => {
  try {
    const { status } = req.body;
    const exam = await Exam.findById(req.params.id);
    
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    if (status === 'ready') {
      const questions = await Question.find({ examId: exam._id });
      if (questions.length === 0) {
        return res.status(400).json({ 
          success: false, 
          error: 'Cannot mark exam as ready without questions' 
        });
      }
      
      const totalMarks = questions.reduce((sum, q) => sum + q.marks, 0);
      exam.totalMarks = totalMarks;
    }
    
    if (status === 'live') {
      exam.startedAt = new Date();
    }
    
    if (status === 'completed') {
      exam.endedAt = new Date();
    }
    
    exam.status = status;
    await exam.save();
    
    // Emit socket event for real-time updates
    req.app.get('io').emit('examStatusChanged', { examId: exam._id, status, exam });
    
    res.json({ success: true, exam });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Add question to exam
router.post('/:id/questions', async (req, res) => {
  try {
    const { questionText, options, marks, order } = req.body;
    
    const question = new Question({
      examId: req.params.id,
      questionText,
      options: options.map((opt, index) => ({
        optionId: `option_${index + 1}`,
        text: opt.text,
        isCorrect: opt.isCorrect
      })),
      marks,
      order
    });
    
    await question.save();
    res.status(201).json({ success: true, question });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Get questions for an exam
router.get('/:id/questions', async (req, res) => {
  try {
    const questions = await Question.find({ examId: req.params.id }).sort({ order: 1 });
    res.json({ success: true, questions });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Update question
router.put('/:id/questions/:questionId', async (req, res) => {
  try {
    const question = await Question.findByIdAndUpdate(
      req.params.questionId,
      req.body,
      { new: true, runValidators: true }
    );
    
    if (!question) {
      return res.status(404).json({ success: false, error: 'Question not found' });
    }
    
    res.json({ success: true, question });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Delete question
router.delete('/:id/questions/:questionId', async (req, res) => {
  try {
    const question = await Question.findByIdAndDelete(req.params.questionId);
    if (!question) {
      return res.status(404).json({ success: false, error: 'Question not found' });
    }
    
    res.json({ success: true, message: 'Question deleted successfully' });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get exam results/submissions
router.get('/:id/results', async (req, res) => {
  try {
    const submissions = await Submission.find({ examId: req.params.id })
      .sort({ totalScore: -1, submittedAt: 1 });
    
    res.json({ success: true, submissions });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

export default router;