import express from 'express';
import moment from 'moment-timezone';
import Class from '../../models/Class.js';
import Student from '../../models/student/Student.js';
import Teacher from '../../models/teacher/Teacher.js';
import { teacherAuth } from '../../middleware/auth.js';

const router = express.Router();

// Get attendance statistics (must be before /:classId route)
router.get('/stats', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;
    const { studentId, startDate, endDate } = req.query;

    // Build query
    const query = {
      teacherId: teacherId,
      // status: 'completed',
      'attendance.markedAt': { $exists: true }
    };

    // Date range filter
    if (startDate && endDate) {
      query.scheduledDate = {
        $gte: moment.utc(startDate).startOf('day').toDate(),
        $lte: moment.utc(endDate).endOf('day').toDate()
      };
    } else {
      // Default to last 30 days
      query.scheduledDate = {
        $gte: moment().subtract(30, 'days').startOf('day').toDate(),
        $lte: moment().endOf('day').toDate()
      };
    }

    // Student filter
    if (studentId) {
      query.studentId = studentId;
    }

    const classes = await Class.find(query)
      .populate('studentId', 'studentName email class')
      .sort({ scheduledDate: -1 });

    // Calculate statistics
    const stats = {
      total: classes.length,
      present: 0,
      late: 0,
      absent: 0,
      attendanceRate: 0,
      punctualityRate: 0,
      byStudent: {}
    };

    classes.forEach(cls => {
      const studentId = cls.studentId._id.toString();

      // Overall stats
      if (cls.attendance.present && !cls.attendance.late) {
        stats.present++;
      } else if (cls.attendance.late) {
        stats.late++;
      } else if (cls.attendance.absent) {
        stats.absent++;
      }

      // Per-student stats
      if (!stats.byStudent[studentId]) {
        stats.byStudent[studentId] = {
          studentName: cls.studentId.studentName,
          class: cls.studentId.class,
          total: 0,
          present: 0,
          late: 0,
          absent: 0
        };
      }

      stats.byStudent[studentId].total++;
      if (cls.attendance.present && !cls.attendance.late) {
        stats.byStudent[studentId].present++;
      } else if (cls.attendance.late) {
        stats.byStudent[studentId].late++;
      } else if (cls.attendance.absent) {
        stats.byStudent[studentId].absent++;
      }
    });

    // Calculate rates
    if (stats.total > 0) {
      stats.attendanceRate = (((stats.present + stats.late) / stats.total) * 100).toFixed(1);
      stats.punctualityRate = ((stats.present / stats.total) * 100).toFixed(1);
    }

    // Calculate per-student rates
    Object.keys(stats.byStudent).forEach(studentId => {
      const studentStats = stats.byStudent[studentId];
      if (studentStats.total > 0) {
        studentStats.attendanceRate = (((studentStats.present + studentStats.late) / studentStats.total) * 100).toFixed(1);
        studentStats.punctualityRate = ((studentStats.present / studentStats.total) * 100).toFixed(1);
      }
    });

    res.json({
      success: true,
      stats
    });
  } catch (error) {
    console.error('Get attendance stats error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Get students list for the teacher (must be before /:classId route)
router.get('/students', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;

    // Find all unique students who have classes with this teacher
    const classes = await Class.find({ teacherId: teacherId })
      .populate('studentId', 'studentName email class country subjects')
      .distinct('studentId');

    const students = await Student.find({ _id: { $in: classes } })
      .select('studentName email class country subjects');

    res.json({
      success: true,
      students
    });
  } catch (error) {
    console.error('Get students error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Get attendance data for teacher with filters
router.get('/', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;
    const { date, startDate, endDate, studentId, subject } = req.query;

    // Build query
    const query = {
      teacherId: teacherId,
      status: { $in: ['completed', 'scheduled'] }
    };

    // Date filters
    if (date) {
      const selectedDate = moment(date).startOf('day');
      const nextDay = selectedDate.clone().add(1, 'day');
      query.scheduledDate = {
        $gte: selectedDate.toDate(),
        $lt: nextDay.toDate()
      };
    } else if (startDate && endDate) {
      query.scheduledDate = {
        $gte: moment(startDate).startOf('day').toDate(),
        $lte: moment(endDate).endOf('day').toDate()
      };
    }

    // Student filter
    if (studentId) {
      query.studentId = studentId;
    }

    // Subject filter
    if (subject && subject !== 'all') {
      query.subject = subject;
    }

    const classes = await Class.find(query)
      .populate('studentId', 'studentName email class country')
      .populate('teacherId', 'name')
      .sort({ scheduledDate: -1 });

    res.json({
      success: true,
      classes,
      count: classes.length
    });
  } catch (error) {
    console.error('Get attendance error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Save attendance for a single class
router.post('/:classId', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;
    const { classId } = req.params;
    const { present, late, absent, notes } = req.body;

    // Validate attendance data
    if (![present, late, absent].includes(true)) {
      return res.status(400).json({
        message: 'Please select attendance status (present, late, or absent)'
      });
    }

    const classData = await Class.findOne({
      _id: classId,
      teacherId: teacherId
    });

    if (!classData) {
      return res.status(404).json({ message: 'Class not found' });
    }

    // Check if class is completed
    // if (classData.status !== 'completed') {
    //   return res.status(400).json({
    //     message: 'Attendance can only be marked for completed classes'
    //   });
    // }

    // Update attendance
    classData.attendance = {
      ...classData.attendance,
      studentJoined: present || late,
      present: present,
      late: late,
      absent: absent,
      notes: notes || '',
      markedAt: new Date(),
      markedBy: teacherId
    };

    await classData.save();

    const updatedClass = await Class.findById(classId)
      .populate('studentId', 'studentName email class country')
      .populate('teacherId', 'name');

    res.json({
      success: true,
      message: 'Attendance saved successfully',
      class: updatedClass
    });
  } catch (error) {
    console.error('Save attendance error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Bulk save attendance for multiple classes
router.post('/bulk', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;
    const { attendanceData } = req.body; // Array of { classId, present, late, absent, notes }

    if (!Array.isArray(attendanceData) || attendanceData.length === 0) {
      return res.status(400).json({
        message: 'Invalid attendance data format'
      });
    }

    const results = [];
    const errors = [];

    for (const attendance of attendanceData) {
      try {
        const { classId, present, late, absent, notes } = attendance;

        const classData = await Class.findOne({
          _id: classId,
          teacherId: teacherId,
          status: 'completed'
        });

        if (!classData) {
          errors.push({ classId, error: 'Class not found or not completed' });
          continue;
        }

        // Update attendance
        classData.attendance = {
          ...classData.attendance,
          studentJoined: present || late,
          present: present,
          late: late,
          absent: absent,
          notes: notes || '',
          markedAt: new Date(),
          markedBy: teacherId
        };

        await classData.save();
        results.push({ classId, status: 'success' });
      } catch (error) {
        errors.push({ classId: attendance.classId, error: error.message });
      }
    }

    res.json({
      success: true,
      message: `Attendance saved for ${results.length} classes`,
      results,
      errors: errors.length > 0 ? errors : undefined
    });
  } catch (error) {
    console.error('Bulk save attendance error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Update attendance for a class
router.put('/:classId', teacherAuth, async (req, res) => {
  try {
    const teacherId = req.user._id;
    const { classId } = req.params;
    const { present, late, absent, notes } = req.body;

    const classData = await Class.findOne({
      _id: classId,
      teacherId: teacherId
    });

    if (!classData) {
      return res.status(404).json({ message: 'Class not found' });
    }

    // Update attendance
    classData.attendance = {
      ...classData.attendance,
      studentJoined: present || late,
      present: present,
      late: late,
      absent: absent,
      notes: notes || '',
      markedAt: new Date(),
      markedBy: teacherId,
      teacherJoined: true

    };
    classData.status = 'completed';
    classData.attendanceStatus = {
      studentAttended: present || late,
      teacherAttended: true,
    }

    await classData.save();

    const updatedClass = await Class.findById(classId)
      .populate('studentId', 'studentName email class country')
      .populate('teacherId', 'name');

    res.json({
      success: true,
      message: 'Attendance updated successfully',
      class: updatedClass
    });
  } catch (error) {
    console.error('Update attendance error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

export default router;