const express = require('express');
const router = express.Router();
const QuickTest = require('../../models/teacher/QuickTest');

// Get available quick tests for student
router.get('/', async (req, res) => {
  try {
    const { status, limit = 50, page = 1 } = req.query;
    
    const filter = {
      studentId: req.user.userId
    };
    
    if (status) {
      filter.status = status;
    } else {
      // Default to active tests
      filter.status = 'active';
    }

    const skip = (page - 1) * limit;

    const quickTests = await QuickTest.find(filter)
      .populate('teacherId', 'teacherName')
      .populate('studentId', 'studentName')
      .select('-questions.correctAnswer -submissions.answers')
      .sort({ createdAt: -1 })
      .limit(parseInt(limit))
      .skip(skip);

    // Filter out questions correct answers and add submission status
    const testsWithStatus = quickTests.map(test => {
      const testObj = test.toObject();
      
      // Remove correct answers from questions for students
      if (testObj.questions) {
        testObj.questions = testObj.questions.map(q => ({
          question: q.question,
          options: q.options.map(opt => ({
            text: opt.text,
            _id: opt._id
          })),
          marks: q.marks,
          _id: q._id
        }));
      }

      // Check if student has already submitted
      const hasSubmitted = test.submissions.some(
        sub => sub.studentId.toString() === req.user.userId
      );

      testObj.hasSubmitted = hasSubmitted;
      testObj.canSubmit = test.canSubmit(req.user.userId);
      testObj.isActive = test.isActive();
      
      // Remove all submissions data for privacy
      delete testObj.submissions;

      return testObj;
    });

    const total = await QuickTest.countDocuments(filter);

    res.json({
      success: true,
      data: testsWithStatus,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get student quick tests error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick tests',
      error: error.message
    });
  }
});

// Get specific quick test for taking
router.get('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findById(testId)
      .populate('teacherId', 'teacherName')
      .populate('classId', 'className subject');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Check if test is assigned to this student
    if (quickTest.studentId.toString() !== req.user.userId) {
      return res.status(403).json({
        success: false,
        message: 'This test is not assigned to you'
      });
    }

    // Check if student can submit
    if (!quickTest.canSubmit(req.user.userId)) {
      const hasSubmitted = quickTest.submissions.some(
        sub => sub.studentId.toString() === req.user.userId
      );

      return res.status(400).json({
        success: false,
        message: hasSubmitted ? 
          'You have already submitted this test' : 
          'This test is not available for submission',
        hasSubmitted,
        isActive: quickTest.isActive()
      });
    }

    // Remove correct answers from questions
    const testData = quickTest.toObject();
    testData.questions = testData.questions.map(q => ({
      question: q.question,
      options: q.options.map(opt => ({
        text: opt.text,
        _id: opt._id
      })),
      marks: q.marks,
      _id: q._id
    }));

    // Remove submissions data
    delete testData.submissions;

    // Add test status info
    testData.canSubmit = true;
    testData.isActive = quickTest.isActive();
    testData.timeRemaining = quickTest.endTime ? 
      Math.max(0, Math.floor((quickTest.endTime - new Date()) / 1000)) : null;

    res.json({
      success: true,
      data: testData
    });

  } catch (error) {
    console.error('Get quick test for student error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick test',
      error: error.message
    });
  }
});

// Submit quick test answers
router.post('/:testId/submit', async (req, res) => {
  try {
    const { testId } = req.params;
    const { answers, startTime } = req.body;

    if (!answers || !Array.isArray(answers)) {
      return res.status(400).json({
        success: false,
        message: 'Answers are required and must be an array'
      });
    }

    const quickTest = await QuickTest.findById(testId);

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Check if student can submit
    if (!quickTest.canSubmit(req.user.userId)) {
      const hasSubmitted = quickTest.submissions.some(
        sub => sub.studentId.toString() === req.user.userId
      );

      return res.status(400).json({
        success: false,
        message: hasSubmitted ? 
          'You have already submitted this test' : 
          'This test is not available for submission'
      });
    }

    // Validate answers format
    if (answers.length !== quickTest.questions.length) {
      return res.status(400).json({
        success: false,
        message: 'Number of answers does not match number of questions'
      });
    }

    // Calculate score and prepare submission
    const submissionAnswers = [];
    let totalMarks = 0;

    answers.forEach((answer, index) => {
      const question = quickTest.questions[index];
      if (question) {
        const isCorrect = answer.selectedOption === question.correctAnswer;
        const marksObtained = isCorrect ? question.marks : 0;
        
        submissionAnswers.push({
          questionIndex: index,
          selectedOption: answer.selectedOption,
          isCorrect,
          marksObtained
        });

        totalMarks += marksObtained;
      }
    });

    const percentage = quickTest.totalMarks > 0 ? 
      Math.round((totalMarks / quickTest.totalMarks) * 100) : 0;

    // Calculate test duration
    const submissionTime = new Date();
    const testDuration = startTime ? 
      Math.floor((submissionTime - new Date(startTime)) / 1000) : null;

    // Add submission to quick test
    const submission = {
      studentId: req.user.userId,
      answers: submissionAnswers,
      totalMarks,
      percentage,
      submittedAt: submissionTime,
      duration: testDuration
    };

    quickTest.submissions.push(submission);
    await quickTest.save();

    // Emit to teacher that student submitted
    const io = req.app.get('io');
    if (io) {
      io.to(`teacher-${quickTest.teacherId}`).emit('quick-test-submission', {
        testId: quickTest._id,
        studentId: req.user.userId,
        percentage,
        submittedAt: submissionTime
      });
    }

    res.json({
      success: true,
      message: 'Test submitted successfully',
      data: {
        totalMarks: quickTest.totalMarks,
        obtainedMarks: totalMarks,
        percentage,
        submittedAt: submissionTime,
        duration: testDuration
      }
    });

  } catch (error) {
    console.error('Submit quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to submit test',
      error: error.message
    });
  }
});

// Get student's submission for a test
router.get('/:testId/submission', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findById(testId)
      .populate('teacherId', 'teacherName')
      .populate('classId', 'className subject');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Find student's submission
    const submission = quickTest.submissions.find(
      sub => sub.studentId.toString() === req.user.userId
    );

    if (!submission) {
      return res.status(404).json({
        success: false,
        message: 'Submission not found'
      });
    }

    // Prepare detailed results with correct answers (only after test completion)
    let detailedAnswers = null;
    if (quickTest.status === 'completed') {
      detailedAnswers = submission.answers.map((answer, index) => {
        const question = quickTest.questions[index];
        return {
          questionIndex: index,
          question: question.question,
          options: question.options,
          selectedOption: answer.selectedOption,
          correctOption: question.correctAnswer,
          isCorrect: answer.isCorrect,
          marksObtained: answer.marksObtained,
          totalMarks: question.marks
        };
      });
    }

    res.json({
      success: true,
      data: {
        test: {
          _id: quickTest._id,
          title: quickTest.title,
          subject: quickTest.subject,
          teacherName: quickTest.teacherId.teacherName,
          className: quickTest.classId.className,
          totalMarks: quickTest.totalMarks,
          status: quickTest.status
        },
        submission: {
          totalMarks: submission.totalMarks,
          percentage: submission.percentage,
          submittedAt: submission.submittedAt,
          duration: submission.duration,
          answers: detailedAnswers
        }
      }
    });

  } catch (error) {
    console.error('Get student submission error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch submission',
      error: error.message
    });
  }
});

// Get student's quick test history
router.get('/history/all', async (req, res) => {
  try {
    const { limit = 50, page = 1 } = req.query;
    
    const quickTests = await QuickTest.find({
      studentId: req.user.userId,
      'submissions.studentId': req.user.userId
    })
    .populate('teacherId', 'teacherName')
    .populate('studentId', 'studentName')
    .select('title subject topic teacherId studentId totalMarks status startTime endTime submissions.$ createdAt')
    .sort({ createdAt: -1 })
    .limit(parseInt(limit))
    .skip((page - 1) * limit);

    const history = quickTests.map(test => {
      const submission = test.submissions.find(
        sub => sub.studentId.toString() === req.user.userId
      );

      return {
        testId: test._id,
        title: test.title,
        subject: test.subject,
        topic: test.topic,
        teacherName: test.teacherId.teacherName,
        totalMarks: test.totalMarks,
        status: test.status,
        startTime: test.startTime,
        endTime: test.endTime,
        createdAt: test.createdAt,
        submission: {
          totalMarks: submission.totalMarks,
          percentage: submission.percentage,
          submittedAt: submission.submittedAt,
          duration: submission.duration
        }
      };
    });

    const total = await QuickTest.countDocuments({
      studentId: req.user.userId,
      'submissions.studentId': req.user.userId
    });

    res.json({
      success: true,
      data: history,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get student history error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch test history',
      error: error.message
    });
  }
});

module.exports = router;