import express from 'express';
import mongoose from 'mongoose';
import crypto from 'crypto';
import QuickTest from '../../models/teacher/QuickTest.js';

const router = express.Router();

// Helper function to convert development IDs to valid ObjectIds
const getValidObjectId = (id) => {
  if (mongoose.Types.ObjectId.isValid(id)) {
    return id; // Valid ObjectId
  } else if (typeof id === 'string' && id.startsWith('dev-')) {
    // For development mode with string IDs, create a consistent ObjectId
    const hash = crypto.createHash('md5').update(id).digest('hex');
    const objectIdHex = hash.substring(0, 24);
    return new mongoose.Types.ObjectId(objectIdHex);
  }
  return id; // Return as-is if not a dev string
};

// Get available quick tests for student
router.get('/', async (req, res) => {
  try {
    const { status = 'all', limit = 50, page = 1 } = req.query;
    const studentId = req.user._id;

    // Build query
    const query = { studentId: getValidObjectId(studentId) };
    if (status !== 'all') {
      query.status = status;
    }

    const skip = (page - 1) * limit;

    const quickTests = await QuickTest.find(query)
      .populate('teacherId', 'name')
      .sort({ createdAt: -1 })
      .limit(parseInt(limit))
      .skip(skip);

    const total = await QuickTest.countDocuments(query);

    // Format tests for frontend
    const formattedTests = quickTests.map(test => {
      const isActive = test.status === 'active';

      const hasSubmitted = test.submissions.some(sub =>
        sub.studentId.toString() === studentId.toString()
      );

      // Check if student has started the test
      const session = test.studentSessions.find(s =>
        s.studentId.toString() === studentId.toString()
      );

      let timeRemaining = null;
      let studentStartTime = null;
      let studentEndTime = null;

      if (session) {
        // Student has started - calculate remaining time based on their session
        studentStartTime = session.startedAt;
        studentEndTime = new Date(session.startedAt.getTime() + test.duration * 60000);
        const now = new Date();
        timeRemaining = Math.max(0, Math.floor((studentEndTime - now) / 1000));
      } else if (isActive) {
        // Student hasn't started yet - show full duration available
        timeRemaining = test.duration * 60;
      }

      return {
        _id: test._id,
        title: test.title,
        description: test.description,
        subject: test.subject,
        topic: test.topic,
        teacherId: {
          _id: test.teacherId?._id || test.teacherId,
          teacherName: test.teacherId?.name || 'Teacher'
        },
        duration: test.duration,
        totalMarks: test.totalMarks,
        questions: test.questions,
        status: test.status,
        isActive,
        hasSubmitted,
        hasStarted: !!session,
        canSubmit: isActive && !hasSubmitted,
        timeRemaining,
        studentStartTime,
        studentEndTime,
        createdAt: test.createdAt,
        startTime: test.startTime,
        endTime: test.endTime
      };
    });

    res.json({
      success: true,
      data: formattedTests,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get student quick tests error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick tests',
      error: error.message
    });
  }
});

// Get specific quick test for taking
router.get('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;
    const studentId = req.user._id;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      studentId: getValidObjectId(studentId)
    }).populate('teacherId', 'name');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Check if test is active
    if (quickTest.status !== 'active') {
      return res.status(400).json({
        success: false,
        message: 'This test is not currently active'
      });
    }

    // Check if student already submitted
    const hasSubmitted = quickTest.submissions.some(sub =>
      sub.studentId.toString() === studentId.toString()
    );

    if (hasSubmitted) {
      return res.status(400).json({
        success: false,
        message: 'You have already submitted this test'
      });
    }

    // Create or get student session (this is when student clicks "Take Test")
    let session = quickTest.studentSessions.find(s =>
      s.studentId.toString() === studentId.toString()
    );

    const now = new Date();

    if (!session) {
      // Student is starting the test for the first time - create session
      console.log(`🚀 Student ${studentId} starting test ${testId} at ${now}`);
      quickTest.studentSessions.push({
        studentId: getValidObjectId(studentId),
        startedAt: now
      });
      await quickTest.save();
      session = quickTest.studentSessions[quickTest.studentSessions.length - 1];
    }

    // Calculate time remaining based on student's individual start time
    const timeRemaining = quickTest.getStudentTimeRemaining(studentId);
    const studentEndTime = new Date(session.startedAt.getTime() + quickTest.duration * 60000);

    console.log(`⏱️ Student session started at: ${session.startedAt}`);
    console.log(`⏱️ Student end time: ${studentEndTime}`);
    console.log(`⏱️ Time remaining: ${timeRemaining} seconds`);

    const testData = {
      _id: quickTest._id,
      title: quickTest.title,
      description: quickTest.description,
      subject: quickTest.subject,
      topic: quickTest.topic,
      teacherId: {
        _id: quickTest.teacherId?._id || quickTest.teacherId,
        teacherName: quickTest.teacherId?.name || 'Teacher'
      },
      duration: quickTest.duration,
      totalMarks: quickTest.totalMarks,
      questions: quickTest.questions,
      status: quickTest.status,
      canSubmit: quickTest.canSubmit(studentId),
      isActive: true,
      timeRemaining,
      studentStartTime: session.startedAt,
      studentEndTime,
      createdAt: quickTest.createdAt,
      startTime: quickTest.startTime,
      endTime: quickTest.endTime
    };

    res.json({
      success: true,
      data: testData
    });

  } catch (error) {
    console.error('Get quick test for student error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick test',
      error: error.message
    });
  }
});

// Submit quick test answers
router.post('/:testId/submit', async (req, res) => {
  try {
    const { testId } = req.params;
    const { answers, startTime } = req.body;
    const studentId = req.user._id;

    if (!answers || !Array.isArray(answers)) {
      return res.status(400).json({
        success: false,
        message: 'Answers are required and must be an array'
      });
    }

    const quickTest = await QuickTest.findOne({
      _id: testId,
      studentId: getValidObjectId(studentId)
    }).populate('teacherId', 'name');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Check if test can be submitted
    if (!quickTest.canSubmit(studentId)) {
      return res.status(400).json({
        success: false,
        message: 'Cannot submit this test'
      });
    }

    // Validate answers count
    if (answers.length !== quickTest.questions.length) {
      return res.status(400).json({
        success: false,
        message: 'All questions must be answered'
      });
    }

    // Calculate score
    const scoreResult = quickTest.calculateScore(answers);

    // Get student's session to calculate accurate duration
    const session = quickTest.studentSessions.find(s =>
      s.studentId.toString() === studentId.toString()
    );

    const submittedAt = new Date();
    let duration = 0;
    let studentStartTime = null;

    if (session) {
      // Use the session start time (when student clicked "Take Test")
      studentStartTime = session.startedAt;
      duration = Math.floor((submittedAt - session.startedAt) / 1000);
    } else if (startTime) {
      // Fallback to startTime from request body
      studentStartTime = new Date(startTime);
      duration = Math.floor((submittedAt - studentStartTime) / 1000);
    } else {
      // Last resort - use test global start time (shouldn't happen now)
      studentStartTime = quickTest.startTime || submittedAt;
      duration = Math.floor((submittedAt - studentStartTime) / 1000);
    }

    console.log(`📝 Student ${studentId} submitting test ${testId}`);
    console.log(`⏱️ Started at: ${studentStartTime}`);
    console.log(`⏱️ Submitted at: ${submittedAt}`);
    console.log(`⏱️ Duration: ${duration} seconds`);

    // Prepare submission data
    const submissionAnswers = answers.map((answer, index) => {
      const question = quickTest.questions[index];
      const isCorrect = answer.selectedOption === question.correctAnswer;
      const marksObtained = isCorrect ? question.marks : 0;

      return {
        questionIndex: index,
        selectedOption: answer.selectedOption,
        isCorrect,
        marksObtained
      };
    });

    // Add submission to test
    quickTest.submissions.push({
      studentId: getValidObjectId(studentId),
      studentStartTime,
      answers: submissionAnswers,
      totalMarks: scoreResult.obtainedMarks,
      percentage: scoreResult.percentage,
      submittedAt,
      duration
    });

    // Auto-complete the test when student submits
    // Since this is a one-on-one test (teacher to single student),
    // when the student submits, the test should be marked as completed
    if (quickTest.status === 'active') {
      quickTest.status = 'completed';
    }

    await quickTest.save();

    // Emit socket event to notify teacher that test is completed
    const io = req.app.get('io');
    if (io) {
      // Notify teacher that the test has been completed by student
      io.to(`teacher-${quickTest.teacherId}`).emit('quick-test-completed', {
        testId: quickTest._id,
        studentId: studentId,
        title: quickTest.title,
        percentage: scoreResult.percentage,
        totalMarks: scoreResult.obtainedMarks,
        submittedAt
      });

      // Send notification to teacher about test submission
      try {
        const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');

        const notificationData = NotificationTemplates.quicktestGraded(
          quickTest.title,
          `${scoreResult.percentage}%`
        );

        // Extract teacher ID (in case it's populated)
        const teacherId = quickTest.teacherId?._id || quickTest.teacherId;
        console.log('📢 Creating QuickTest submission notification for teacher:', teacherId);

        await createNotification({
          userId: teacherId,
          userType: 'teacher',
          type: 'quicktest_submitted',
          title: 'Quick Test Submitted',
          message: `Student has submitted "${quickTest.title}". Score: ${scoreResult.obtainedMarks}/${scoreResult.totalMarks} (${scoreResult.percentage}%)`,
          relatedId: quickTest._id,
          relatedModel: 'QuickTest',
          priority: 'medium',
          actionUrl: '/teacher/quick-test',
          io
        });
        console.log('✅ QuickTest submission notification created successfully');
      } catch (notifError) {
        console.error('❌ Failed to send QuickTest submission notification:', notifError);
      }
    }

    const submissionData = {
      testId,
      studentId,
      submittedAt,
      answers: submissionAnswers,
      totalQuestions: quickTest.questions.length,
      totalMarks: scoreResult.totalMarks,
      marksObtained: scoreResult.obtainedMarks,
      percentage: scoreResult.percentage,
      duration,
      status: 'completed'
    };

    res.json({
      success: true,
      message: 'Test submitted successfully!',
      data: submissionData
    });

  } catch (error) {
    console.error('Submit quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to submit test',
      error: error.message
    });
  }
});

// Get student's submission for a test
router.get('/:testId/submission', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findById(testId)
      .populate('teacherId', 'teacherName')
      .populate('classId', 'className subject');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Find student's submission
    const submission = quickTest.submissions.find(
      sub => sub.studentId.toString() === req.user._id.toString()
    );

    if (!submission) {
      return res.status(404).json({
        success: false,
        message: 'Submission not found'
      });
    }

    // Prepare detailed results with correct answers (only after test completion)
    let detailedAnswers = null;
    if (quickTest.status === 'completed') {
      detailedAnswers = submission.answers.map((answer, index) => {
        const question = quickTest.questions[index];
        return {
          questionIndex: index,
          question: question.question,
          options: question.options,
          selectedOption: answer.selectedOption,
          correctOption: question.correctAnswer,
          isCorrect: answer.isCorrect,
          marksObtained: answer.marksObtained,
          totalMarks: question.marks
        };
      });
    }

    res.json({
      success: true,
      data: {
        test: {
          _id: quickTest._id,
          title: quickTest.title,
          subject: quickTest.subject,
          teacherName: quickTest.teacherId.teacherName,
          className: quickTest.classId.className,
          totalMarks: quickTest.totalMarks,
          status: quickTest.status
        },
        submission: {
          totalMarks: submission.totalMarks,
          percentage: submission.percentage,
          submittedAt: submission.submittedAt,
          duration: submission.duration,
          answers: detailedAnswers
        }
      }
    });

  } catch (error) {
    console.error('Get student submission error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch submission',
      error: error.message
    });
  }
});

// Get student's quick test history
router.get('/history/all', async (req, res) => {
  try {
    const { limit = 50, page = 1 } = req.query;
    const studentId = req.user._id;

    const skip = (page - 1) * limit;

    // Find tests where student has submitted
    const quickTests = await QuickTest.find({
      studentId: getValidObjectId(studentId),
      'submissions.studentId': getValidObjectId(studentId)
    })
    .populate('teacherId', 'name')
    .sort({ createdAt: -1 })
    .limit(parseInt(limit))
    .skip(skip);

    const total = await QuickTest.countDocuments({
      studentId: getValidObjectId(studentId),
      'submissions.studentId': getValidObjectId(studentId)
    });

    // Format test history with questions and answers for review
    const testHistory = quickTests.map(test => {
      const submission = test.submissions.find(sub =>
        sub.studentId.toString() === studentId.toString()
      );

      return {
        testId: test._id,
        _id: test._id,
        title: test.title,
        subject: test.subject,
        topic: test.topic,
        teacherName: test.teacherId?.name || 'Teacher',
        totalMarks: test.totalMarks,
        status: test.status,
        startTime: test.startTime,
        endTime: test.endTime,
        createdAt: test.createdAt,
        questions: test.questions, // Include questions for review
        submission: {
          totalMarks: submission.totalMarks,
          percentage: submission.percentage,
          submittedAt: submission.submittedAt,
          duration: submission.duration,
          answers: submission.answers // Include answers for review
        }
      };
    });

    res.json({
      success: true,
      data: testHistory,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get student history error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch test history',
      error: error.message
    });
  }
});

export default router;