import express from 'express';
import mongoose from 'mongoose';
import Homework from '../../models/teacher/Homework.js';
import multer from 'multer';
import fs from 'fs';
import crypto from 'crypto';
import path from 'path';
import { fileURLToPath } from 'url';
import { createNotification } from '../../utils/notificationHelper.js';
import moment from 'moment';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const router = express.Router();

// Helper function to convert development IDs to valid ObjectIds
const getValidObjectId = (id) => {
  if (mongoose.Types.ObjectId.isValid(id)) {
    return id;
  } else if (typeof id === 'string' && id.startsWith('dev-')) {
    const hash = crypto.createHash('md5').update(id).digest('hex');
    const objectIdHex = hash.substring(0, 24);
    return new mongoose.Types.ObjectId(objectIdHex);
  }
  return id;
};

// Configure multer for file uploads

const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadPath = path.join(__dirname, '../../uploads/homework/submissions');
    if (!fs.existsSync(uploadPath)) {
      fs.mkdirSync(uploadPath, { recursive: true });
    }
    cb(null, uploadPath);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 10 * 1024 * 1024 // 10MB limit
  },
  fileFilter: function (req, file, cb) {
    const allowedTypes = /jpeg|jpg|png|gif|pdf|doc|docx|txt|zip|rar/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Only images, documents, and archive files are allowed'));
    }
  }
});

// Get all homework for student with filters
router.get('/', async (req, res) => {
  try {
    const { status = 'all', limit = 50, page = 1 } = req.query;
    const studentId = req.user._id;

    console.log('Fetching homework for student:', {
      studentId: studentId.toString(),
      status
    });

    const skip = (page - 1) * limit;

    // Build query directly instead of using findByStudent
    let query = { studentId };
    if (status && status !== 'all') {
      if (status === 'pending') {
        query.status = 'assigned';
        query.dueDate = { $gte: new Date() };
      } else if (status === 'overdue') {
        query.$or = [
          { status: 'overdue' },
          { status: 'assigned', dueDate: { $lt: new Date() } }
        ];
      } else {
        query.status = status;
      }
    }

    console.log('Homework query:', JSON.stringify(query));

    const homework = await Homework.find(query)
      .populate('teacherId', 'name')
      .sort({ createdAt: -1 })
      .limit(parseInt(limit))
      .skip(skip);

    console.log('Found homework count:', homework.length);

    const total = await Homework.countDocuments({ studentId });

    // Format homework for frontend
    const formattedHomework = homework.map(hw => {
      const now = new Date();
      const isActive = hw.status === 'assigned' && hw.dueDate >= now;
      const isOverdue = hw.status === 'assigned' && hw.dueDate < now;

      let timeRemaining = null;
      if (isActive && hw.dueDate) {
        timeRemaining = Math.max(0, Math.floor((hw.dueDate - now) / 1000));
      }

      // Safely call methods
      const canSubmit = typeof hw.canSubmit === 'function' ? hw.canSubmit() : (hw.status === 'assigned' || hw.status === 'overdue');
      const hasSubmitted = typeof hw.isSubmitted === 'function' ? hw.isSubmitted() : (hw.submission && hw.submission.submittedAt);
      const isGraded = typeof hw.isGraded === 'function' ? hw.isGraded() : (hw.submission && hw.submission.grade !== undefined && hw.submission.grade !== null);

      return {
        _id: hw._id,
        title: hw.title,
        description: hw.description,
        subject: hw.subject,
        topic: hw.topic,
        teacherId: {
          _id: hw.teacherId?._id || hw.teacherId,
          teacherName: hw.teacherId?.name || 'Teacher'
        },
        instructions: hw.instructions,
        dueDate: hw.dueDate,
        points: hw.points,
        status: isOverdue ? 'overdue' : hw.status,
        attachment: hw.attachment,
        submission: hw.submission,
        createdAt: hw.createdAt,
        isActive,
        isOverdue,
        canSubmit,
        hasSubmitted,
        isGraded,
        timeRemaining,
        materials: hw.materials || []
      };
    });

    res.json({
      success: true,
      data: formattedHomework,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get student homework error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework',
      error: error.message
    });
  }
});

// Get specific homework by ID for student
router.get('/:homeworkId', async (req, res) => {
  try {
    const { homeworkId } = req.params;
    const studentId = getValidObjectId(req.user._id);

    const homework = await Homework.findOne({
      _id: homeworkId,
      studentId
    }).populate('teacherId', 'name');

    if (!homework) {
      return res.status(404).json({
        success: false,
        message: 'Homework not found'
      });
    }

    const now = new Date();
    const isActive = homework.status === 'assigned' && homework.dueDate >= now;
    const isOverdue = homework.status === 'assigned' && homework.dueDate < now;

    let timeRemaining = null;
    if (isActive && homework.dueDate) {
      timeRemaining = Math.max(0, Math.floor((homework.dueDate - now) / 1000));
    }

    const formattedHomework = {
      _id: homework._id,
      title: homework.title,
      description: homework.description,
      subject: homework.subject,
      teacherId: {
        _id: homework.teacherId?._id || homework.teacherId,
        teacherName: homework.teacherId?.name || 'Teacher'
      },
      instructions: homework.instructions,
      dueDate: homework.dueDate,
      points: homework.points,
      status: isOverdue ? 'overdue' : homework.status,
      attachment: homework.attachment,
      submission: homework.submission,
      createdAt: homework.createdAt,
      isActive,
      isOverdue,
      canSubmit: homework.canSubmit(),
      hasSubmitted: homework.isSubmitted(),
      isGraded: homework.isGraded(),
      timeRemaining
    };

    res.json({
      success: true,
      data: formattedHomework
    });

  } catch (error) {
    console.error('Get homework by ID error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework',
      error: error.message
    });
  }
});

// Submit homework
router.post('/:homeworkId/submit', upload.single('attachment'), async (req, res) => {
  try {
    const { homeworkId } = req.params;
    const { content, notes } = req.body;
    const studentId = getValidObjectId(req.user._id);

    if (!content || !content.trim()) {
      return res.status(400).json({
        success: false,
        message: 'Homework content is required'
      });
    }

    const homework = await Homework.findOne({
      _id: homeworkId,
      studentId
    }).populate('teacherId', 'name');

    if (!homework) {
      return res.status(404).json({
        success: false,
        message: 'Homework not found'
      });
    }

    if (homework.isSubmitted()) {
      return res.status(400).json({
        success: false,
        message: 'Homework has already been submitted'
      });
    }

    if (!homework.canSubmit()) {
      return res.status(400).json({
        success: false,
        message: 'This homework cannot be submitted'
      });
    }

    const submissionData = {
      content: content.trim(),
      notes: notes ? notes.trim() : undefined
    };

    // Handle file attachment
    if (req.file) {
      submissionData.attachment = {
        filename: req.file.filename,
        originalName: req.file.originalname,
        path: req.file.path,
        size: req.file.size,
        mimetype: req.file.mimetype
      };
    }

    await homework.submitHomework(submissionData);

    // Create notification for teacher
    const io = req.app.get('io');
    if (homework.teacherId && io) {
      await createNotification({
        userId: homework.teacherId,
        userType: 'teacher',
        type: 'homework_submitted',
        title: 'Homework Submitted',
        message: `A student has submitted homework for ${homework.subject}: ${homework.title}`,
        relatedId: homework._id,
        relatedModel: 'Homework',
        priority: 'medium',
        actionUrl: '/teacher/homework',
        io
      });
    }

    res.json({
      success: true,
      message: 'Homework submitted successfully!',
      data: {
        homeworkId: homework._id,
        title: homework.title,
        submittedAt: homework.submission.submittedAt,
        status: homework.status,
        content: homework.submission.content,
        attachment: homework.submission.attachment,
        notes: homework.submission.notes
      }
    });

  } catch (error) {
    console.error('Submit homework error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to submit homework',
      error: error.message
    });
  }
});

// Get student's homework history (submitted and graded)
router.get('/history/all', async (req, res) => {
  try {
    const { limit = 50, page = 1 } = req.query;
    const studentId = getValidObjectId(req.user._id);

    const skip = (page - 1) * limit;

    // Find homework where student has submitted
    const homework = await Homework.find({
      studentId,
      'submission.submittedAt': { $exists: true }
    })
    .populate('teacherId', 'name')
    .sort({ 'submission.submittedAt': -1 })
    .limit(parseInt(limit))
    .skip(skip);

    const total = await Homework.countDocuments({
      studentId,
      'submission.submittedAt': { $exists: true }
    });

    // Format homework history
    const homeworkHistory = homework.map(hw => ({
      homeworkId: hw._id,
      title: hw.title,
      subject: hw.subject,
      teacherName: hw.teacherId?.name || 'Teacher',
      points: hw.points,
      status: hw.status,
      dueDate: hw.dueDate,
      createdAt: hw.createdAt,
      submission: {
        submittedAt: hw.submission.submittedAt,
        grade: hw.submission.grade,
        feedback: hw.submission.feedback,
        corrections: hw.submission.corrections,
        gradedAt: hw.submission.gradedAt,
        percentage: hw.submission.grade && hw.points ?
          Math.round((hw.submission.grade / hw.points) * 100) : null
      }
    }));

    res.json({
      success: true,
      data: homeworkHistory,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get homework history error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework history',
      error: error.message
    });
  }
});

// Get homework submission details
router.get('/:homeworkId/submission', async (req, res) => {
  try {
    const { homeworkId } = req.params;
    const studentId = getValidObjectId(req.user._id);

    const homework = await Homework.findOne({
      _id: homeworkId,
      studentId
    }).populate('teacherId', 'name');

    if (!homework) {
      return res.status(404).json({
        success: false,
        message: 'Homework not found'
      });
    }

    if (!homework.isSubmitted()) {
      return res.status(404).json({
        success: false,
        message: 'No submission found for this homework'
      });
    }

    const submissionData = {
      homework: {
        _id: homework._id,
        title: homework.title,
        subject: homework.subject,
        teacherName: homework.teacherId?.name || 'Teacher',
        points: homework.points,
        dueDate: homework.dueDate,
        status: homework.status
      },
      submission: {
        content: homework.submission.content,
        attachment: homework.submission.attachment,
        notes: homework.submission.notes,
        submittedAt: homework.submission.submittedAt,
        grade: homework.submission.grade,
        feedback: homework.submission.feedback,
        corrections: homework.submission.corrections,
        gradedAt: homework.submission.gradedAt,
        percentage: homework.submission.grade && homework.points ?
          Math.round((homework.submission.grade / homework.points) * 100) : null
      }
    };

    res.json({
      success: true,
      data: submissionData
    });

  } catch (error) {
    console.error('Get homework submission error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework submission',
      error: error.message
    });
  }
});

// Get homework statistics for student dashboard
router.get('/stats/overview', async (req, res) => {
  try {
    const studentId = getValidObjectId(req.user._id);
    const now = new Date();

    const stats = await Promise.all([
      // Total homework count
      Homework.countDocuments({ studentId }),

      // Pending homework (assigned and not overdue)
      Homework.countDocuments({
        studentId,
        status: 'assigned',
        dueDate: { $gte: now }
      }),

      // Submitted homework (waiting for grade)
      Homework.countDocuments({
        studentId,
        status: 'submitted'
      }),

      // Graded homework
      Homework.countDocuments({
        studentId,
        status: 'graded'
      }),

      // Overdue homework
      Homework.countDocuments({
        studentId,
        $or: [
          { status: 'overdue' },
          { status: 'assigned', dueDate: { $lt: now } }
        ]
      })
    ]);

    const [total, pending, submitted, graded, overdue] = stats;

    // Calculate average grade
    const gradedHomework = await Homework.find({
      studentId,
      status: 'graded',
      'submission.grade': { $exists: true }
    }).select('submission.grade points');

    let averageGrade = 0;
    if (gradedHomework.length > 0) {
      const totalPercentage = gradedHomework.reduce((sum, hw) => {
        return sum + ((hw.submission.grade / hw.points) * 100);
      }, 0);
      averageGrade = Math.round(totalPercentage / gradedHomework.length);
    }

    res.json({
      success: true,
      data: {
        total,
        pending,
        submitted,
        graded,
        overdue,
        averageGrade,
        completionRate: total > 0 ? Math.round(((submitted + graded) / total) * 100) : 0
      }
    });

  } catch (error) {
    console.error('Get homework stats error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch homework statistics',
      error: error.message
    });
  }
});

export default router;