import express from 'express';
import moment from 'moment-timezone';
import Class from '../../models/Class.js';
import { studentAuth } from '../../middleware/auth.js';

const router = express.Router();

// Get attendance report for student
router.get('/report', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;
    const { month, startDate, endDate, subject } = req.query;

    // Build query
    const query = {
      studentId: studentId,
      status: 'completed'
    };

    // Date filters
    if (month) {
      // Month format: YYYY-MM
      const startOfMonth = moment(month).startOf('month');
      const endOfMonth = moment(month).endOf('month');
      query.scheduledDate = {
        $gte: startOfMonth.toDate(),
        $lte: endOfMonth.toDate()
      };
    } else if (startDate && endDate) {
      query.scheduledDate = {
        $gte: moment(startDate).startOf('day').toDate(),
        $lte: moment(endDate).endOf('day').toDate()
      };
    } else {
      // Default to current month
      query.scheduledDate = {
        $gte: moment().startOf('month').toDate(),
        $lte: moment().endOf('month').toDate()
      };
    }

    // Subject filter
    if (subject && subject !== 'all') {
      query.subject = subject;
    }

    const classes = await Class.find(query)
      .populate('teacherId', 'name')
      .sort({ scheduledDate: -1 });

    // Format attendance data
    const attendance = classes.map(cls => ({
      _id: cls._id,
      date: moment(cls.scheduledDate).format('YYYY-MM-DD'),
      subject: cls.subject,
      teacherName: cls.teacherId?.name || 'Unknown Teacher',
      scheduledTime: cls.startTime,
      duration: cls.duration,
      status: cls.attendance?.absent ? 'absent' : cls.attendance?.late ? 'late' : cls.attendance?.present ? 'present' : 'unmarked',
      joinTime: cls.attendance?.studentJoinTime || null,
      notes: cls.attendance?.notes || ''
    }));

    // Calculate statistics
    const totalClasses = attendance.length;
    const presentClasses = attendance.filter(a => a.status === 'present').length;
    const lateClasses = attendance.filter(a => a.status === 'late').length;
    const absentClasses = attendance.filter(a => a.status === 'absent').length;
    const unmarkedClasses = attendance.filter(a => a.status === 'unmarked').length;

    const attendanceRate = totalClasses > 0 ? (((presentClasses + lateClasses) / totalClasses) * 100).toFixed(1) : '0.0';
    const punctualityRate = totalClasses > 0 ? ((presentClasses / totalClasses) * 100).toFixed(1) : '0.0';

    // Subject-wise stats
    const subjectMap = {};
    attendance.forEach(att => {
      if (!subjectMap[att.subject]) {
        subjectMap[att.subject] = {
          subject: att.subject,
          total: 0,
          present: 0,
          late: 0,
          absent: 0,
          unmarked: 0
        };
      }
      subjectMap[att.subject].total++;
      if (att.status === 'present') subjectMap[att.subject].present++;
      else if (att.status === 'late') subjectMap[att.subject].late++;
      else if (att.status === 'absent') subjectMap[att.subject].absent++;
      else subjectMap[att.subject].unmarked++;
    });

    const subjectStats = Object.values(subjectMap).map(subject => ({
      ...subject,
      rate: subject.total > 0 ? (((subject.present + subject.late) / subject.total) * 100) : 0
    }));

    const stats = {
      totalClasses,
      presentClasses,
      lateClasses,
      absentClasses,
      unmarkedClasses,
      attendanceRate,
      punctualityRate,
      subjectStats
    };

    res.json({
      success: true,
      attendance,
      stats
    });
  } catch (error) {
    console.error('Get student attendance report error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Get attendance statistics for student
router.get('/stats', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;
    const { startDate, endDate } = req.query;

    // Build query
    const query = {
      studentId: studentId,
      status: 'completed'
    };

    // Date range filter
    if (startDate && endDate) {
      query.scheduledDate = {
        $gte: moment(startDate).startOf('day').toDate(),
        $lte: moment(endDate).endOf('day').toDate()
      };
    } else {
      // Default to last 30 days
      query.scheduledDate = {
        $gte: moment().subtract(30, 'days').startOf('day').toDate(),
        $lte: moment().endOf('day').toDate()
      };
    }

    const classes = await Class.find(query)
      .populate('teacherId', 'name')
      .sort({ scheduledDate: -1 });

    // Calculate statistics
    const stats = {
      total: classes.length,
      present: 0,
      late: 0,
      absent: 0,
      unmarked: 0,
      attendanceRate: 0,
      punctualityRate: 0,
      bySubject: {},
      byMonth: {}
    };

    classes.forEach(cls => {
      const subject = cls.subject;
      const month = moment(cls.scheduledDate).format('YYYY-MM');

      // Determine status
      let status = 'unmarked';
      if (cls.attendance?.absent) status = 'absent';
      else if (cls.attendance?.late) status = 'late';
      else if (cls.attendance?.present) status = 'present';

      // Overall stats
      if (status === 'present') stats.present++;
      else if (status === 'late') stats.late++;
      else if (status === 'absent') stats.absent++;
      else stats.unmarked++;

      // Per-subject stats
      if (!stats.bySubject[subject]) {
        stats.bySubject[subject] = {
          subject,
          total: 0,
          present: 0,
          late: 0,
          absent: 0,
          unmarked: 0
        };
      }
      stats.bySubject[subject].total++;
      stats.bySubject[subject][status]++;

      // Per-month stats
      if (!stats.byMonth[month]) {
        stats.byMonth[month] = {
          month,
          total: 0,
          present: 0,
          late: 0,
          absent: 0,
          unmarked: 0
        };
      }
      stats.byMonth[month].total++;
      stats.byMonth[month][status]++;
    });

    // Calculate rates
    if (stats.total > 0) {
      stats.attendanceRate = (((stats.present + stats.late) / stats.total) * 100).toFixed(1);
      stats.punctualityRate = ((stats.present / stats.total) * 100).toFixed(1);
    }

    // Calculate per-subject rates
    Object.keys(stats.bySubject).forEach(subject => {
      const subjectStats = stats.bySubject[subject];
      if (subjectStats.total > 0) {
        subjectStats.attendanceRate = (((subjectStats.present + subjectStats.late) / subjectStats.total) * 100).toFixed(1);
        subjectStats.punctualityRate = ((subjectStats.present / subjectStats.total) * 100).toFixed(1);
      }
    });

    // Calculate per-month rates
    Object.keys(stats.byMonth).forEach(month => {
      const monthStats = stats.byMonth[month];
      if (monthStats.total > 0) {
        monthStats.attendanceRate = (((monthStats.present + monthStats.late) / monthStats.total) * 100).toFixed(1);
        monthStats.punctualityRate = ((monthStats.present / monthStats.total) * 100).toFixed(1);
      }
    });

    res.json({
      success: true,
      stats
    });
  } catch (error) {
    console.error('Get student attendance stats error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Get attendance summary for dashboard
router.get('/summary', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;

    // Get current month attendance
    const currentMonthStart = moment().startOf('month');
    const currentMonthEnd = moment().endOf('month');

    const currentMonthClasses = await Class.find({
      studentId: studentId,
      status: 'completed',
      scheduledDate: {
        $gte: currentMonthStart.toDate(),
        $lte: currentMonthEnd.toDate()
      }
    });

    const present = currentMonthClasses.filter(c => c.attendance?.present && !c.attendance?.late).length;
    const late = currentMonthClasses.filter(c => c.attendance?.late).length;
    const absent = currentMonthClasses.filter(c => c.attendance?.absent).length;
    const total = currentMonthClasses.length;

    const attendanceRate = total > 0 ? (((present + late) / total) * 100).toFixed(1) : '0.0';

    res.json({
      success: true,
      summary: {
        month: moment().format('MMMM YYYY'),
        total,
        present,
        late,
        absent,
        attendanceRate
      }
    });
  } catch (error) {
    console.error('Get student attendance summary error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// Download attendance report as CSV
router.get('/download', studentAuth, async (req, res) => {
  try {
    const studentId = req.user._id;
    const { month, startDate, endDate, subject, format = 'csv' } = req.query;

    // Build query
    const query = {
      studentId: studentId,
      status: 'completed'
    };

    // Date filters
    if (month) {
      const startOfMonth = moment(month).startOf('month');
      const endOfMonth = moment(month).endOf('month');
      query.scheduledDate = {
        $gte: startOfMonth.toDate(),
        $lte: endOfMonth.toDate()
      };
    } else if (startDate && endDate) {
      query.scheduledDate = {
        $gte: moment(startDate).startOf('day').toDate(),
        $lte: moment(endDate).endOf('day').toDate()
      };
    } else {
      query.scheduledDate = {
        $gte: moment().startOf('month').toDate(),
        $lte: moment().endOf('month').toDate()
      };
    }

    if (subject && subject !== 'all') {
      query.subject = subject;
    }

    const classes = await Class.find(query)
      .populate('studentId', 'studentName email class')
      .populate('teacherId', 'name')
      .sort({ scheduledDate: 1 });

    if (format === 'csv') {
      // Generate CSV
      const csvRows = [];

      // Header
      csvRows.push([
        'Date',
        'Day',
        'Subject',
        'Teacher',
        'Scheduled Time',
        'Duration (min)',
        'Status',
        'Join Time',
        'Notes'
      ].join(','));

      // Data rows
      classes.forEach(cls => {
        const date = moment(cls.scheduledDate);
        const status = cls.attendance?.absent ? 'Absent' :
                      cls.attendance?.late ? 'Late' :
                      cls.attendance?.present ? 'Present' : 'Unmarked';

        const joinTime = cls.attendance?.studentJoinTime ?
                        moment(cls.attendance.studentJoinTime).format('HH:mm') : '-';

        const notes = (cls.attendance?.notes || '').replace(/,/g, ';').replace(/\n/g, ' ');

        csvRows.push([
          date.format('YYYY-MM-DD'),
          date.format('dddd'),
          cls.subject,
          cls.teacherId?.name || 'Unknown',
          cls.startTime,
          cls.duration,
          status,
          joinTime,
          `"${notes}"`
        ].join(','));
      });

      // Calculate statistics
      const totalClasses = classes.length;
      const presentClasses = classes.filter(c => c.attendance?.present && !c.attendance?.late).length;
      const lateClasses = classes.filter(c => c.attendance?.late).length;
      const absentClasses = classes.filter(c => c.attendance?.absent).length;
      const attendanceRate = totalClasses > 0 ? (((presentClasses + lateClasses) / totalClasses) * 100).toFixed(1) : '0.0';
      const punctualityRate = totalClasses > 0 ? ((presentClasses / totalClasses) * 100).toFixed(1) : '0.0';

      // Add summary
      csvRows.push('');
      csvRows.push('SUMMARY');
      csvRows.push(`Total Classes,${totalClasses}`);
      csvRows.push(`Present,${presentClasses}`);
      csvRows.push(`Late,${lateClasses}`);
      csvRows.push(`Absent,${absentClasses}`);
      csvRows.push(`Attendance Rate,${attendanceRate}%`);
      csvRows.push(`Punctuality Rate,${punctualityRate}%`);

      const csv = csvRows.join('\n');

      // Get student info for filename
      const student = classes[0]?.studentId;
      const studentName = student?.studentName?.replace(/\s+/g, '_') || 'Student';
      const monthStr = month ? moment(month).format('YYYY-MM') : moment().format('YYYY-MM');
      const filename = `Attendance_Report_${studentName}_${monthStr}.csv`;

      res.setHeader('Content-Type', 'text/csv');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.send(csv);
    } else {
      res.status(400).json({ message: 'Unsupported format. Use format=csv' });
    }
  } catch (error) {
    console.error('Download attendance report error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

export default router;