import express from 'express';
import mongoose from 'mongoose';
import Exam from '../../models/teacher/Exam.js';
import Question from '../../models/teacher/Question.js';
import Submission from '../../models/student/Submission.js';

const router = express.Router();

// Get all exams (admin overview)
router.get('/', async (req, res) => {
  try {
    const { status, teacherId, page = 1, limit = 10 } = req.query;
    const filter = {};
    
    if (status) filter.status = status;
    if (teacherId) filter.teacherId = teacherId;
    
    const skip = (page - 1) * limit;
    
    const exams = await Exam.find(filter)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit));
    
    const total = await Exam.countDocuments(filter);
    
    res.json({ 
      success: true, 
      exams, 
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get exam statistics
router.get('/stats', async (req, res) => {
  try {
    const totalExams = await Exam.countDocuments();
    const liveExams = await Exam.countDocuments({ status: 'live' });
    const completedExams = await Exam.countDocuments({ status: 'completed' });
    const draftExams = await Exam.countDocuments({ status: 'draft' });
    const readyExams = await Exam.countDocuments({ status: 'ready' });
    
    const totalSubmissions = await Submission.countDocuments();
    const passedSubmissions = await Submission.countDocuments({ passed: true });
    
    // Average scores
    const avgScoreResult = await Submission.aggregate([
      {
        $group: {
          _id: null,
          avgScore: { $avg: '$percentage' },
          avgMarks: { $avg: '$totalScore' }
        }
      }
    ]);
    
    const avgScore = avgScoreResult.length > 0 ? avgScoreResult[0].avgScore : 0;
    const avgMarks = avgScoreResult.length > 0 ? avgScoreResult[0].avgMarks : 0;
    
    // Recent activity
    const recentExams = await Exam.find()
      .sort({ createdAt: -1 })
      .limit(5)
      .select('title status createdAt teacherId');
    
    const recentSubmissions = await Submission.find()
      .sort({ submittedAt: -1 })
      .limit(5)
      .populate({
        path: 'examId',
        model: 'Exam',
        select: 'title'
      })
      .select('studentId totalScore percentage submittedAt');
    
    // Monthly exam trends
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
    
    const monthlyTrends = await Exam.aggregate([
      {
        $match: {
          createdAt: { $gte: thirtyDaysAgo }
        }
      },
      {
        $group: {
          _id: {
            year: { $year: '$createdAt' },
            month: { $month: '$createdAt' },
            day: { $dayOfMonth: '$createdAt' }
          },
          count: { $sum: 1 }
        }
      },
      {
        $sort: { '_id.year': 1, '_id.month': 1, '_id.day': 1 }
      }
    ]);
    
    res.json({
      success: true,
      stats: {
        totalExams,
        liveExams,
        completedExams,
        draftExams,
        readyExams,
        totalSubmissions,
        passedSubmissions,
        passRate: totalSubmissions > 0 ? (passedSubmissions / totalSubmissions) * 100 : 0,
        avgScore: avgScore || 0,
        avgMarks: avgMarks || 0,
        recentExams,
        recentSubmissions,
        monthlyTrends
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get detailed exam analytics
router.get('/:id/analytics', async (req, res) => {
  try {
    const examId = req.params.id;
    
    const exam = await Exam.findById(examId);
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    const submissions = await Submission.find({ examId });
    const questions = await Question.find({ examId });
    
    // Basic stats
    const totalSubmissions = submissions.length;
    const passedCount = submissions.filter(s => s.passed).length;
    const failedCount = totalSubmissions - passedCount;
    
    // Score distribution
    const scoreRanges = {
      '0-20': 0,
      '21-40': 0,
      '41-60': 0,
      '61-80': 0,
      '81-100': 0
    };
    
    submissions.forEach(submission => {
      const percentage = submission.percentage;
      if (percentage <= 20) scoreRanges['0-20']++;
      else if (percentage <= 40) scoreRanges['21-40']++;
      else if (percentage <= 60) scoreRanges['41-60']++;
      else if (percentage <= 80) scoreRanges['61-80']++;
      else scoreRanges['81-100']++;
    });
    
    // Question-wise analysis
    const questionAnalysis = questions.map(question => {
      const questionSubmissions = submissions.map(s => 
        s.answers.find(a => a.questionId.toString() === question._id.toString())
      ).filter(Boolean);
      
      const correctAnswers = questionSubmissions.filter(a => a.isCorrect).length;
      const totalAnswers = questionSubmissions.length;
      
      return {
        questionId: question._id,
        questionText: question.questionText,
        correctAnswers,
        totalAnswers,
        accuracy: totalAnswers > 0 ? (correctAnswers / totalAnswers) * 100 : 0,
        marks: question.marks
      };
    });
    
    // Time analysis
    const avgTimeTaken = submissions.reduce((sum, s) => sum + (s.timeTaken || 0), 0) / totalSubmissions;
    
    res.json({
      success: true,
      analytics: {
        exam: {
          id: exam._id,
          title: exam.title,
          totalMarks: exam.totalMarks,
          passingMarks: exam.passingMarks,
          duration: exam.duration,
          status: exam.status
        },
        participation: {
          totalSubmissions,
          passedCount,
          failedCount,
          passRate: totalSubmissions > 0 ? (passedCount / totalSubmissions) * 100 : 0
        },
        scoreDistribution: scoreRanges,
        questionAnalysis,
        timing: {
          avgTimeTaken: Math.round(avgTimeTaken / 60), // Convert to minutes
          examDuration: exam.duration
        },
        topPerformers: submissions
          .sort((a, b) => b.totalScore - a.totalScore)
          .slice(0, 10)
          .map(s => ({
            studentId: s.studentId,
            score: s.totalScore,
            percentage: s.percentage,
            timeTaken: Math.round((s.timeTaken || 0) / 60)
          }))
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Force stop any live exam (emergency control)
router.put('/:id/force-stop', async (req, res) => {
  try {
    const exam = await Exam.findById(req.params.id);
    
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    if (exam.status !== 'live') {
      return res.status(400).json({ success: false, error: 'Exam is not currently live' });
    }
    
    exam.status = 'completed';
    exam.endedAt = new Date();
    await exam.save();
    
    // Emit socket event for real-time updates
    req.app.get('io').emit('examStatusChanged', { 
      examId: exam._id, 
      status: 'completed', 
      exam,
      forceStopped: true 
    });
    
    res.json({ success: true, message: 'Exam force-stopped successfully', exam });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Delete exam (admin privilege)
router.delete('/:id', async (req, res) => {
  try {
    const exam = await Exam.findByIdAndDelete(req.params.id);
    if (!exam) {
      return res.status(404).json({ success: false, error: 'Exam not found' });
    }
    
    // Clean up related data
    await Question.deleteMany({ examId: exam._id });
    await Submission.deleteMany({ examId: exam._id });
    
    res.json({ success: true, message: 'Exam and all related data deleted successfully' });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

export default router;