const express = require('express');
const bcrypt = require('bcryptjs');
const multer = require('multer');
const path = require('path');
const mongoose = require('mongoose');
const User = require('../../models/admin/User');
const Student = require('../../models/student/Student');
const Teacher = require('../../models/teacher/Teacher');
const TrialRequest = require('../../models/admin/TrialRequest');
const TeacherClass = require('../../models/teacher/Class');
const Class = require('../../models/Class');
const { adminAuth } = require('../../middleware/auth');
const { generatePassword, generateUsername } = require('../../utils/passwordGenerator');
const { getSyllabusOptions } = require('../../utils/syllabusHelper');

const router = express.Router();

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/');
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + Math.round(Math.random() * 1E9) + path.extname(file.originalname));
  }
});

const upload = multer({ storage });

router.post('/students', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const {
      studentName,
      parentName,
      email,
      countryCode,
      phoneNumber,
      hourlyRate,
      currency,
      country,
      syllabus,
      class: studentClass,
      subjects,
      googleMeetLink,
      trialRequestId
    } = req.body;

    if (!getSyllabusOptions(country).includes(syllabus)) {
      return res.status(400).json({ message: 'Invalid syllabus for selected country' });
    }

    const username = generateUsername(studentName);
    const password = generatePassword(studentName);
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = new User({
      username,
      password: hashedPassword,
      userType: 'student'
    });

    await user.save();

    // Handle profile picture upload for new student
    let profilePictureData = null;
    let profilePictureType = null;
    let profilePictureFilename = null;

    if (req.file) {
      const fs = require('fs');
      const path = require('path');
      
      try {
        // Read the uploaded file and convert to base64
        const filePath = path.join(__dirname, '../../uploads', req.file.filename);
        const fileBuffer = fs.readFileSync(filePath);
        profilePictureData = fileBuffer.toString('base64');
        profilePictureType = req.file.mimetype;
        profilePictureFilename = req.file.filename;
      } catch (fileError) {
        console.error('Error processing uploaded file:', fileError);
        // Fall back to filename only
        profilePictureData = req.file.filename;
      }
    }

    const student = new Student({
      userId: user._id,
      studentName,
      parentName,
      email,
      phoneNumber: { countryCode, number: phoneNumber },
      profilePicture: profilePictureData,
      profilePictureType: profilePictureType,
      profilePictureFilename: profilePictureFilename,
      hourlyRate: parseFloat(hourlyRate),
      currency,
      country,
      syllabus,
      class: studentClass,
      subjects: JSON.parse(subjects),
      googleMeetLink,
      originalPassword: password // Store original password for admin access
    });

    await student.save();

    // If this student was created from a trial request, mark it as processed
    if (trialRequestId) {
      try {
        console.log('Updating trial request status for ID:', trialRequestId);
        await TrialRequest.findByIdAndUpdate(
          trialRequestId, 
          { 
            status: 'processed',
            processedAt: new Date(),
            studentId: student._id 
          }
        );
        console.log('Trial request marked as processed successfully');
      } catch (trialError) {
        console.error('Failed to update trial request:', trialError);
        // Don't fail the student creation if trial request update fails
      }
    }

    res.status(201).json({
      message: 'Student created successfully',
      credentials: { username, password },
      student: await Student.findById(student._id).populate('userId', 'username')
    });
  } catch (error) {
    console.error('Create student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/teachers', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const {
      name,
      phoneNumber,
      email,
      subjects,
      salaryType,
      salaryAmount,
      accountHolderName,
      aadhaarNumber,
      panNumber,
      accountNumber,
      bankName,
      ifscCode,
      branchName,
      startTime,
      endTime,
      availableDays
    } = req.body;

    const username = generateUsername(name);
    const password = generatePassword(name);
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = new User({
      username,
      password: hashedPassword,
      userType: 'teacher'
    });

    await user.save();

    const teacher = new Teacher({
      userId: user._id,
      name,
      phoneNumber,
      email,
      profilePicture: req.file ? req.file.filename : null,
      subjects: JSON.parse(subjects),
      salaryType,
      salary: {
        amount: parseFloat(salaryAmount),
        currency: 'INR'
      },
      bankingDetails: {
        accountHolderName,
        accountNumber,
        bankName,
        ifscCode,
        branchName,
        aadhaarNumber,
        panNumber
      },
      availableTime: {
        startTime,
        endTime
      },
      availableDays: JSON.parse(availableDays),
      originalPassword: password // Store original password for admin access
    });

    await teacher.save();

    res.status(201).json({
      message: 'Teacher created successfully',
      credentials: { username, password },
      teacher: await Teacher.findById(teacher._id).populate('userId', 'username')
    });
  } catch (error) {
    console.error('Create teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/trial-requests', adminAuth, async (req, res) => {
  try {
    console.log('📝 Received trial request data:', JSON.stringify(req.body, null, 2));
    console.log('👤 User info:', req.user ? req.user._id : 'No user found');
    
    const trialRequest = new TrialRequest({
      ...req.body,
      source: 'manual'
    });

    console.log('💾 About to save trial request with data:', JSON.stringify(trialRequest.toObject(), null, 2));
    await trialRequest.save();
    console.log('✅ Trial request saved successfully');
    res.status(201).json({ message: 'Trial request created successfully', trialRequest });
  } catch (error) {
    console.error('Create trial request error:', error);
    console.error('Error details:', error.errors);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

router.get('/students', adminAuth, async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 10, 
      search = '', 
      isDeleted = 'false',
      country,
      syllabus,
      class: studentClass,
      currency,
      isActive,
      sortBy = 'createdAt',
      sortOrder = 'desc'
    } = req.query;
    
    const query = {
      isDeleted: isDeleted === 'true',
      ...(search && {
        $or: [
          { studentName: { $regex: search, $options: 'i' } },
          { parentName: { $regex: search, $options: 'i' } },
          { email: { $regex: search, $options: 'i' } }
        ]
      }),
      ...(country && { country }),
      ...(syllabus && { syllabus }),
      ...(studentClass && { class: studentClass }),
      ...(currency && { currency }),
      ...(isActive !== undefined && { isActive: isActive === 'true' })
    };

    // Build sort object
    const sortObject = {};
    const validSortFields = ['studentName', 'parentName', 'email', 'country', 'class', 'syllabus', 'hourlyRate', 'createdAt', 'isActive'];
    const sortField = validSortFields.includes(sortBy) ? sortBy : 'createdAt';
    const sortDirection = sortOrder === 'asc' ? 1 : -1;
    sortObject[sortField] = sortDirection;

    const students = await Student.find(query)
      .populate('userId', 'username isActive')
      .populate('assignedTeachers.teacherId', 'name')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort(sortObject);

    const total = await Student.countDocuments(query);

    res.json({
      students,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total,
      sortBy: sortField,
      sortOrder
    });
  } catch (error) {
    console.error('Get students error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get individual student details with credentials
router.get('/students/:id', adminAuth, async (req, res) => {
  try {
    const student = await Student.findById(req.params.id)
      .populate('userId', 'username')
      .populate('assignedTeachers.teacherId', 'name');
    
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    // Get the user's credentials
    const user = await User.findById(student.userId._id);
    
    let password = student.originalPassword;
    
    // If originalPassword doesn't exist, generate a new one and save it
    if (!password) {
      const { generatePassword } = require('../../utils/passwordGenerator');
      password = generatePassword(student.studentName);
      
      // Update the student record with the new password
      await Student.findByIdAndUpdate(req.params.id, { originalPassword: password });
      
      // Also update the user's hashed password
      const bcrypt = require('bcryptjs');
      const hashedPassword = await bcrypt.hash(password, 10);
      await User.findByIdAndUpdate(student.userId._id, { password: hashedPassword });
    }
    
    res.json({
      student,
      credentials: {
        username: user.username,
        password: password || 'Unable to generate password'
      }
    });
  } catch (error) {
    console.error('Get student details error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/teachers', adminAuth, async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 10, 
      search = '', 
      isDeleted = 'false',
      salaryType,
      isActive,
      sortBy = 'createdAt',
      sortOrder = 'desc'
    } = req.query;
    
    const query = {
      isDeleted: isDeleted === 'true',
      ...(search && {
        $or: [
          { name: { $regex: search, $options: 'i' } },
          { email: { $regex: search, $options: 'i' } }
        ]
      }),
      ...(salaryType && { salaryType }),
      ...(isActive !== undefined && { isActive: isActive === 'true' })
    };

    // Build sort object  
    const sortObject = {};
    const validSortFields = ['name', 'email', 'salaryType', 'salary.amount', 'rating', 'createdAt', 'isActive'];
    const sortField = validSortFields.includes(sortBy) ? sortBy : 'createdAt';
    const sortDirection = sortOrder === 'asc' ? 1 : -1;
    sortObject[sortField] = sortDirection;

    const teachers = await Teacher.find(query)
      .populate('userId', 'username isActive')
      .populate('assignedStudents.studentId', 'studentName')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort(sortObject);

    const total = await Teacher.countDocuments(query);

    res.json({
      teachers,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total,
      sortBy: sortField,
      sortOrder
    });
  } catch (error) {
    console.error('Get teachers error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get individual teacher details with credentials
router.get('/teachers/:id', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findById(req.params.id)
      .populate('userId', 'username')
      .populate('assignedStudents.studentId', 'studentName');
    
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Get the user's credentials
    const user = await User.findById(teacher.userId._id);
    
    let password = teacher.originalPassword;
    
    // If originalPassword doesn't exist, generate a new one and save it
    if (!password) {
      const { generatePassword } = require('../../utils/passwordGenerator');
      password = generatePassword(teacher.name);
      
      // Update the teacher record with the new password
      await Teacher.findByIdAndUpdate(req.params.id, { originalPassword: password });
      
      // Also update the user's hashed password
      const bcrypt = require('bcryptjs');
      const hashedPassword = await bcrypt.hash(password, 10);
      await User.findByIdAndUpdate(teacher.userId._id, { password: hashedPassword });
    }
    
    res.json({
      teacher,
      credentials: {
        username: user.username,
        password: password || 'Unable to generate password'
      }
    });
  } catch (error) {
    console.error('Get teacher details error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/trial-requests', adminAuth, async (req, res) => {
  try {
    const { page = 1, limit = 10, status = 'all' } = req.query;
    
    // Build query - exclude 'processed' requests from 'all' view
    let query = {};
    if (status === 'all') {
      query = { status: { $nin: ['processed'] } }; // Exclude processed requests
    } else {
      query = { status };
    }

    const trialRequests = await TrialRequest.find(query)
      .populate('assignedTeacher', 'name')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ createdAt: -1 });

    const total = await TrialRequest.countDocuments(query);

    res.json({
      trialRequests,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get trial requests error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update trial request
router.put('/trial-requests/:id', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;
    
    const trialRequest = await TrialRequest.findByIdAndUpdate(
      id,
      { ...updates, updatedAt: new Date() },
      { new: true, runValidators: true }
    );

    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    res.json({
      message: 'Trial request updated successfully',
      trialRequest
    });
  } catch (error) {
    console.error('Update trial request error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Update trial request status
router.put('/trial-requests/:id/status', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    const { status } = req.body;

    if (!status) {
      return res.status(400).json({ message: 'Status is required' });
    }

    const validStatuses = ['pending', 'scheduled', 'completed', 'cancelled', 'converted'];
    if (!validStatuses.includes(status)) {
      return res.status(400).json({ message: 'Invalid status' });
    }

    const trialRequest = await TrialRequest.findByIdAndUpdate(
      id,
      { status, updatedAt: new Date() },
      { new: true, runValidators: true }
    );

    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    res.json({
      message: `Trial request status updated to ${status}`,
      trialRequest
    });
  } catch (error) {
    console.error('Update trial request status error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Delete trial request
router.delete('/trial-requests/:id', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;

    const trialRequest = await TrialRequest.findByIdAndDelete(id);

    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    res.json({
      message: 'Trial request deleted successfully'
    });
  } catch (error) {
    console.error('Delete trial request error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Convert trial request to student
router.post('/trial-requests/:id/convert', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    const studentData = req.body;

    // Find the trial request
    const trialRequest = await TrialRequest.findById(id);
    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    // Create user account for the student
    const username = generateUsername(studentData.studentName);
    const password = generatePassword(studentData.studentName);
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = new User({
      username,
      password: hashedPassword,
      userType: 'student'
    });

    await user.save();

    // Create student record
    const student = new Student({
      userId: user._id,
      studentName: studentData.studentName,
      parentName: studentData.parentName,
      email: studentData.email,
      phoneNumber: studentData.phoneNumber,
      hourlyRate: parseFloat(studentData.hourlyRate),
      currency: studentData.currency,
      country: studentData.country,
      syllabus: studentData.syllabus,
      class: studentData.class,
      subjects: studentData.subjects,
      originalPassword: password,
      trialRequestId: id
    });

    await student.save();

    // Update trial request status to converted
    await TrialRequest.findByIdAndUpdate(id, { 
      status: 'converted',
      convertedAt: new Date(),
      convertedToStudentId: student._id
    });

    res.status(201).json({
      message: 'Trial request converted to student successfully',
      student: await Student.findById(student._id).populate('userId', 'username'),
      credentials: { username, password }
    });
  } catch (error) {
    console.error('Convert trial to student error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

router.put('/students/:id', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const updateData = {
      ...req.body,
      subjects: req.body.subjects ? JSON.parse(req.body.subjects) : undefined
    };

    // Handle profile picture upload
    if (req.file) {
      const fs = require('fs');
      const path = require('path');
      
      try {
        // Read the uploaded file and convert to base64
        const filePath = path.join(__dirname, '../../uploads', req.file.filename);
        const fileBuffer = fs.readFileSync(filePath);
        const base64Data = fileBuffer.toString('base64');
        
        // Store both filename and base64 data
        updateData.profilePicture = base64Data;
        updateData.profilePictureType = req.file.mimetype;
        updateData.profilePictureFilename = req.file.filename;
      } catch (fileError) {
        console.error('Error processing uploaded file:', fileError);
        // Fall back to filename only
        updateData.profilePicture = req.file.filename;
      }
    }

    const student = await Student.findByIdAndUpdate(
      req.params.id,
      updateData,
      { new: true }
    );

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    res.json({ message: 'Student updated successfully', student });
  } catch (error) {
    console.error('Update student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/teachers/:id', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const teacher = await Teacher.findByIdAndUpdate(
      req.params.id,
      {
        ...req.body,
        ...(req.file && { profilePicture: req.file.filename }),
        subjects: req.body.subjects ? JSON.parse(req.body.subjects) : undefined,
        availableDays: req.body.availableDays ? JSON.parse(req.body.availableDays) : undefined
      },
      { new: true }
    );

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    res.json({ message: 'Teacher updated successfully', teacher });
  } catch (error) {
    console.error('Update teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.delete('/students/:id', adminAuth, async (req, res) => {
  try {
    const student = await Student.findByIdAndUpdate(
      req.params.id,
      { 
        isDeleted: true, 
        isActive: false,
        deletedAt: new Date() 
      },
      { new: true }
    );

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    await User.findByIdAndUpdate(student.userId, { 
      isDeleted: true, 
      isActive: false,
      deletedAt: new Date() 
    });

    res.json({ message: 'Student deleted successfully' });
  } catch (error) {
    console.error('Delete student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.delete('/teachers/:id', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findByIdAndUpdate(
      req.params.id,
      { 
        isDeleted: true, 
        isActive: false,
        deletedAt: new Date() 
      },
      { new: true }
    );

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    await User.findByIdAndUpdate(teacher.userId, { 
      isDeleted: true, 
      isActive: false,
      deletedAt: new Date() 
    });

    res.json({ message: 'Teacher deleted successfully' });
  } catch (error) {
    console.error('Delete teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/students/:id/restore', adminAuth, async (req, res) => {
  try {
    const student = await Student.findByIdAndUpdate(
      req.params.id,
      { 
        isDeleted: false, 
        isActive: true,
        deletedAt: null 
      },
      { new: true }
    );

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    await User.findByIdAndUpdate(student.userId, { 
      isDeleted: false, 
      isActive: true,
      deletedAt: null 
    });

    res.json({ message: 'Student restored successfully', student });
  } catch (error) {
    console.error('Restore student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/teachers/:id/restore', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findByIdAndUpdate(
      req.params.id,
      { 
        isDeleted: false, 
        isActive: true,
        deletedAt: null 
      },
      { new: true }
    );

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    await User.findByIdAndUpdate(teacher.userId, { 
      isDeleted: false, 
      isActive: true,
      deletedAt: null 
    });

    res.json({ message: 'Teacher restored successfully', teacher });
  } catch (error) {
    console.error('Restore teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Pause/Unpause student
router.put('/students/:id/toggle-status', adminAuth, async (req, res) => {
  try {
    const student = await Student.findById(req.params.id);
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    const newStatus = !student.isActive;
    
    // Update student status
    await Student.findByIdAndUpdate(req.params.id, { isActive: newStatus });
    
    // If pausing student, remove/cancel all their scheduled classes
    if (!newStatus) {
      // Remove from all future classes
      if (Class && typeof Class.updateMany === 'function') {
        await Class.updateMany(
          { 
            students: req.params.id,
            date: { $gte: new Date() }
          },
          { 
            $pull: { students: req.params.id },
            $push: { 
              cancelledStudents: {
                studentId: req.params.id,
                reason: 'Student paused',
                cancelledAt: new Date()
              }
            }
          }
        );
      }
    }
    
    res.json({ 
      message: `Student ${newStatus ? 'activated' : 'paused'} successfully`,
      isActive: newStatus 
    });
  } catch (error) {
    console.error('Toggle student status error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Permanently delete student
router.delete('/students/:id/permanent', adminAuth, async (req, res) => {
  try {
    const student = await Student.findById(req.params.id);
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    // Remove from all classes
    if (Class && typeof Class.updateMany === 'function') {
      await Class.updateMany(
        { students: req.params.id },
        { 
          $pull: { students: req.params.id },
          $push: { 
            cancelledStudents: {
              studentId: req.params.id,
              reason: 'Student deleted',
              cancelledAt: new Date()
            }
          }
        }
      );
    }

    // Delete user account
    await User.findByIdAndDelete(student.userId);
    
    // Delete student record
    await Student.findByIdAndDelete(req.params.id);

    res.json({ message: 'Student permanently deleted' });
  } catch (error) {
    console.error('Permanent delete student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Pause/Unpause teacher
router.put('/teachers/:id/toggle-status', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findById(req.params.id);
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    const newStatus = !teacher.isActive;
    
    // Update teacher status
    await Teacher.findByIdAndUpdate(req.params.id, { isActive: newStatus });
    
    // If pausing teacher, cancel all their scheduled classes
    if (!newStatus) {
      // Cancel all future classes taught by this teacher
      if (Class && typeof Class.updateMany === 'function') {
        await Class.updateMany(
          { 
            teacherId: req.params.id,
            date: { $gte: new Date() }
          },
          { 
            status: 'cancelled',
            cancelReason: 'Teacher paused',
            cancelledAt: new Date()
          }
        );
      }
    }
    
    res.json({ 
      message: `Teacher ${newStatus ? 'activated' : 'paused'} successfully`,
      isActive: newStatus 
    });
  } catch (error) {
    console.error('Toggle teacher status error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Permanently delete teacher
router.delete('/teachers/:id/permanent', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findById(req.params.id);
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Cancel all classes taught by this teacher
    if (Class && typeof Class.updateMany === 'function') {
      await Class.updateMany(
        { teacherId: req.params.id },
        { 
          status: 'cancelled',
          cancelReason: 'Teacher deleted',
          cancelledAt: new Date()
        }
      );
    }

    // Delete user account
    await User.findByIdAndDelete(teacher.userId);
    
    // Delete teacher record
    await Teacher.findByIdAndDelete(req.params.id);

    res.json({ message: 'Teacher permanently deleted' });
  } catch (error) {
    console.error('Permanent delete teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/profile/picture', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const userId = req.user._id;
    
    if (!req.file) {
      return res.status(400).json({ message: 'No file uploaded' });
    }

    // Handle mock user (bypass database)
    if (userId === 'admin123') {
      return res.json({
        message: 'Profile picture updated successfully',
        profilePicture: req.file.filename,
        user: {
          id: 'admin123',
          username: 'admin@edumetrix.uk',
          userType: 'admin',
          profile: {
            name: 'System Administrator',
            email: 'admin@edumetrix.uk',
            position: 'System Administrator',
            department: 'Administration',
            profilePicture: req.file.filename
          }
        }
      });
    }

    const user = await User.findByIdAndUpdate(
      userId,
      { 'profile.profilePicture': req.file.filename },
      { new: true }
    );

    if (!user) {
      return res.status(404).json({ message: 'User not found' });
    }

    res.json({
      message: 'Profile picture updated successfully',
      profilePicture: req.file.filename,
      user: {
        id: user._id,
        username: user.username,
        userType: user.userType,
        profile: user.profile
      }
    });
  } catch (error) {
    console.error('Update profile picture error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/syllabus-options/:country', adminAuth, (req, res) => {
  try {
    const options = getSyllabusOptions(req.params.country);
    res.json({ syllabusOptions: options });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Class Scheduling API endpoints

// Schedule a class (single or recurring)
router.post('/schedule-class', adminAuth, async (req, res) => {
  try {
    console.log('Schedule class request received:', {
      body: req.body,
      user: req.user,
      userId: req.user?._id || req.user?.id || req.user?.userId,
      userIdType: typeof (req.user?._id || req.user?.id || req.user?.userId),
      headers: req.headers
    });

    const {
      studentId,
      teacherId,
      subject,
      scheduledDate,
      startTime,
      endTime,
      duration,
      timezone,
      googleMeetLink,
      classColor,
      recurringSchedule
    } = req.body;

    // Validate required fields
    if (!studentId || !teacherId || !subject || !scheduledDate || !googleMeetLink || !startTime || !endTime) {
      return res.status(400).json({ 
        message: 'Missing required fields',
        missing: {
          studentId: !studentId,
          teacherId: !teacherId,
          subject: !subject,
          scheduledDate: !scheduledDate,
          googleMeetLink: !googleMeetLink,
          startTime: !startTime,
          endTime: !endTime
        }
      });
    }

    // Verify student and teacher exist
    const [student, teacher] = await Promise.all([
      Student.findById(studentId),
      Teacher.findById(teacherId)
    ]);

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Verify teacher teaches the subject
    if (!teacher.subjects.includes(subject)) {
      return res.status(400).json({ 
        message: `Teacher ${teacher.name} does not teach ${subject}` 
      });
    }

    // Verify student studies the subject
    if (!student.subjects.includes(subject)) {
      return res.status(400).json({ 
        message: `Student ${student.studentName} does not study ${subject}` 
      });
    }

    const classData = {
      studentId,
      teacherId,
      subject,
      scheduledDate: new Date(scheduledDate),
      startTime,
      endTime,
      duration: duration || 60,
      timezone: timezone || 'Asia/Dubai',
      googleMeetLink,
      classColor: classColor || '#1976d2',
      recurringSchedule: recurringSchedule || { isRecurring: false },
      createdBy: (() => {
        const userId = req.user._id || req.user.id || req.user.userId;
        // Check if userId is a valid ObjectId format
        if (userId && mongoose.Types.ObjectId.isValid(userId)) {
          return userId;
        }
        // For development/testing, create a valid ObjectId or skip this field
        console.warn('Invalid ObjectId for createdBy:', userId, 'Skipping field...');
        return null;
      })()
    };

    console.log('Creating class with data:', classData);

    let createdClasses = [];

    if (!recurringSchedule.isRecurring) {
      // Create single class
      const newClass = new Class(classData);
      await newClass.save();
      console.log('🎓 Single class created:', {
        id: newClass._id,
        subject: newClass.subject,
        scheduledDate: newClass.scheduledDate,
        student: newClass.studentId,
        teacher: newClass.teacherId
      });
      createdClasses = [newClass];
    } else {
      // Create recurring classes using the static method
      createdClasses = await Class.createRecurringClasses(classData, { maxClasses: 100 });
      console.log('🎓 Recurring classes created:', createdClasses.length);
    }

    // Populate the created classes with student and teacher details
    const populatedClasses = await Class.find({
      _id: { $in: createdClasses.map(c => c._id) }
    })
    .populate('studentId', 'studentName email country class')
    .populate('teacherId', 'name email subjects')
    .sort({ scheduledDate: 1 });

    res.status(201).json({
      message: `Successfully scheduled ${populatedClasses.length} class${populatedClasses.length > 1 ? 'es' : ''}`,
      count: populatedClasses.length,
      classes: populatedClasses
    });

  } catch (error) {
    console.error('Schedule class error:', error);
    console.error('Error details:', {
      message: error.message,
      stack: error.stack,
      reqBody: req.body,
      userId: req.user?._id || req.user?.id || req.user?.userId
    });
    res.status(500).json({ 
      message: 'Server error: ' + error.message, 
      error: process.env.NODE_ENV === 'development' ? error.message : undefined 
    });
  }
});

// Get all scheduled classes
router.get('/classes', adminAuth, async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 50, 
      studentId, 
      teacherId, 
      subject, 
      status = 'all',
      startDate,
      endDate
    } = req.query;

    // Build query
    const query = {
      isDeleted: false
    };

    if (studentId) query.studentId = studentId;
    if (teacherId) query.teacherId = teacherId;
    if (subject) query.subject = subject;
    if (status !== 'all') query.status = status;

    // Date range filter
    if (startDate || endDate) {
      query.scheduledDate = {};
      if (startDate) query.scheduledDate.$gte = new Date(startDate);
      if (endDate) query.scheduledDate.$lte = new Date(endDate);
    }

    const classes = await Class.find(query)
      .populate('studentId', 'studentName email country class timezone')
      .populate('teacherId', 'name email subjects')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ scheduledDate: 1 });

    const total = await Class.countDocuments(query);

    res.json({
      classes,
      totalPages: Math.ceil(total / limit),
      currentPage: parseInt(page),
      total
    });

  } catch (error) {
    console.error('Get classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get classes for calendar view
router.get('/classes/calendar', async (req, res) => {
  try {
    const { startDate, endDate, timezone = 'Asia/Kolkata' } = req.query;

    console.log('📅 Calendar API called with:', {
      startDate,
      endDate,
      timezone,
      user: req.user?._id
    });

    if (!startDate || !endDate) {
      return res.status(400).json({ 
        message: 'startDate and endDate are required' 
      });
    }

    const query = {
      scheduledDate: {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      },
      isDeleted: false
    };

    console.log('📅 Database query:', query);
    console.log('📅 Date range:', {
      start: new Date(startDate),
      end: new Date(endDate)
    });

    const classes = await Class.find(query)
    .populate('studentId', 'studentName email country class')
    .populate('teacherId', 'name email')
    .sort({ scheduledDate: 1 });

    console.log('📅 Found classes:', classes.length);
    console.log('📅 Class details:', classes.map(c => ({
      id: c._id,
      subject: c.subject,
      scheduledDate: c.scheduledDate,
      student: c.studentId?.studentName,
      teacher: c.teacherId?.name
    })));

    // Also check total classes in database for debugging
    const totalClasses = await Class.countDocuments({ isDeleted: false });
    console.log('📅 Total classes in database:', totalClasses);

    if (classes.length === 0 && totalClasses > 0) {
      // Get a few sample classes to see the date formats
      const sampleClasses = await Class.find({ isDeleted: false }).limit(3);
      console.log('📅 Sample classes with dates:', sampleClasses.map(c => ({
        id: c._id,
        scheduledDate: c.scheduledDate,
        dateString: c.scheduledDate.toISOString()
      })));
    }

    // Format classes for calendar display
    const calendarClasses = classes.map(cls => ({
      id: cls._id.toString(), // Convert ObjectId to string
      _id: cls._id.toString(), // Also include _id as string
      title: `${cls.subject} - ${cls.studentId?.studentName || 'Unknown Student'}`,
      subject: cls.subject,
      date: cls.scheduledDate, // Add date field for frontend compatibility
      time: cls.startTime, // Add time field for frontend compatibility  
      start: cls.scheduledDate,
      end: new Date(cls.scheduledDate.getTime() + (cls.duration * 60000)),
      duration: cls.duration,
      backgroundColor: cls.classColor,
      borderColor: cls.classColor,
      status: cls.status || 'scheduled',
      meetingLink: cls.googleMeetLink,
      student: cls.studentId,
      teacher: cls.teacherId,
      extendedProps: {
        student: cls.studentId,
        teacher: cls.teacherId,
        subject: cls.subject,
        duration: cls.duration,
        googleMeetLink: cls.googleMeetLink,
        status: cls.status,
        timezone: cls.timezone,
        startTime: cls.startTime,
        endTime: cls.endTime
      }
    }));

    console.log('📅 Formatted calendar classes:', calendarClasses.length);
    console.log('📅 Sample formatted class:', calendarClasses[0]);

    res.json({ classes: calendarClasses });

  } catch (error) {
    console.error('Get calendar classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update class
router.put('/classes/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    console.log('📝 Class update request:', {
      classId: id,
      updates: updates,
      updateKeys: Object.keys(updates)
    });

    // Remove fields that shouldn't be updated directly
    delete updates._id;
    delete updates.createdAt;
    delete updates.updatedAt;

    // Validate required fields if they're being updated
    if (updates.scheduledDate && !updates.scheduledDate) {
      return res.status(400).json({ message: 'Invalid scheduledDate format' });
    }

    if (updates.duration && (isNaN(updates.duration) || updates.duration < 15)) {
      return res.status(400).json({ message: 'Duration must be a number >= 15 minutes' });
    }

    const updatedClass = await Class.findByIdAndUpdate(
      id,
      { ...updates, updatedAt: new Date() },
      { new: true, runValidators: true }
    )
    .populate('studentId', 'studentName email country class')
    .populate('teacherId', 'name email subjects');

    if (!updatedClass) {
      return res.status(404).json({ message: 'Class not found' });
    }

    console.log('✅ Class updated successfully:', {
      id: updatedClass._id,
      scheduledDate: updatedClass.scheduledDate,
      startTime: updatedClass.startTime,
      endTime: updatedClass.endTime,
      duration: updatedClass.duration,
      timezone: updatedClass.timezone
    });

    res.json({
      message: 'Class updated successfully',
      class: updatedClass,
      debug: {
        originalUpdates: req.body,
        finalScheduledDate: updatedClass.scheduledDate,
        finalStartTime: updatedClass.startTime,
        finalEndTime: updatedClass.endTime
      }
    });

  } catch (error) {
    console.error('Update class error:', {
      error: error.message,
      stack: error.stack,
      classId: req.params.id,
      updates: req.body
    });
    
    // Send more specific error message
    if (error.name === 'ValidationError') {
      return res.status(400).json({ 
        message: 'Validation error: ' + Object.values(error.errors).map(e => e.message).join(', ')
      });
    }
    
    if (error.name === 'CastError') {
      return res.status(400).json({ message: 'Invalid data format: ' + error.message });
    }
    
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Delete class
router.delete('/classes/:id', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    console.log('🗑️ Delete class request:', { classId: id, userId: req.user?._id });

    // Validate ObjectId format
    if (!mongoose.Types.ObjectId.isValid(id)) {
      console.log('❌ Invalid class ID format:', id);
      return res.status(400).json({ message: 'Invalid class ID format' });
    }

    // First check if the class exists
    const existingClass = await Class.findById(id);
    if (!existingClass) {
      console.log('❌ Class not found:', id);
      return res.status(404).json({ message: 'Class not found' });
    }

    if (existingClass.isDeleted) {
      console.log('⚠️ Class already deleted:', id);
      return res.status(400).json({ message: 'Class is already deleted' });
    }

    console.log('📋 Found class to delete:', {
      id: existingClass._id,
      subject: existingClass.subject,
      scheduledDate: existingClass.scheduledDate
    });

    const deletedClass = await Class.findByIdAndUpdate(
      id,
      { 
        isDeleted: true,
        deletedAt: new Date()
      },
      { new: true }
    );

    if (!deletedClass) {
      console.log('❌ Class not found for deletion:', id);
      return res.status(404).json({ message: 'Class not found' });
    }

    console.log('✅ Class deleted successfully:', {
      classId: id,
      subject: deletedClass.subject,
      scheduledDate: deletedClass.scheduledDate
    });

    res.json({ message: 'Class deleted successfully' });

  } catch (error) {
    console.error('❌ Delete class error:', error);
    console.error('Error details:', error.message);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Cancel class
router.put('/classes/:id/cancel', async (req, res) => {
  try {
    const { id } = req.params;
    const { reason } = req.body;

    const cancelledClass = await Class.findByIdAndUpdate(
      id,
      { 
        status: 'cancelled',
        notes: reason || 'Cancelled by admin'
      },
      { new: true }
    )
    .populate('studentId', 'studentName email')
    .populate('teacherId', 'name email');

    if (!cancelledClass) {
      return res.status(404).json({ message: 'Class not found' });
    }

    res.json({
      message: 'Class cancelled successfully',
      class: cancelledClass
    });

  } catch (error) {
    console.error('Cancel class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get class statistics
router.get('/classes/stats', async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    const dateFilter = {};
    if (startDate || endDate) {
      dateFilter.scheduledDate = {};
      if (startDate) dateFilter.scheduledDate.$gte = new Date(startDate);
      if (endDate) dateFilter.scheduledDate.$lte = new Date(endDate);
    }

    const [
      totalClasses,
      scheduledClasses,
      completedClasses,
      cancelledClasses,
      ongoingClasses,
      subjectStats,
      monthlyStats
    ] = await Promise.all([
      Class.countDocuments({ isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'scheduled', isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'completed', isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'cancelled', isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'ongoing', isDeleted: false, ...dateFilter }),
      
      // Subject-wise statistics
      Class.aggregate([
        { $match: { isDeleted: false, ...dateFilter } },
        { $group: { _id: '$subject', count: { $sum: 1 } } },
        { $sort: { count: -1 } }
      ]),
      
      // Monthly statistics
      Class.aggregate([
        { $match: { isDeleted: false, ...dateFilter } },
        {
          $group: {
            _id: {
              year: { $year: '$scheduledDate' },
              month: { $month: '$scheduledDate' }
            },
            count: { $sum: 1 }
          }
        },
        { $sort: { '_id.year': 1, '_id.month': 1 } }
      ])
    ]);

    res.json({
      summary: {
        total: totalClasses,
        scheduled: scheduledClasses,
        completed: completedClasses,
        cancelled: cancelledClasses,
        ongoing: ongoingClasses
      },
      bySubject: subjectStats,
      monthly: monthlyStats
    });

  } catch (error) {
    console.error('Get class stats error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;