import express from 'express';
import bcrypt from 'bcryptjs';
import multer from 'multer';
import path from 'path';
import fs from 'fs';
import mongoose from 'mongoose';
import User from '../../models/admin/User.js';
import Student from '../../models/student/Student.js';
import Teacher from '../../models/teacher/Teacher.js';
import TrialRequest from '../../models/admin/TrialRequest.js';
import TeacherClass from '../../models/teacher/Class.js';
import Class from '../../models/Class.js';
import Invoice from '../../models/admin/Invoice.js';
import Homework from '../../models/teacher/Homework.js';
import Assignment from '../../models/teacher/Assignment.js';
import Exam from '../../models/teacher/Exam.js';
import AssignmentSubmission from '../../models/student/AssignmentSubmission.js';
import Submission from '../../models/student/Submission.js';
import QuickTest from '../../models/teacher/QuickTest.js';
import StudentSettings from '../../models/student/StudentSettings.js';
import TeacherSettings from '../../models/teacher/TeacherSettings.js';
import { adminAuth } from '../../middleware/auth.js';
import { generatePassword, generateUsername } from '../../utils/passwordGenerator.js';
import { getSyllabusOptions } from '../../utils/syllabusHelper.js';
import RescheduleRequest from '../../models/shared/RescheduleRequest.js';
import { createNotification, NotificationTemplates } from '../../utils/notificationHelper.js';
import moment from 'moment-timezone';

const router = express.Router();

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/');
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + Math.round(Math.random() * 1E9) + path.extname(file.originalname));
  }
});

const upload = multer({ storage });


router.get('/detail', async (req, res) => {
  try {
    const admin = await User.findOne({ userType: 'admin', isActive: true, isDeleted: false })
      .select('-password'); // never send password

    if (!admin) {
      return res.status(404).json({ message: 'Admin not found' });
    }

    res.json({ admin });
  } catch (err) {
    console.error('Error fetching admin:', err);
    res.status(500).json({ message: 'Server error fetching admin' });
  }
});
router.post('/students', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const {
      studentName,
      parentName,
      email,
      countryCode,
      phoneNumber,
      hourlyRate,
      currency,
      country,
      syllabus,
      class: studentClass,
      subjects,
      googleMeetLink,
      trialRequestId
    } = req.body;

    if (!getSyllabusOptions(country).includes(syllabus)) {
      return res.status(400).json({ message: 'Invalid syllabus for selected country' });
    }

    const username = generateUsername(studentName);
    const password = generatePassword(studentName);
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = new User({
      username,
      password: hashedPassword,
      userType: 'student'
    });

    await user.save();

    // Handle profile picture upload for new student
    let profilePictureData = null;
    let profilePictureType = null;
    let profilePictureFilename = null;

    if (req.file) {
      
      try {
        // Read the uploaded file and convert to base64
        const filePath = path.join(__dirname, '../../uploads', req.file.filename);
        const fileBuffer = fs.readFileSync(filePath);
        profilePictureData = fileBuffer.toString('base64');
        profilePictureType = req.file.mimetype;
        profilePictureFilename = req.file.filename;
      } catch (fileError) {
        console.error('Error processing uploaded file:', fileError);
        // Fall back to filename only
        profilePictureData = req.file.filename;
      }
    }

    // Map country to timezone
    const countryTimezoneMap = {
      'UAE': 'Asia/Dubai',
      'Saudi Arabia': 'Asia/Riyadh',
      'Qatar': 'Asia/Qatar',
      'Kuwait': 'Asia/Kuwait',
      'Bahrain': 'Asia/Bahrain',
      'Oman': 'Asia/Muscat',
      'India': 'Asia/Kolkata',
      'UK': 'Europe/London',
      'Australia': 'Australia/Sydney',
      'New Zealand': 'Pacific/Auckland'
    };

    const student = new Student({
      userId: user._id,
      studentName,
      parentName,
      email,
      phoneNumber: { countryCode, number: phoneNumber },
      profilePicture: profilePictureData,
      profilePictureType: profilePictureType,
      profilePictureFilename: profilePictureFilename,
      hourlyRate: parseFloat(hourlyRate),
      currency,
      country,
      timezone: countryTimezoneMap[country] || 'UTC', // Set timezone based on country
      syllabus,
      class: studentClass,
      subjects: JSON.parse(subjects),
      googleMeetLink,
      originalPassword: password // Store original password for admin access
    });

    await student.save();

    // If this student was created from a trial request, mark it as processed
    if (trialRequestId) {
      try {
        console.log('Updating trial request status for ID:', trialRequestId);
        await TrialRequest.findByIdAndUpdate(
          trialRequestId, 
          { 
            status: 'processed',
            processedAt: new Date(),
            studentId: student._id 
          }
        );
        console.log('Trial request marked as processed successfully');
      } catch (trialError) {
        console.error('Failed to update trial request:', trialError);
        // Don't fail the student creation if trial request update fails
      }
    }

    res.status(201).json({
      message: 'Student created successfully',
      credentials: { username, password },
      student: await Student.findById(student._id).populate('userId', 'username')
    });
  } catch (error) {
    console.error('Create student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/teachers', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const {
      name,
      countryCode,
      phoneNumber,
      email,
      subjects,
      salaryType,
      salaryAmount,
      accountHolderName,
      aadhaarNumber,
      panNumber,
      accountNumber,
      bankName,
      ifscCode,
      branchName,
      startTime,
      endTime,
      availableDays,
      bio,
      qualification,
      specialization,
      joiningDate
    } = req.body;

    const username = generateUsername(name);
    const password = generatePassword(name);
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = new User({
      username,
      password: hashedPassword,
      userType: 'teacher'
    });

    await user.save();

    const teacher = new Teacher({
      userId: user._id,
      name,
      phoneNumber: { countryCode, number: phoneNumber },
      email,
      bio,
      profilePicture: req.file ? req.file.filename : null,
      subjects: JSON.parse(subjects),
      salaryType,
      salary: {
        amount: parseFloat(salaryAmount),
        currency: 'INR'
      },
      bankingDetails: {
        accountHolderName,
        accountNumber,
        bankName,
        ifscCode,
        branchName,
        aadhaarNumber,
        panNumber
      },
      availableTime: {
        startTime,
        endTime
      },
      availableDays: JSON.parse(availableDays),
      timezone: 'Asia/Kolkata', // IST timezone for all teachers
      professionalInfo: {
        qualification: qualification || null,
        specialization: specialization || null,
        joiningDate: joiningDate ? new Date(joiningDate) : null
      },
      originalPassword: password // Store original password for admin access
    });

    await teacher.save();

    res.status(201).json({
      message: 'Teacher created successfully',
      credentials: { username, password },
      teacher: await Teacher.findById(teacher._id).populate('userId', 'username')
    });
  } catch (error) {
    console.error('Create teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/trial-requests', adminAuth, async (req, res) => {
  try {
    console.log('📝 Received trial request data:', JSON.stringify(req.body, null, 2));
    console.log('👤 User info:', req.user ? req.user._id : 'No user found');
    
    const trialRequest = new TrialRequest({
      ...req.body,
      source: 'manual'
    });

    console.log('💾 About to save trial request with data:', JSON.stringify(trialRequest.toObject(), null, 2));
    await trialRequest.save();
    console.log('✅ Trial request saved successfully');
    res.status(201).json({ message: 'Trial request created successfully', trialRequest });
  } catch (error) {
    console.error('Create trial request error:', error);
    console.error('Error details:', error.errors);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

router.get('/students', adminAuth, async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 10, 
      search = '', 
      isDeleted = 'false',
      country,
      syllabus,
      class: studentClass,
      currency,
      isActive,
      sortBy = 'createdAt',
      sortOrder = 'desc'
    } = req.query;
    
    const query = {
      isDeleted: isDeleted === 'true',
      ...(search && {
        $or: [
          { studentName: { $regex: search, $options: 'i' } },
          { parentName: { $regex: search, $options: 'i' } },
          { email: { $regex: search, $options: 'i' } }
        ]
      }),
      ...(country && { country }),
      ...(syllabus && { syllabus }),
      ...(studentClass && { class: studentClass }),
      ...(currency && { currency }),
      ...(isActive !== undefined && { isActive: isActive === 'true' })
    };

    // Build sort object
    const sortObject = {};
    const validSortFields = ['studentName', 'parentName', 'email', 'country', 'class', 'syllabus', 'hourlyRate', 'createdAt', 'isActive'];
    const sortField = validSortFields.includes(sortBy) ? sortBy : 'createdAt';
    const sortDirection = sortOrder === 'asc' ? 1 : -1;
    sortObject[sortField] = sortDirection;

    const students = await Student.find(query)
      .populate('userId', 'username isActive')
      .populate('assignedTeachers.teacherId', 'name')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort(sortObject);

    const total = await Student.countDocuments(query);

    res.json({
      students,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total,
      sortBy: sortField,
      sortOrder
    });
  } catch (error) {
    console.error('Get students error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get individual student details with credentials
router.get('/students/:id', adminAuth, async (req, res) => {
  try {
    const student = await Student.findById(req.params.id)
      .populate('userId', 'username')
      .populate('assignedTeachers.teacherId', 'name');
    
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    // Get the user's credentials
    const user = await User.findById(student.userId._id);
    
    let password = student.originalPassword;

    // If originalPassword doesn't exist, generate a new one and save it
    if (!password) {
      password = generatePassword(student.studentName);

      // Update the student record with the new password
      await Student.findByIdAndUpdate(req.params.id, { originalPassword: password });

      // Also update the user's hashed password
      const hashedPassword = await bcrypt.hash(password, 10);
      await User.findByIdAndUpdate(student.userId._id, { password: hashedPassword });
    }
    
    res.json({
      student,
      credentials: {
        username: user.username,
        password: password || 'Unable to generate password'
      }
    });
  } catch (error) {
    console.error('Get student details error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/teachers', adminAuth, async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 10, 
      search = '', 
      isDeleted = 'false',
      salaryType,
      isActive,
      sortBy = 'createdAt',
      sortOrder = 'desc'
    } = req.query;
    
    const query = {
      isDeleted: isDeleted === 'true',
      ...(search && {
        $or: [
          { name: { $regex: search, $options: 'i' } },
          { email: { $regex: search, $options: 'i' } }
        ]
      }),
      ...(salaryType && { salaryType }),
      ...(isActive !== undefined && { isActive: isActive === 'true' })
    };

    // Build sort object  
    const sortObject = {};
    const validSortFields = ['name', 'email', 'salaryType', 'salary.amount', 'rating', 'createdAt', 'isActive'];
    const sortField = validSortFields.includes(sortBy) ? sortBy : 'createdAt';
    const sortDirection = sortOrder === 'asc' ? 1 : -1;
    sortObject[sortField] = sortDirection;

    console.log('📚 Teacher Query:', {
      query,
      page,
      limit,
      limitType: typeof limit,
      skip: (page - 1) * limit
    });

    // Convert limit to integer explicitly
    const limitInt = parseInt(limit);
    const pageInt = parseInt(page);

    const teachers = await Teacher.find(query)
      .populate('userId', 'username isActive')
      .populate('assignedStudents.studentId', 'studentName')
      .limit(limitInt)
      .skip((pageInt - 1) * limitInt)
      .sort(sortObject);

    const total = await Teacher.countDocuments(query);

    console.log('📚 Teacher Results:', {
      foundTeachers: teachers.length,
      totalInDB: total,
      totalPages: Math.ceil(total / limitInt),
      currentPage: pageInt,
      limitUsed: limitInt,
      skipUsed: (pageInt - 1) * limitInt
    });

    res.json({
      teachers,
      totalPages: Math.ceil(total / limitInt),
      currentPage: pageInt,
      total,
      sortBy: sortField,
      sortOrder
    });
  } catch (error) {
    console.error('Get teachers error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get individual teacher details with credentials
router.get('/teachers/:id', adminAuth, async (req, res) => {
  try {
    console.log('🔍 Fetching teacher with ID:', req.params.id);

    const teacher = await Teacher.findById(req.params.id)
      .populate('userId', 'username')
      .populate('assignedStudents.studentId', 'studentName');

    if (!teacher) {
      console.log('❌ Teacher not found with ID:', req.params.id);
      return res.status(404).json({ message: 'Teacher not found' });
    }

    console.log('✅ Teacher found:', teacher.name);
    console.log('👤 Teacher userId:', teacher.userId);
    console.log('👥 Assigned students count:', teacher.assignedStudents?.length || 0);
    console.log('👥 Assigned students:', teacher.assignedStudents);

    let user;
    let password = teacher.originalPassword;

    // Handle missing userId - create a new user account
    if (!teacher.userId) {
      console.log('⚠️ Teacher has no userId, creating new user account...');

      // Generate credentials
      const username = generateUsername(teacher.name);
      password = password || generatePassword(teacher.name);
      const hashedPassword = await bcrypt.hash(password, 10);

      // Create new user
      user = new User({
        username,
        password: hashedPassword,
        userType: 'teacher'
      });
      await user.save();

      // Update teacher with userId and originalPassword
      await Teacher.findByIdAndUpdate(req.params.id, {
        userId: user._id,
        originalPassword: password
      });

      // Re-populate the teacher object
      teacher.userId = user;

      console.log('✅ Created new user account:', username);
    } else {
      // Get the existing user's credentials
      const userId = teacher.userId._id || teacher.userId;
      user = await User.findById(userId);

      if (!user) {
        console.error('❌ User not found for teacher, creating new one...');

        // User was deleted, create a new one
        const username = generateUsername(teacher.name);
        password = password || generatePassword(teacher.name);
        const hashedPassword = await bcrypt.hash(password, 10);

        user = new User({
          username,
          password: hashedPassword,
          userType: 'teacher'
        });
        await user.save();

        // Update teacher with new userId and originalPassword
        await Teacher.findByIdAndUpdate(req.params.id, {
          userId: user._id,
          originalPassword: password
        });

        teacher.userId = user;

        console.log('✅ Recreated user account:', username);
      }
    }

    console.log('✅ User found:', user.username);

    // If originalPassword doesn't exist, generate a new one and save it
    if (!password) {
      console.log('⚠️ No originalPassword found, generating new one');
      password = generatePassword(teacher.name);

      // Update the teacher record with the new password
      await Teacher.findByIdAndUpdate(req.params.id, { originalPassword: password });

      // Also update the user's hashed password
      const hashedPassword = await bcrypt.hash(password, 10);
      await User.findByIdAndUpdate(user._id, { password: hashedPassword });

      console.log('✅ Password generated and saved');
    }

    // Calculate total classes for this teacher
    const totalClasses = await Class.countDocuments({ teacherId: req.params.id });
    console.log(`📊 Total classes for teacher: ${totalClasses}`);

    // Convert teacher to plain object and add totalClasses
    const teacherObject = teacher.toObject();
    teacherObject.totalClasses = totalClasses;

    res.json({
      teacher: teacherObject,
      credentials: {
        username: user.username,
        password: password || 'Unable to generate password'
      }
    });
  } catch (error) {
    console.error('❌ Get teacher details error:', error);
    console.error('Error stack:', error.stack);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

router.get('/trial-requests', adminAuth, async (req, res) => {
  try {
    const { page = 1, limit = 10, status = 'all' } = req.query;
    
    // Build query - exclude 'processed' requests from 'all' view
    let query = {};
    if (status === 'all') {
      query = { status: { $nin: ['processed'] } }; // Exclude processed requests
    } else {
      query = { status };
    }

    const trialRequests = await TrialRequest.find(query)
      .populate('assignedTeacher', 'name')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ createdAt: -1 });

    const total = await TrialRequest.countDocuments(query);

    res.json({
      trialRequests,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get trial requests error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update trial request
router.put('/trial-requests/:id', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;
    
    const trialRequest = await TrialRequest.findByIdAndUpdate(
      id,
      { ...updates, updatedAt: new Date() },
      { new: true, runValidators: true }
    );

    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    res.json({
      message: 'Trial request updated successfully',
      trialRequest
    });
  } catch (error) {
    console.error('Update trial request error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Update trial request status
router.put('/trial-requests/:id/status', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    const { status } = req.body;

    if (!status) {
      return res.status(400).json({ message: 'Status is required' });
    }

    const validStatuses = ['pending', 'scheduled', 'completed', 'cancelled', 'converted'];
    if (!validStatuses.includes(status)) {
      return res.status(400).json({ message: 'Invalid status' });
    }

    const trialRequest = await TrialRequest.findByIdAndUpdate(
      id,
      { status, updatedAt: new Date() },
      { new: true, runValidators: true }
    );

    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    res.json({
      message: `Trial request status updated to ${status}`,
      trialRequest
    });
  } catch (error) {
    console.error('Update trial request status error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Delete trial request
router.delete('/trial-requests/:id', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;

    const trialRequest = await TrialRequest.findByIdAndDelete(id);

    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    res.json({
      message: 'Trial request deleted successfully'
    });
  } catch (error) {
    console.error('Delete trial request error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Convert trial request to student
router.post('/trial-requests/:id/convert', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    const studentData = req.body;

    // Find the trial request
    const trialRequest = await TrialRequest.findById(id);
    if (!trialRequest) {
      return res.status(404).json({ message: 'Trial request not found' });
    }

    // Create user account for the student
    const username = generateUsername(studentData.studentName);
    const password = generatePassword(studentData.studentName);
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = new User({
      username,
      password: hashedPassword,
      userType: 'student'
    });

    await user.save();

    // Create student record
    const student = new Student({
      userId: user._id,
      studentName: studentData.studentName,
      parentName: studentData.parentName,
      email: studentData.email,
      phoneNumber: studentData.phoneNumber,
      hourlyRate: parseFloat(studentData.hourlyRate),
      currency: studentData.currency,
      country: studentData.country,
      syllabus: studentData.syllabus,
      class: studentData.class,
      subjects: studentData.subjects,
      originalPassword: password,
      trialRequestId: id
    });

    await student.save();

    // Update trial request status to converted
    await TrialRequest.findByIdAndUpdate(id, { 
      status: 'converted',
      convertedAt: new Date(),
      convertedToStudentId: student._id
    });

    res.status(201).json({
      message: 'Trial request converted to student successfully',
      student: await Student.findById(student._id).populate('userId', 'username'),
      credentials: { username, password }
    });
  } catch (error) {
    console.error('Convert trial to student error:', error);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

router.put('/students/:id', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const updateData = {
      ...req.body,
      subjects: req.body.subjects ? JSON.parse(req.body.subjects) : undefined
    };

    // Handle profile picture upload
    if (req.file) {
      try {
        // Read the uploaded file and convert to base64
        const filePath = path.join(__dirname, '../../uploads', req.file.filename);
        const fileBuffer = fs.readFileSync(filePath);
        const base64Data = fileBuffer.toString('base64');
        
        // Store both filename and base64 data
        updateData.profilePicture = base64Data;
        updateData.profilePictureType = req.file.mimetype;
        updateData.profilePictureFilename = req.file.filename;
      } catch (fileError) {
        console.error('Error processing uploaded file:', fileError);
        // Fall back to filename only
        updateData.profilePicture = req.file.filename;
      }
    }

    const student = await Student.findByIdAndUpdate(
      req.params.id,
      updateData,
      { new: true }
    );

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    res.json({ message: 'Student updated successfully', student });
  } catch (error) {
    console.error('Update student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/teachers/:id', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const updateData = { ...req.body };

    // Handle profile picture
    if (req.file) {
      updateData.profilePicture = req.file.filename;
    }

    // Handle subjects array
    if (updateData.subjects) {
      updateData.subjects = JSON.parse(updateData.subjects);
    }

    // Handle availableDays array
    if (updateData.availableDays) {
      updateData.availableDays = JSON.parse(updateData.availableDays);
    }

    // Handle phoneNumber nested object
    if (updateData.countryCode || updateData.phoneNumber) {
      updateData.phoneNumber = {
        countryCode: updateData.countryCode,
        number: updateData.phoneNumber
      };
      delete updateData.countryCode;
    }

    // Handle salary nested object
    if (updateData.salaryAmount || updateData.salaryType) {
      updateData.salary = {
        amount: parseFloat(updateData.salaryAmount),
        currency: 'INR'
      };
      delete updateData.salaryAmount;
    }

    // Handle availableTime nested object
    if (updateData.startTime || updateData.endTime) {
      updateData.availableTime = {
        startTime: updateData.startTime,
        endTime: updateData.endTime
      };
      delete updateData.startTime;
      delete updateData.endTime;
    }

    // Handle bankingDetails nested object
    if (updateData.accountHolderName || updateData.accountNumber ||
        updateData.bankName || updateData.ifscCode ||
        updateData.branchName || updateData.aadhaarNumber || updateData.panNumber) {
      updateData.bankingDetails = {
        accountHolderName: updateData.accountHolderName,
        accountNumber: updateData.accountNumber,
        bankName: updateData.bankName,
        ifscCode: updateData.ifscCode,
        branchName: updateData.branchName,
        aadhaarNumber: updateData.aadhaarNumber,
        panNumber: updateData.panNumber
      };
      delete updateData.accountHolderName;
      delete updateData.accountNumber;
      delete updateData.bankName;
      delete updateData.ifscCode;
      delete updateData.branchName;
      delete updateData.aadhaarNumber;
      delete updateData.panNumber;
    }

    // Handle professionalInfo nested object
    if (updateData.qualification !== undefined || updateData.specialization !== undefined || updateData.joiningDate !== undefined) {
      updateData.professionalInfo = {
        qualification: updateData.qualification || null,
        specialization: updateData.specialization || null,
        joiningDate: updateData.joiningDate ? new Date(updateData.joiningDate) : null
      };
      delete updateData.qualification;
      delete updateData.specialization;
      delete updateData.joiningDate;
    }

    const teacher = await Teacher.findByIdAndUpdate(
      req.params.id,
      updateData,
      { new: true, runValidators: true }
    );

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    res.json({ message: 'Teacher updated successfully', teacher });
  } catch (error) {
    console.error('Update teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.delete('/students/:id', adminAuth, async (req, res) => {
  try {
    const student = await Student.findByIdAndUpdate(
      req.params.id,
      {
        isDeleted: true,
        isActive: false,
        deletedAt: new Date()
      },
      { new: true }
    );

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    // Mark user account as deleted
    await User.findByIdAndUpdate(student.userId, {
      isDeleted: true,
      isActive: false,
      deletedAt: new Date()
    });

    // Mark all student's classes as deleted
    await Class.updateMany(
      { studentId: req.params.id },
      {
        isDeleted: true,
        deletedAt: new Date()
      }
    );

    // Delete all student's invoices
    await Invoice.deleteMany({ studentId: req.params.id });

    // Delete all student's reschedule requests
    await RescheduleRequest.deleteMany({ studentId: req.params.id });

    // Delete all student's homework assignments
    await Homework.deleteMany({ studentId: req.params.id });

    // Delete all student's assignment submissions
    await AssignmentSubmission.deleteMany({ studentId: req.params.id });

    // Delete all student's exam submissions
    await Submission.deleteMany({ studentId: req.params.id });

    // Delete all student's quick test records
    await QuickTest.deleteMany({ studentId: req.params.id });

    // Delete student's settings
    await StudentSettings.deleteMany({ studentId: req.params.id });

    res.json({ message: 'Student deleted successfully' });
  } catch (error) {
    console.error('Delete student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.delete('/teachers/:id', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findByIdAndUpdate(
      req.params.id,
      {
        isDeleted: true,
        isActive: false,
        deletedAt: new Date()
      },
      { new: true }
    );

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Mark user account as deleted
    await User.findByIdAndUpdate(teacher.userId, {
      isDeleted: true,
      isActive: false,
      deletedAt: new Date()
    });

    // Mark all teacher's classes as deleted
    await Class.updateMany(
      { teacherId: req.params.id },
      {
        isDeleted: true,
        deletedAt: new Date()
      }
    );

    // Delete all teacher's homework assignments
    await Homework.deleteMany({ teacherId: req.params.id });

    // Delete all teacher's assignments
    await Assignment.deleteMany({ teacherId: req.params.id });

    // Delete all teacher's exams
    await Exam.deleteMany({ teacherId: req.params.id });

    // Delete all teacher's quick tests
    await QuickTest.deleteMany({ teacherId: req.params.id });

    // Delete teacher's settings
    await TeacherSettings.deleteMany({ teacherId: req.params.id });

    res.json({ message: 'Teacher deleted successfully' });
  } catch (error) {
    console.error('Delete teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/students/:id/restore', adminAuth, async (req, res) => {
  try {
    const student = await Student.findByIdAndUpdate(
      req.params.id,
      {
        isDeleted: false,
        isActive: true,
        deletedAt: null
      },
      { new: true }
    );

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    // Restore user account
    await User.findByIdAndUpdate(student.userId, {
      isDeleted: false,
      isActive: true,
      deletedAt: null
    });

    // Restore student's classes
    await Class.updateMany(
      { studentId: req.params.id },
      {
        isDeleted: false,
        deletedAt: null
      }
    );

    // Note: Invoices and reschedule requests are permanently deleted on soft delete,
    // so they won't be restored. This is intentional to avoid confusion.

    res.json({ message: 'Student restored successfully', student });
  } catch (error) {
    console.error('Restore student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.post('/teachers/:id/restore', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findByIdAndUpdate(
      req.params.id,
      {
        isDeleted: false,
        isActive: true,
        deletedAt: null
      },
      { new: true }
    );

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Restore user account
    await User.findByIdAndUpdate(teacher.userId, {
      isDeleted: false,
      isActive: true,
      deletedAt: null
    });

    // Restore teacher's classes
    await Class.updateMany(
      { teacherId: req.params.id },
      {
        isDeleted: false,
        deletedAt: null
      }
    );

    // Note: Homework, assignments, exams, quick tests, and settings are permanently deleted on soft delete,
    // so they won't be restored. This is intentional to avoid confusion.

    res.json({ message: 'Teacher restored successfully', teacher });
  } catch (error) {
    console.error('Restore teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Pause/Unpause student
router.put('/students/:id/toggle-status', adminAuth, async (req, res) => {
  try {
    const student = await Student.findById(req.params.id);
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    const newStatus = !student.isActive;
    
    // Update student status
    await Student.findByIdAndUpdate(req.params.id, { isActive: newStatus });
    
    // If pausing student, remove/cancel all their scheduled classes
    if (!newStatus) {
      // Remove from all future classes
      if (Class && typeof Class.updateMany === 'function') {
        await Class.updateMany(
          { 
            students: req.params.id,
            date: { $gte: new Date() }
          },
          { 
            $pull: { students: req.params.id },
            $push: { 
              cancelledStudents: {
                studentId: req.params.id,
                reason: 'Student paused',
                cancelledAt: new Date()
              }
            }
          }
        );
      }
    }
    
    res.json({ 
      message: `Student ${newStatus ? 'activated' : 'paused'} successfully`,
      isActive: newStatus 
    });
  } catch (error) {
    console.error('Toggle student status error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Permanently delete student
router.delete('/students/:id/permanent', adminAuth, async (req, res) => {
  try {
    const student = await Student.findById(req.params.id);
    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    // Remove from all classes
    if (Class && typeof Class.updateMany === 'function') {
      await Class.updateMany(
        { students: req.params.id },
        {
          $pull: { students: req.params.id },
          $push: {
            cancelledStudents: {
              studentId: req.params.id,
              reason: 'Student deleted',
              cancelledAt: new Date()
            }
          }
        }
      );
    }

    // Delete all student's classes permanently
    await Class.deleteMany({ studentId: req.params.id });

    // Delete all student's invoices permanently
    await Invoice.deleteMany({ studentId: req.params.id });

    // Delete all student's reschedule requests permanently
    await RescheduleRequest.deleteMany({ studentId: req.params.id });

    // Delete all student's homework assignments permanently
    await Homework.deleteMany({ studentId: req.params.id });

    // Delete all student's assignment submissions permanently
    await AssignmentSubmission.deleteMany({ studentId: req.params.id });

    // Delete all student's exam submissions permanently
    await Submission.deleteMany({ studentId: req.params.id });

    // Delete all student's quick test records permanently
    await QuickTest.deleteMany({ studentId: req.params.id });

    // Delete student's settings permanently
    await StudentSettings.deleteMany({ studentId: req.params.id });

    // Delete user account
    await User.findByIdAndDelete(student.userId);

    // Delete student record
    await Student.findByIdAndDelete(req.params.id);

    res.json({ message: 'Student permanently deleted' });
  } catch (error) {
    console.error('Permanent delete student error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Delete all classes for a student
router.delete('/students/:id/classes/all', adminAuth, async (req, res) => {
  try {
    const studentId = req.params.id;

    // Verify student exists
    const student = await Student.findById(studentId);
    if (!student) {
      return res.status(404).json({
        success: false,
        message: 'Student not found'
      });
    }

    // Find all classes for this student
    const studentClasses = await Class.find({ studentId: studentId });
    const classCount = studentClasses.length;

    console.log(`🗑️ Deleting ${classCount} classes for student ${student.studentName}`);

    // Remove student from any classes they're enrolled in (group classes)
    if (Class && typeof Class.updateMany === 'function') {
      await Class.updateMany(
        { students: studentId },
        {
          $pull: { students: studentId },
          $push: {
            cancelledStudents: {
              studentId: studentId,
              reason: 'All classes deleted by admin',
              cancelledAt: new Date()
            }
          }
        }
      );
    }

    // Delete all classes where this student is the primary student
    await Class.deleteMany({ studentId: studentId });

    // Remove these classes from associated teachers' schedules
    // (Classes are already deleted, so teachers won't see them)

    console.log(`✅ Successfully deleted ${classCount} classes for student ${student.studentName}`);

    res.json({
      success: true,
      message: `Successfully deleted ${classCount} classes for ${student.studentName}`,
      deletedCount: classCount
    });
  } catch (error) {
    console.error('Delete all student classes error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete classes',
      error: error.message
    });
  }
});

// Pause/Unpause teacher
router.put('/teachers/:id/toggle-status', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findById(req.params.id);
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    const newStatus = !teacher.isActive;
    
    // Update teacher status
    await Teacher.findByIdAndUpdate(req.params.id, { isActive: newStatus });
    
    // If pausing teacher, cancel all their scheduled classes
    if (!newStatus) {
      // Cancel all future classes taught by this teacher
      if (Class && typeof Class.updateMany === 'function') {
        await Class.updateMany(
          { 
            teacherId: req.params.id,
            date: { $gte: new Date() }
          },
          { 
            status: 'cancelled',
            cancelReason: 'Teacher paused',
            cancelledAt: new Date()
          }
        );
      }
    }
    
    res.json({ 
      message: `Teacher ${newStatus ? 'activated' : 'paused'} successfully`,
      isActive: newStatus 
    });
  } catch (error) {
    console.error('Toggle teacher status error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Permanently delete teacher
router.delete('/teachers/:id/permanent', adminAuth, async (req, res) => {
  try {
    const teacher = await Teacher.findById(req.params.id);
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Cancel all classes taught by this teacher
    if (Class && typeof Class.updateMany === 'function') {
      await Class.updateMany(
        { teacherId: req.params.id },
        {
          status: 'cancelled',
          cancelReason: 'Teacher deleted',
          cancelledAt: new Date()
        }
      );
    }

    // Delete all teacher's classes permanently
    await Class.deleteMany({ teacherId: req.params.id });

    // Delete all teacher's homework assignments permanently
    await Homework.deleteMany({ teacherId: req.params.id });

    // Delete all teacher's assignments permanently
    await Assignment.deleteMany({ teacherId: req.params.id });

    // Delete all teacher's exams permanently
    await Exam.deleteMany({ teacherId: req.params.id });

    // Delete all teacher's quick tests permanently
    await QuickTest.deleteMany({ teacherId: req.params.id });

    // Delete teacher's settings permanently
    await TeacherSettings.deleteMany({ teacherId: req.params.id });

    // Delete user account
    await User.findByIdAndDelete(teacher.userId);

    // Delete teacher record
    await Teacher.findByIdAndDelete(req.params.id);

    res.json({ message: 'Teacher permanently deleted' });
  } catch (error) {
    console.error('Permanent delete teacher error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/profile/picture', adminAuth, upload.single('profilePicture'), async (req, res) => {
  try {
    const userId = req.user._id;
    
    if (!req.file) {
      return res.status(400).json({ message: 'No file uploaded' });
    }

    // Handle mock user (bypass database)
    if (userId === 'admin123') {
      return res.json({
        message: 'Profile picture updated successfully',
        profilePicture: req.file.filename,
        user: {
          id: 'admin123',
          username: 'admin@edumetrix.uk',
          userType: 'admin',
          profile: {
            name: 'System Administrator',
            email: 'admin@edumetrix.uk',
            position: 'System Administrator',
            department: 'Administration',
            profilePicture: req.file.filename
          }
        }
      });
    }

    const user = await User.findByIdAndUpdate(
      userId,
      { 'profile.profilePicture': req.file.filename },
      { new: true }
    );

    if (!user) {
      return res.status(404).json({ message: 'User not found' });
    }

    res.json({
      message: 'Profile picture updated successfully',
      profilePicture: req.file.filename,
      user: {
        id: user._id,
        username: user.username,
        userType: user.userType,
        profile: user.profile
      }
    });
  } catch (error) {
    console.error('Update profile picture error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/syllabus-options/:country', adminAuth, (req, res) => {
  try {
    const options = getSyllabusOptions(req.params.country);
    res.json({ syllabusOptions: options });
  } catch (error) {
    res.status(500).json({ message: 'Server error' });
  }
});

// Class Scheduling API endpoints

// Schedule a class (single or recurring)
router.post('/schedule-class', adminAuth, async (req, res) => {
  try {
    console.log('Schedule class request received:', {
      body: req.body,
      user: req.user,
      userId: req.user?._id || req.user?.id || req.user?.userId,
      userIdType: typeof (req.user?._id || req.user?.id || req.user?.userId),
      headers: req.headers
    });

    const {
      studentId,
      teacherId,
      subject,
      scheduledDate,
      startTime,
      endTime,
      duration,
      timezone,
      googleMeetLink,
      classColor,
      recurringSchedule
    } = req.body;

    // Validate required fields
    if (!studentId || !teacherId || !subject || !scheduledDate || !googleMeetLink || !startTime || !endTime) {
      return res.status(400).json({ 
        message: 'Missing required fields',
        missing: {
          studentId: !studentId,
          teacherId: !teacherId,
          subject: !subject,
          scheduledDate: !scheduledDate,
          googleMeetLink: !googleMeetLink,
          startTime: !startTime,
          endTime: !endTime
        }
      });
    }

    // Verify student and teacher exist
    const [student, teacher] = await Promise.all([
      Student.findById(studentId),
      Teacher.findById(teacherId)
    ]);

    console.log('Found student:', {
      id: student?._id,
      name: student?.studentName,
      subjects: student?.subjects,
      subjectsType: typeof student?.subjects
    });

    console.log('Found teacher:', {
      id: teacher?._id,
      name: teacher?.name,
      subjects: teacher?.subjects,
      subjectsType: typeof teacher?.subjects
    });

    if (!student) {
      return res.status(404).json({ message: 'Student not found' });
    }

    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Verify teacher teaches the subject (with safe checking)
    const teacherSubjects = Array.isArray(teacher.subjects) ? teacher.subjects : [];
    if (!teacherSubjects.includes(subject)) {
      return res.status(400).json({
        message: `Teacher ${teacher.name} does not teach ${subject}. Teacher subjects: ${teacherSubjects.join(', ')}`
      });
    }

    // Verify student studies the subject (with safe checking)
    const studentSubjects = Array.isArray(student.subjects) ? student.subjects : [];
    if (!studentSubjects.includes(subject)) {
      return res.status(400).json({
        message: `Student ${student.studentName} does not study ${subject}. Student subjects: ${studentSubjects.join(', ')}`
      });
    }

    // Check teacher availability
    const classDate = new Date(scheduledDate);
    const classDayName = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'][classDate.getDay()];

    // Check if teacher is available on this day
    const teacherAvailableDays = Array.isArray(teacher.availableDays) ? teacher.availableDays : [];
    if (!teacherAvailableDays.includes(classDayName)) {
      return res.status(400).json({
        message: `Teacher ${teacher.name} is not available on ${classDayName}. Available days: ${teacherAvailableDays.join(', ')}`,
        availabilityError: true,
        teacherAvailability: {
          availableDays: teacherAvailableDays,
          availableTime: teacher.availableTime
        }
      });
    }

    // Check if the class time is within teacher's available hours
    const teacherStartTime = teacher.availableTime?.startTime || '00:00';
    const teacherEndTime = teacher.availableTime?.endTime || '23:59';

    // Helper function to convert time string to minutes for comparison
    const timeToMinutes = (timeStr) => {
      const [hours, minutes] = timeStr.split(':').map(Number);
      return hours * 60 + minutes;
    };

    const classStartMinutes = timeToMinutes(startTime);
    const classEndMinutes = timeToMinutes(endTime);
    const teacherStartMinutes = timeToMinutes(teacherStartTime);
    const teacherEndMinutes = timeToMinutes(teacherEndTime);

    // Check if teacher's shift crosses midnight (e.g., 14:00 - 00:30)
    const teacherShiftCrossesMidnight = teacherEndMinutes < teacherStartMinutes;

    let isTimeAvailable = false;

    if (teacherShiftCrossesMidnight) {
      // Teacher is available from startTime to 23:59 OR from 00:00 to endTime
      // Example: 14:00 - 00:30 means available 14:00-23:59 and 00:00-00:30

      // Class is available if:
      // 1. Class starts and ends in the evening period (>= startTime)
      // 2. Class starts and ends in the morning period (<= endTime)

      const classInEveningPeriod = classStartMinutes >= teacherStartMinutes && classEndMinutes >= teacherStartMinutes;
      const classInMorningPeriod = classStartMinutes <= teacherEndMinutes && classEndMinutes <= teacherEndMinutes;

      isTimeAvailable = classInEveningPeriod || classInMorningPeriod;
    } else {
      // Normal shift (doesn't cross midnight)
      isTimeAvailable = classStartMinutes >= teacherStartMinutes && classEndMinutes <= teacherEndMinutes;
    }

    if (!isTimeAvailable) {
      return res.status(400).json({
        message: `Teacher ${teacher.name} is not available at this time (${startTime} - ${endTime}). Available time: ${teacherStartTime} - ${teacherEndTime}`,
        availabilityError: true,
        teacherAvailability: {
          availableDays: teacherAvailableDays,
          availableTime: teacher.availableTime
        }
      });
    }

    // Check if teacher already has a class scheduled at this time
    const classStartTime = new Date(scheduledDate);
    const classEndTime = new Date(classStartTime.getTime() + (duration || 60) * 60000);

    const conflictingClass = await Class.findOne({
      teacherId: teacherId,
      isDeleted: false,
      status: { $in: ['scheduled', 'ongoing'] },
      $or: [
        // New class starts during existing class
        {
          scheduledDate: { $lte: classStartTime },
          $expr: {
            $gte: [
              { $add: ['$scheduledDate', { $multiply: ['$duration', 60000] }] },
              classStartTime
            ]
          }
        },
        // New class ends during existing class
        {
          scheduledDate: { $lte: classEndTime },
          $expr: {
            $gte: [
              { $add: ['$scheduledDate', { $multiply: ['$duration', 60000] }] },
              classEndTime
            ]
          }
        },
        // New class completely overlaps existing class
        {
          scheduledDate: { $gte: classStartTime, $lte: classEndTime }
        }
      ]
    }).populate('studentId', 'studentName');

    if (conflictingClass) {
      return res.status(400).json({
        message: `Teacher ${teacher.name} already has a class scheduled at this time with ${conflictingClass.studentId?.studentName || 'another student'}`,
        availabilityError: true,
        conflict: {
          subject: conflictingClass.subject,
          student: conflictingClass.studentId?.studentName,
          scheduledDate: conflictingClass.scheduledDate,
          startTime: conflictingClass.startTime,
          endTime: conflictingClass.endTime
        }
      });
    }

    const classData = {
      studentId,
      teacherId,
      subject,
      scheduledDate: new Date(scheduledDate),
      startTime,
      endTime,
      duration: duration || 60,
      timezone: timezone || 'Asia/Dubai',
      googleMeetLink,
      classColor: classColor || '#1976d2',
      recurringSchedule: recurringSchedule || { isRecurring: false },
      createdBy: (() => {
        const userId = req.user._id || req.user.id || req.user.userId;
        // Check if userId is a valid ObjectId format
        if (userId && mongoose.Types.ObjectId.isValid(userId)) {
          return userId;
        }
        // For development/testing, create a valid ObjectId or skip this field
        console.warn('Invalid ObjectId for createdBy:', userId, 'Skipping field...');
        return null;
      })()
    };

    console.log('Creating class with data:', classData);

    let createdClasses = [];

    if (!recurringSchedule.isRecurring) {
      // Create single class
      const newClass = new Class(classData);
      await newClass.save();
      console.log('🎓 Single class created:', {
        id: newClass._id,
        subject: newClass.subject,
        scheduledDate: newClass.scheduledDate,
        student: newClass.studentId,
        teacher: newClass.teacherId
      });
      createdClasses = [newClass];
    } else {
      // Create recurring classes using the static method
      createdClasses = await Class.createRecurringClasses(classData, { maxClasses: 100 });
      console.log('🎓 Recurring classes created:', createdClasses.length);
    }

    // Assign student to teacher if not already assigned
    const isStudentAssigned = teacher.assignedStudents.some(
      assigned => assigned.studentId && assigned.studentId.toString() === studentId.toString()
    );

    if (!isStudentAssigned) {
      await Teacher.findByIdAndUpdate(
        teacherId,
        {
          $push: {
            assignedStudents: {
              studentId: studentId,
              subjects: [subject],
              assignedDate: new Date()
            }
          }
        },
        { runValidators: false }
      );
      console.log('✅ Assigned student to teacher:', {
        teacher: teacher.name,
        student: student.studentName,
        subject: subject
      });
    } else {
      // Student is already assigned, check if subject needs to be added
      const assignedStudent = teacher.assignedStudents.find(
        assigned => assigned.studentId && assigned.studentId.toString() === studentId.toString()
      );
      if (assignedStudent && Array.isArray(assignedStudent.subjects) && !assignedStudent.subjects.includes(subject)) {
        await Teacher.findOneAndUpdate(
          {
            _id: teacherId,
            'assignedStudents.studentId': studentId
          },
          {
            $addToSet: {
              'assignedStudents.$.subjects': subject
            }
          },
          { runValidators: false }
        );
        console.log('✅ Added subject to existing student-teacher assignment:', {
          teacher: teacher.name,
          student: student.studentName,
          subject: subject
        });
      }
    }

    // Assign teacher to student if not already assigned
    const isTeacherAssigned = student.assignedTeachers.some(
      assigned => assigned.teacherId && assigned.teacherId.toString() === teacherId.toString()
    );

    if (!isTeacherAssigned) {
      await Student.findByIdAndUpdate(
        studentId,
        {
          $push: {
            assignedTeachers: {
              teacherId: teacherId,
              subjects: [subject],
              assignedDate: new Date()
            }
          }
        },
        { runValidators: false }
      );
      console.log('✅ Assigned teacher to student:', {
        student: student.studentName,
        teacher: teacher.name,
        subject: subject
      });
    } else {
      // Teacher is already assigned, check if subject needs to be added
      const assignedTeacher = student.assignedTeachers.find(
        assigned => assigned.teacherId && assigned.teacherId.toString() === teacherId.toString()
      );
      console.log('student assignedTeacher:', assignedTeacher)
      if (assignedTeacher && !assignedTeacher?.subjects?.includes(subject)) {
        await Student.findOneAndUpdate(
          {
            _id: studentId,
            'assignedTeachers.teacherId': teacherId
          },
          {
            $addToSet: {
              'assignedTeachers.$.subjects': subject
            }
          },
          { runValidators: false }
        );
        console.log('✅ Added subject to existing teacher-student assignment:', {
          student: student.studentName,
          teacher: teacher.name,
          subject: subject
        });
      }
    }

    // Populate the created classes with student and teacher details
    const populatedClasses = await Class.find({
      _id: { $in: createdClasses.map(c => c._id) }
    })
    .populate('studentId', 'studentName email country class')
    .populate('teacherId', 'name email subjects')
    .sort({ scheduledDate: 1 });

    // Send notifications to both student and teacher for each scheduled class
    try {
      const io = req.app.get('io'); // Get Socket.IO instance

      for (const classItem of populatedClasses) {
        const classDate = moment(classItem.scheduledDate).format('MMMM DD, YYYY');
        const classTime = classItem.startTime;
        const notificationTemplate = NotificationTemplates.classScheduled(
          classItem.subject,
          classDate,
          classTime
        );

        // Notify student
        if (classItem.studentId && classItem.studentId._id) {
          await createNotification({
            userId: classItem.studentId._id,
            userType: 'student',
            ...notificationTemplate,
            relatedId: classItem._id,
            relatedModel: 'Class',
            actionUrl: `/student/classes`,
            io
          });
        }

        // Notify teacher
        if (classItem.teacherId && classItem.teacherId._id) {
          await createNotification({
            userId: classItem.teacherId._id,
            userType: 'teacher',
            ...notificationTemplate,
            relatedId: classItem._id,
            relatedModel: 'Class',
            actionUrl: `/teacher/classes`,
            io
          });
        }
      }

      console.log(`📢 Sent notifications for ${populatedClasses.length} scheduled class(es)`);
    } catch (notificationError) {
      console.error('Failed to send class scheduling notifications:', notificationError);
      // Don't fail the request if notifications fail
    }

    res.status(201).json({
      message: `Successfully scheduled ${populatedClasses.length} class${populatedClasses.length > 1 ? 'es' : ''}`,
      count: populatedClasses.length,
      classes: populatedClasses
    });

  } catch (error) {
    console.error('Schedule class error:', error);
    console.error('Error details:', {
      message: error.message,
      stack: error.stack,
      reqBody: req.body,
      userId: req.user?._id || req.user?.id || req.user?.userId
    });
    res.status(500).json({ 
      message: 'Server error: ' + error.message, 
      error: process.env.NODE_ENV === 'development' ? error.message : undefined 
    });
  }
});

// Get all scheduled classes
router.get('/classes', adminAuth, async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 50, 
      studentId, 
      teacherId, 
      subject, 
      status = 'all',
      startDate,
      endDate
    } = req.query;

    // Build query
    const query = {
      isDeleted: false
    };

    if (studentId) query.studentId = studentId;
    if (teacherId) query.teacherId = teacherId;
    if (subject) query.subject = subject;
    if (status !== 'all') query.status = status;

    // Date range filter
    if (startDate || endDate) {
      query.scheduledDate = {};
      if (startDate) query.scheduledDate.$gte = new Date(startDate);
      if (endDate) query.scheduledDate.$lte = new Date(endDate);
    }

    const classes = await Class.find(query)
      .populate('studentId', 'studentName email parentName country class timezone syllabus')
      .populate('teacherId', 'name email subjects')
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .sort({ scheduledDate: 1 });

    const total = await Class.countDocuments(query);

    res.json({
      classes,
      totalPages: Math.ceil(total / limit),
      currentPage: parseInt(page),
      total
    });

  } catch (error) {
    console.error('Get classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get classes for calendar view
router.get('/classes/calendar', async (req, res) => {
  try {
    const { startDate, endDate, timezone = 'Asia/Kolkata' } = req.query;

    console.log('📅 Calendar API called with:', {
      startDate,
      endDate,
      timezone,
      user: req.user?._id
    });

    if (!startDate || !endDate) {
      return res.status(400).json({ 
        message: 'startDate and endDate are required' 
      });
    }

    const query = {
      scheduledDate: {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      },
      isDeleted: false
    };

    console.log('📅 Database query:', query);
    console.log('📅 Date range:', {
      start: new Date(startDate),
      end: new Date(endDate)
    });

    const classes = await Class.find(query)
    .populate('studentId', 'studentName email country class')
    .populate('teacherId', 'name email')
    .sort({ scheduledDate: 1 });

    console.log('📅 Found classes:', classes.length);
    console.log('📅 Class details:', classes.map(c => ({
      id: c._id,
      subject: c.subject,
      scheduledDate: c.scheduledDate,
      student: c.studentId?.studentName,
      teacher: c.teacherId?.name
    })));

    // Also check total classes in database for debugging
    const totalClasses = await Class.countDocuments({ isDeleted: false });
    console.log('📅 Total classes in database:', totalClasses);

    if (classes.length === 0 && totalClasses > 0) {
      // Get a few sample classes to see the date formats
      const sampleClasses = await Class.find({ isDeleted: false }).limit(3);
      console.log('📅 Sample classes with dates:', sampleClasses.map(c => ({
        id: c._id,
        scheduledDate: c.scheduledDate,
        dateString: c.scheduledDate.toISOString()
      })));
    }

    // Format classes for calendar display
    const calendarClasses = classes.map(cls => ({
      id: cls._id.toString(), // Convert ObjectId to string
      _id: cls._id.toString(), // Also include _id as string
      title: `${cls.subject} - ${cls.studentId?.studentName || 'Unknown Student'}`,
      subject: cls.subject,
      date: cls.scheduledDate, // Add date field for frontend compatibility
      time: cls.startTime, // Add time field for frontend compatibility  
      start: cls.scheduledDate,
      end: new Date(cls.scheduledDate.getTime() + (cls.duration * 60000)),
      duration: cls.duration,
      backgroundColor: cls.classColor,
      borderColor: cls.classColor,
      status: cls.status || 'scheduled',
      meetingLink: cls.googleMeetLink,
      student: cls.studentId,
      teacher: cls.teacherId,
      extendedProps: {
        student: cls.studentId,
        teacher: cls.teacherId,
        subject: cls.subject,
        duration: cls.duration,
        googleMeetLink: cls.googleMeetLink,
        status: cls.status,
        timezone: cls.timezone,
        startTime: cls.startTime,
        endTime: cls.endTime
      }
    }));

    console.log('📅 Formatted calendar classes:', calendarClasses.length);
    console.log('📅 Sample formatted class:', calendarClasses[0]);

    res.json({ classes: calendarClasses });

  } catch (error) {
    console.error('Get calendar classes error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Update class
router.put('/classes/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    console.log('📝 Class update request:', {
      classId: id,
      updates: updates,
      updateKeys: Object.keys(updates)
    });

    // Remove fields that shouldn't be updated directly
    delete updates._id;
    delete updates.createdAt;
    delete updates.updatedAt;

    // Validate required fields if they're being updated
    if (updates.scheduledDate && !updates.scheduledDate) {
      return res.status(400).json({ message: 'Invalid scheduledDate format' });
    }

    if (updates.duration && (isNaN(updates.duration) || updates.duration < 15)) {
      return res.status(400).json({ message: 'Duration must be a number >= 15 minutes' });
    }

    const updatedClass = await Class.findByIdAndUpdate(
      id,
      { ...updates, updatedAt: new Date() },
      { new: true, runValidators: true }
    )
    .populate('studentId', 'studentName email country class')
    .populate('teacherId', 'name email subjects');

    if (!updatedClass) {
      return res.status(404).json({ message: 'Class not found' });
    }

    console.log('✅ Class updated successfully:', {
      id: updatedClass._id,
      scheduledDate: updatedClass.scheduledDate,
      startTime: updatedClass.startTime,
      endTime: updatedClass.endTime,
      duration: updatedClass.duration,
      timezone: updatedClass.timezone
    });

    // Send notifications to student and teacher if schedule changed
    if (updates.scheduledDate || updates.startTime || updates.endTime) {
      try {
        const io = req.app.get('io');
        const { createNotification, NotificationTemplates } = await import('../utils/notificationHelper.js');

        const newDate = moment(updatedClass.scheduledDate).format('MMMM DD, YYYY');
        const newTime = updatedClass.startTime;
        const notificationTemplate = NotificationTemplates.classRescheduled(
          updatedClass.subject,
          newDate,
          newTime
        );

        // Notify student
        if (updatedClass.studentId && updatedClass.studentId._id) {
          await createNotification({
            userId: updatedClass.studentId._id,
            userType: 'student',
            type: notificationTemplate.type,
            title: notificationTemplate.title,
            message: notificationTemplate.message,
            relatedId: updatedClass._id,
            relatedModel: 'Class',
            priority: notificationTemplate.priority,
            actionUrl: '/student/classes',
            io
          });
          console.log(`📢 Sent reschedule notification to student ${updatedClass.studentId._id}`);
        }

        // Notify teacher
        if (updatedClass.teacherId && updatedClass.teacherId._id) {
          await createNotification({
            userId: updatedClass.teacherId._id,
            userType: 'teacher',
            type: notificationTemplate.type,
            title: notificationTemplate.title,
            message: notificationTemplate.message,
            relatedId: updatedClass._id,
            relatedModel: 'Class',
            priority: notificationTemplate.priority,
            actionUrl: '/teacher/classes',
            io
          });
          console.log(`📢 Sent reschedule notification to teacher ${updatedClass.teacherId._id}`);
        }
      } catch (notificationError) {
        console.error('Failed to send class reschedule notifications:', notificationError);
        // Don't fail the request if notifications fail
      }
    }

    res.json({
      message: 'Class updated successfully',
      class: updatedClass,
      debug: {
        originalUpdates: req.body,
        finalScheduledDate: updatedClass.scheduledDate,
        finalStartTime: updatedClass.startTime,
        finalEndTime: updatedClass.endTime
      }
    });

  } catch (error) {
    console.error('Update class error:', {
      error: error.message,
      stack: error.stack,
      classId: req.params.id,
      updates: req.body
    });
    
    // Send more specific error message
    if (error.name === 'ValidationError') {
      return res.status(400).json({ 
        message: 'Validation error: ' + Object.values(error.errors).map(e => e.message).join(', ')
      });
    }
    
    if (error.name === 'CastError') {
      return res.status(400).json({ message: 'Invalid data format: ' + error.message });
    }
    
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Delete class
router.delete('/classes/:id', adminAuth, async (req, res) => {
  try {
    const { id } = req.params;
    console.log('🗑️ Delete class request:', { classId: id, userId: req.user?._id });

    // Validate ObjectId format
    if (!mongoose.Types.ObjectId.isValid(id)) {
      console.log('❌ Invalid class ID format:', id);
      return res.status(400).json({ message: 'Invalid class ID format' });
    }

    // First check if the class exists
    const existingClass = await Class.findById(id);
    if (!existingClass) {
      console.log('❌ Class not found:', id);
      return res.status(404).json({ message: 'Class not found' });
    }

    console.log('📋 Found class to delete:', {
      id: existingClass._id,
      subject: existingClass.subject,
      scheduledDate: existingClass.scheduledDate
    });

    // Populate student and teacher details before deletion for notification
    const classToDelete = await Class.findById(id)
      .populate('studentId', 'studentName email')
      .populate('teacherId', 'name email');

    // Permanently delete the class from database
    const deletedClass = await Class.findByIdAndDelete(id);

    if (!deletedClass) {
      console.log('❌ Class not found for deletion:', id);
      return res.status(404).json({ message: 'Class not found' });
    }

    console.log('✅ Class permanently deleted from database:', {
      classId: id,
      subject: deletedClass.subject,
      scheduledDate: deletedClass.scheduledDate
    });

    // Send notifications to both student and teacher about class cancellation
    try {
      const io = req.app.get('io'); // Get Socket.IO instance

      const classDate = moment(deletedClass.scheduledDate).format('MMMM DD, YYYY');
      const classTime = deletedClass.startTime;
      const notificationTemplate = NotificationTemplates.classCancelled(
        deletedClass.subject
      );

      // Notify student
      if (classToDelete.studentId && classToDelete.studentId._id) {
        await createNotification({
          userId: classToDelete.studentId._id,
          userType: 'student',
          ...notificationTemplate,
          message: `Your ${deletedClass.subject} class scheduled for ${classDate} at ${classTime} has been cancelled.`,
          relatedId: deletedClass._id,
          relatedModel: 'Class',
          actionUrl: `/student/classes`,
          io
        });
      }

      // Notify teacher
      if (classToDelete.teacherId && classToDelete.teacherId._id) {
        await createNotification({
          userId: classToDelete.teacherId._id,
          userType: 'teacher',
          ...notificationTemplate,
          message: `Your ${deletedClass.subject} class scheduled for ${classDate} at ${classTime} has been cancelled.`,
          relatedId: deletedClass._id,
          relatedModel: 'Class',
          actionUrl: `/teacher/classes`,
          io
        });
      }

      console.log('📢 Sent class cancellation notifications to student and teacher');
    } catch (notificationError) {
      console.error('Failed to send class cancellation notifications:', notificationError);
      // Don't fail the request if notifications fail
    }

    res.json({ message: 'Class deleted successfully' });

  } catch (error) {
    console.error('❌ Delete class error:', error);
    console.error('Error details:', error.message);
    res.status(500).json({ message: 'Server error: ' + error.message });
  }
});

// Cancel class
router.put('/classes/:id/cancel', async (req, res) => {
  try {
    const { id } = req.params;
    const { reason } = req.body;

    const cancelledClass = await Class.findByIdAndUpdate(
      id,
      { 
        status: 'cancelled',
        notes: reason || 'Cancelled by admin'
      },
      { new: true }
    )
    .populate('studentId', 'studentName email')
    .populate('teacherId', 'name email');

    if (!cancelledClass) {
      return res.status(404).json({ message: 'Class not found' });
    }

    res.json({
      message: 'Class cancelled successfully',
      class: cancelledClass
    });

  } catch (error) {
    console.error('Cancel class error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get class statistics
router.get('/classes/stats', async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    const dateFilter = {};
    if (startDate || endDate) {
      dateFilter.scheduledDate = {};
      if (startDate) dateFilter.scheduledDate.$gte = new Date(startDate);
      if (endDate) dateFilter.scheduledDate.$lte = new Date(endDate);
    }

    const [
      totalClasses,
      scheduledClasses,
      completedClasses,
      cancelledClasses,
      ongoingClasses,
      subjectStats,
      monthlyStats
    ] = await Promise.all([
      Class.countDocuments({ isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'scheduled', isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'completed', isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'cancelled', isDeleted: false, ...dateFilter }),
      Class.countDocuments({ status: 'ongoing', isDeleted: false, ...dateFilter }),
      
      // Subject-wise statistics
      Class.aggregate([
        { $match: { isDeleted: false, ...dateFilter } },
        { $group: { _id: '$subject', count: { $sum: 1 } } },
        { $sort: { count: -1 } }
      ]),
      
      // Monthly statistics
      Class.aggregate([
        { $match: { isDeleted: false, ...dateFilter } },
        {
          $group: {
            _id: {
              year: { $year: '$scheduledDate' },
              month: { $month: '$scheduledDate' }
            },
            count: { $sum: 1 }
          }
        },
        { $sort: { '_id.year': 1, '_id.month': 1 } }
      ])
    ]);

    res.json({
      summary: {
        total: totalClasses,
        scheduled: scheduledClasses,
        completed: completedClasses,
        cancelled: cancelledClasses,
        ongoing: ongoingClasses
      },
      bySubject: subjectStats,
      monthly: monthlyStats
    });

  } catch (error) {
    console.error('Get class stats error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Reschedule request management routes

// router.get('/reschedule-requests', adminAuth, async (req, res) => {
//   try {
//     const { status, page = 1, limit = 10, sortBy = 'createdAt', sortOrder = 'desc' } = req.query;
//     const query = {};
    
//     if (status) {
//       query.status = status;
//     }

//     const skip = (parseInt(page) - 1) * parseInt(limit);
//     const sort = { [sortBy]: sortOrder === 'desc' ? -1 : 1 };
    
//     // const requests = await RescheduleRequest
//     //   .find(query)
//     //   .sort(sort)
//     //   .skip(skip)
//     //   .limit(parseInt(limit))
//     //   .populate('studentId', 'name email profile')
//     //   .populate('processedBy', 'name email');

//     const requests = await RescheduleRequest.find(query)
//   .sort(sort)
//   .skip(skip)
//   .limit(parseInt(limit));

// const populatedRequests = await Promise.all(
//   requests.map(async (req) => {
//     if (mongoose.Types.ObjectId.isValid(req.studentId)) {
//       await req.populate('studentId', 'name email profile');
//     }
//     if (mongoose.Types.ObjectId.isValid(req.processedBy)) {
//       await req.populate('processedBy', 'name email');
//     }
//     return req;
//   })
// );


//     const total = await RescheduleRequest.countDocuments(query);
    
//     const statusCounts = await RescheduleRequest.aggregate([
//       { $group: { _id: '$status', count: { $sum: 1 } } }
//     ]);

//     res.json({
//       requests,
//       pagination: {
//         total,
//         page: parseInt(page),
//         limit: parseInt(limit),
//         pages: Math.ceil(total / parseInt(limit))
//       },
//       statusCounts: statusCounts.reduce((acc, item) => {
//         acc[item._id] = item.count;
//         return acc;
//       }, {})
//     });
//   } catch (error) {
//     console.error('Get reschedule requests error:', error);
//     res.status(500).json({ message: 'Failed to fetch reschedule requests' });
//   }
// });

router.get('/reschedule-requests', adminAuth, async (req, res) => {
  try {
    const { status, page = 1, limit = 10, sortBy = 'createdAt', sortOrder = 'desc' } = req.query;
    const query = {};
    
    if (status) {
      query.status = status;
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);
    const sort = { [sortBy]: sortOrder === 'desc' ? -1 : 1 };
    
    // Fetch requests
    const requests = await RescheduleRequest.find(query)
      .sort(sort)
      .skip(skip)
      .limit(parseInt(limit));

    // Hybrid populate: only when IDs are valid ObjectIds
    const populatedRequests = await Promise.all(
      requests.map(async (req) => {
        const reqObj = req.toObject();

        if (mongoose.Types.ObjectId.isValid(req.studentId)) {
          try {
            await req.populate('studentId', 'studentName email');
            console.log('✅ Populated studentId for request:', req._id, 'Student:', req.studentId?.studentName);
          } catch (error) {
            console.error('❌ Failed to populate studentId:', error);
          }
        } else {
          console.log('⚠️ studentId is not a valid ObjectId:', req.studentId);
        }

        if (mongoose.Types.ObjectId.isValid(req.processedBy)) {
          try {
            await req.populate('processedBy', 'username email');
          } catch (error) {
            console.error('❌ Failed to populate processedBy:', error);
          }
        }
        return req;
      })
    );

    // Count total for pagination
    const total = await RescheduleRequest.countDocuments(query);
    
    // Aggregate status counts
    const statusCounts = await RescheduleRequest.aggregate([
      { $group: { _id: '$status', count: { $sum: 1 } } }
    ]);

    res.json({
      requests: populatedRequests, // ✅ return the populated version
      pagination: {
        total,
        page: parseInt(page),
        limit: parseInt(limit),
        pages: Math.ceil(total / parseInt(limit))
      },
      statusCounts: statusCounts.reduce((acc, item) => {
        acc[item._id] = item.count;
        return acc;
      }, {})
    });
  } catch (error) {
    console.error('Get reschedule requests error:', error);
    res.status(500).json({ message: 'Failed to fetch reschedule requests' });
  }
});


router.get('/reschedule-requests/:id', adminAuth, async (req, res) => {
  try {
    const request = await RescheduleRequest
      .findById(req.params.id)
      .populate('studentId', 'name email profile')
      .populate('processedBy', 'name email');

    if (!request) {
      return res.status(404).json({ message: 'Reschedule request not found' });
    }

    res.json(request);
  } catch (error) {
    console.error('Get reschedule request error:', error);
    res.status(500).json({ message: 'Failed to fetch reschedule request' });
  }
});

// router.put('/reschedule-requests/:id/status', adminAuth, async (req, res) => {
//   try {
//     const { status, adminNotes } = req.body;
    
//     if (!['approved', 'rejected'].includes(status)) {
//       return res.status(400).json({ message: 'Invalid status' });
//     }

//     const request = await RescheduleRequest.findById(req.params.id);
    
//     if (!request) {
//       return res.status(404).json({ message: 'Reschedule request not found' });
//     }

//     request.status = status;
//     request.adminNotes = adminNotes || '';
//     request.processedBy = req.user._id;
//     request.processedAt = new Date();

//     await request.save();

//     const updatedRequest = await RescheduleRequest
//       .findById(req.params.id)
//       .populate('studentId', 'studentName email ')
//       .populate('processedBy', 'username profile');

//     res.json({
//       message: `Reschedule request ${status} successfully`,
//       request: updatedRequest
//     });
//   } catch (error) {
//     console.error('Update reschedule request status error:', error);
//     res.status(500).json({ message: 'Failed to update reschedule request' });
//   }
// });

router.put('/reschedule-requests/:id/status', adminAuth, async (req, res) => {
  try {
    const { status, adminNotes } = req.body;
    
    if (!['approved', 'rejected'].includes(status)) {
      return res.status(400).json({ message: 'Invalid status' });
    }

    const request = await RescheduleRequest.findById(req.params.id);
    if (!request) {
      return res.status(404).json({ message: 'Reschedule request not found' });
    }

    request.status = status;
    request.adminNotes = adminNotes || '';

    // ✅ Handle processedBy as ObjectId or string
    if (mongoose.Types.ObjectId.isValid(req.user._id)) {
      request.processedBy = req.user._id; // real MongoDB user
    } else {
      request.processedBy = req.user._id.toString(); // dev fallback
    }

    request.processedAt = new Date();
    await request.save();

    let updatedRequest = await RescheduleRequest.findById(req.params.id)
      .populate('studentId', 'studentName email ');

    // ✅ If processedBy is ObjectId → populate
    if (mongoose.Types.ObjectId.isValid(updatedRequest.processedBy)) {
      updatedRequest = await updatedRequest.populate('processedBy', 'username profile');
    } else {
      // ✅ If string → fake a user object so frontend won't crash
      updatedRequest = updatedRequest.toObject();
      updatedRequest.processedBy = {
        _id: request.processedBy,
        username: req.user.username || request.processedBy, // fallback
        profile: req.user.profile || null
      };
    }

    // Send notification to student about reschedule request status
    try {
      const io = req.app.get('io'); // Get Socket.IO instance

      // Get student ID (handle both ObjectId and Mixed type)
      let studentId = null;
      if (updatedRequest.studentId) {
        studentId = updatedRequest.studentId._id || updatedRequest.studentId;
      }

      if (studentId) {
        const classDate = moment(updatedRequest.classDetails.scheduledDate).format('MMMM DD, YYYY');
        const classTime = updatedRequest.classDetails.scheduledTime || 'scheduled time';
        const newDate = moment(updatedRequest.requestedDate).format('MMMM DD, YYYY');
        const newTime = updatedRequest.requestedTime;

        let notificationTemplate;
        let message;

        if (status === 'approved') {
          notificationTemplate = NotificationTemplates.classRescheduled(
            updatedRequest.classDetails.subject,
            newDate,
            newTime
          );
          message = `Your reschedule request for ${updatedRequest.classDetails.subject} has been approved.`;
        } else {
          // rejected
          notificationTemplate = {
            type: 'reschedule_rejected',
            title: 'Reschedule Request Rejected',
            priority: 'medium'
          };
          message = `Your reschedule request for ${updatedRequest.classDetails.subject} (originally ${classDate} at ${classTime}) has been rejected.${adminNotes ? ' Reason: ' + adminNotes : ''}`;
        }

        await createNotification({
          userId: studentId,
          userType: 'student',
          ...notificationTemplate,
          message: message,
          relatedId: updatedRequest._id,
          relatedModel: 'RescheduleRequest',
          actionUrl: `/student/classes`,
          io
        });

        console.log(`📢 Sent reschedule ${status} notification to student`);
      }
    } catch (notificationError) {
      console.error('Failed to send reschedule status notification:', notificationError);
      // Don't fail the request if notifications fail
    }

    res.json({
      message: `Reschedule request ${status} successfully`,
      request: updatedRequest
    });
  } catch (error) {
    console.error('Update reschedule request status error:', error);
    res.status(500).json({ message: 'Failed to update reschedule request' });
  }
});



router.delete('/reschedule-requests/:id', adminAuth, async (req, res) => {
  try {
    const request = await RescheduleRequest.findById(req.params.id);
    
    if (!request) {
      return res.status(404).json({ message: 'Reschedule request not found' });
    }

    await RescheduleRequest.findByIdAndDelete(req.params.id);

    res.json({ message: 'Reschedule request deleted successfully' });
  } catch (error) {
    console.error('Delete reschedule request error:', error);
    res.status(500).json({ message: 'Failed to delete reschedule request' });
  }
});

export default router;