import mongoose from 'mongoose';

const questionSchema = new mongoose.Schema({
  question: {
    type: String,
    required: true,
    trim: true
  },
  options: [{
    text: {
      type: String,
      required: true,
      trim: true
    },
    isCorrect: {
      type: Boolean,
      required: true,
      default: false
    }
  }],
  correctAnswer: {
    type: Number,
    required: true,
    min: 0,
    max: 3
  },
  marks: {
    type: Number,
    required: true,
    default: 1,
    min: 1
  }
});

const quickTestSchema = new mongoose.Schema({
  title: {
    type: String,
    required: true,
    trim: true
  },
  description: {
    type: String,
    trim: true
  },
  teacherId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Teacher',
    required: true
  },
  studentId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Student',
    required: true
  },
  subject: {
    type: String,
    required: true,
    trim: true
  },
  topic: {
    type: String,
    required: true,
    trim: true
  },
  questions: [questionSchema],
  duration: {
    type: Number,
    required: true,
    min: 5,
    max: 15,
    enum: [5, 10, 15]
  },
  totalMarks: {
    type: Number,
    required: true,
    min: 1
  },
  status: {
    type: String,
    enum: ['draft', 'active', 'completed', 'cancelled'],
    default: 'draft'
  },
  startTime: {
    type: Date
  },
  endTime: {
    type: Date
  },
  studentSessions: [{
    studentId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'Student',
      required: true
    },
    startedAt: {
      type: Date,
      required: true
    }
  }],
  submissions: [{
    studentId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'Student',
      required: true
    },
    studentStartTime: {
      type: Date,
      required: true
    },
    answers: [{
      questionIndex: {
        type: Number,
        required: true
      },
      selectedOption: {
        type: Number,
        required: true,
        min: 0,
        max: 3
      },
      isCorrect: {
        type: Boolean,
        required: true
      },
      marksObtained: {
        type: Number,
        required: true,
        min: 0
      }
    }],
    totalMarks: {
      type: Number,
      required: true,
      min: 0
    },
    percentage: {
      type: Number,
      required: true,
      min: 0,
      max: 100
    },
    submittedAt: {
      type: Date,
      default: Date.now
    },
    duration: {
      type: Number,
      min: 0
    }
  }],
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  }
}, {
  timestamps: true
});

quickTestSchema.pre('save', function(next) {
  if (this.questions && this.questions.length > 0) {
    this.totalMarks = this.questions.reduce((total, q) => total + q.marks, 0);
  }
  next();
});

quickTestSchema.methods.calculateScore = function(answers) {
  let totalMarks = 0;
  let obtainedMarks = 0;
  
  answers.forEach((answer, index) => {
    const question = this.questions[index];
    if (question) {
      totalMarks += question.marks;
      if (answer.selectedOption === question.correctAnswer) {
        obtainedMarks += question.marks;
      }
    }
  });
  
  return {
    totalMarks,
    obtainedMarks,
    percentage: totalMarks > 0 ? Math.round((obtainedMarks / totalMarks) * 100) : 0
  };
};

quickTestSchema.methods.isActive = function() {
  const now = new Date();
  return this.status === 'active' && 
         this.startTime && 
         this.endTime && 
         now >= this.startTime && 
         now <= this.endTime;
};

quickTestSchema.methods.canSubmit = function(studentId) {
  // Check if test is active
  if (this.status !== 'active') return false;

  // Check if this is the assigned student
  if (this.studentId.toString() !== studentId.toString()) return false;

  // Check if student already submitted
  const existingSubmission = this.submissions.find(s => s.studentId.toString() === studentId.toString());
  if (existingSubmission) return false;

  // Check if student has started and time hasn't expired
  const session = this.studentSessions.find(s => s.studentId.toString() === studentId.toString());
  if (session) {
    const now = new Date();
    const sessionEndTime = new Date(session.startedAt.getTime() + this.duration * 60000);
    return now <= sessionEndTime;
  }

  // Student can start if they haven't started yet
  return true;
};

quickTestSchema.methods.getStudentTimeRemaining = function(studentId) {
  const session = this.studentSessions.find(s => s.studentId.toString() === studentId.toString());
  if (!session) {
    // Student hasn't started yet, return full duration
    return this.duration * 60;
  }

  const now = new Date();
  const sessionEndTime = new Date(session.startedAt.getTime() + this.duration * 60000);
  const remainingSeconds = Math.max(0, Math.floor((sessionEndTime - now) / 1000));

  return remainingSeconds;
};

export default mongoose.model('QuickTest', quickTestSchema);