import mongoose from 'mongoose';

const homeworkSchema = new mongoose.Schema({
  title: {
    type: String,
    required: true,
    trim: true
  },
  description: {
    type: String,
    required: true,
    trim: true
  },
  subject: {
    type: String,
    required: true,
    trim: true
  },
  teacherId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Teacher',
    required: true
  },
  studentId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Student',
    required: true
  },
  dueDate: {
    type: Date,
    required: true
  },
  points: {
    type: Number,
    default: 10,
    min: 1,
    max: 100
  },
  instructions: {
    type: String,
    trim: true
  },
  attachment: {
    filename: String,
    originalName: String,
    path: String,
    size: Number,
    mimetype: String,
    uploadedAt: {
      type: Date,
      default: Date.now
    }
  },
  status: {
    type: String,
    enum: ['assigned', 'submitted', 'graded', 'overdue'],
    default: 'assigned'
  },
  submission: {
    content: {
      type: String,
      trim: true
    },
    attachment: {
      filename: String,
      originalName: String,
      path: String,
      size: Number,
      mimetype: String
    },
    notes: {
      type: String,
      trim: true
    },
    submittedAt: Date,
    grade: {
      type: Number,
      min: 0
    },
    feedback: {
      type: String,
      trim: true
    },
    corrections: {
      type: String,
      trim: true
    },
    gradedAt: Date,
    gradedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'Teacher'
    }
  },
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  }
});

// Pre-save middleware
homeworkSchema.pre('save', function(next) {
  this.updatedAt = Date.now();

  // Auto-update status based on submission and grading
  if (this.submission && this.submission.submittedAt) {
    if (this.submission.grade !== undefined && this.submission.grade !== null) {
      this.status = 'graded';
      if (!this.submission.gradedAt) {
        this.submission.gradedAt = new Date();
      }
    } else if (this.status === 'assigned') {
      this.status = 'submitted';
    }
  }

  // Check if overdue
  if (this.status === 'assigned' && new Date() > this.dueDate) {
    this.status = 'overdue';
  }

  next();
});

// Instance methods
homeworkSchema.methods.isOverdue = function() {
  return this.status === 'assigned' && new Date() > this.dueDate;
};

homeworkSchema.methods.canSubmit = function() {
  return this.status === 'assigned' || this.status === 'overdue';
};

homeworkSchema.methods.isSubmitted = function() {
  return this.submission && this.submission.submittedAt;
};

homeworkSchema.methods.isGraded = function() {
  return this.submission && this.submission.grade !== undefined && this.submission.grade !== null;
};

homeworkSchema.methods.submitHomework = function(submissionData) {
  this.submission = {
    content: submissionData.content,
    attachment: submissionData.attachment,
    notes: submissionData.notes,
    submittedAt: new Date()
  };
  this.status = 'submitted';
  return this.save();
};

homeworkSchema.methods.gradeHomework = function(gradeData, gradedBy) {
  if (!this.submission) {
    throw new Error('Cannot grade homework that has not been submitted');
  }

  this.submission.grade = gradeData.grade;
  this.submission.feedback = gradeData.feedback;
  this.submission.corrections = gradeData.corrections;
  this.submission.gradedAt = new Date();
  this.submission.gradedBy = gradedBy;
  this.status = 'graded';

  return this.save();
};

homeworkSchema.methods.getTimeRemaining = function() {
  if (this.status !== 'assigned') return null;

  const now = new Date();
  const timeLeft = this.dueDate - now;

  if (timeLeft <= 0) return 0;

  return Math.floor(timeLeft / 1000); // Return seconds remaining
};

homeworkSchema.methods.getStatusLabel = function() {
  if (this.isOverdue()) return 'OVERDUE';
  return this.status.toUpperCase();
};

// Static methods
homeworkSchema.statics.findByStudent = function(studentId, status = null) {
  const query = { studentId };
  if (status && status !== 'all') {
    if (status === 'pending') {
      query.status = 'assigned';
      query.dueDate = { $gte: new Date() };
    } else if (status === 'overdue') {
      query.$or = [
        { status: 'overdue' },
        { status: 'assigned', dueDate: { $lt: new Date() } }
      ];
    } else {
      query.status = status;
    }
  }
  return this.find(query).populate('teacherId', 'name').sort({ createdAt: -1 });
};

homeworkSchema.statics.findByTeacher = function(teacherId, filters = {}) {
  const query = { teacherId };

  if (filters.subject && filters.subject !== 'all') {
    query.subject = filters.subject;
  }

  if (filters.studentId && filters.studentId !== 'all') {
    query.studentId = filters.studentId;
  }

  if (filters.status) {
    const now = new Date();
    switch (filters.status) {
      case 'active':
        query.status = 'assigned';
        query.dueDate = { $gte: now };
        break;
      case 'pending_review':
        query.status = 'submitted';
        query['submission.grade'] = { $exists: false };
        break;
      case 'graded':
        query.status = 'graded';
        break;
      case 'overdue':
        query.$or = [
          { status: 'overdue' },
          { status: 'assigned', dueDate: { $lt: now } }
        ];
        break;
    }
  }

  return this.find(query)
    .populate('studentId', 'studentName name email class')
    .populate('teacherId', 'name')
    .sort({ dueDate: -1 });
};

// Indexes
homeworkSchema.index({ teacherId: 1, createdAt: -1 });
homeworkSchema.index({ studentId: 1, createdAt: -1 });
homeworkSchema.index({ status: 1, dueDate: 1 });
homeworkSchema.index({ teacherId: 1, status: 1 });
homeworkSchema.index({ studentId: 1, status: 1 });
homeworkSchema.index({ dueDate: 1 });
homeworkSchema.index({ subject: 1 });

export default mongoose.model('Homework', homeworkSchema);