import mongoose from 'mongoose';

const assignmentSchema = new mongoose.Schema({
  title: {
    type: String,
    required: true,
    trim: true
  },
  description: {
    type: String,
    required: true,
    trim: true
  },
  subject: {
    type: String,
    required: true,
    trim: true
  },
  teacherId: {
    type: String,
    required: true
  },
  assignedStudents: [{
    studentId: {
      type: String,
      required: true
    },
    studentName: {
      type: String,
      required: true
    },
    assignedAt: {
      type: Date,
      default: Date.now
    }
  }],
  dueDate: {
    type: Date,
    required: true
  },
  maxMarks: {
    type: Number,
    default: 100
  },
  attachments: [{
    filename: {
      type: String,
      required: true
    },
    originalName: {
      type: String,
      required: true
    },
    path: {
      type: String,
      required: true
    },
    size: {
      type: Number
    },
    mimetype: {
      type: String
    },
    uploadedAt: {
      type: Date,
      default: Date.now
    }
  }],
  instructions: {
    type: String,
    trim: true
  },
  status: {
    type: String,
    enum: ['draft', 'published', 'closed'],
    default: 'draft'
  },
  allowLateSubmission: {
    type: Boolean,
    default: false
  },
  lateSubmissionPenalty: {
    type: Number,
    default: 0,
    min: 0,
    max: 100
  },
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  },
  publishedAt: {
    type: Date
  }
});

assignmentSchema.pre('save', function(next) {
  this.updatedAt = Date.now();
  
  if (this.status === 'published' && !this.publishedAt) {
    this.publishedAt = new Date();
  }
  
  next();
});

assignmentSchema.methods.isOverdue = function() {
  return new Date() > this.dueDate;
};

assignmentSchema.methods.getSubmissionStats = async function() {
  const AssignmentSubmission = mongoose.model('AssignmentSubmission');
  
  const totalStudents = this.assignedStudents.length;
  const submissions = await AssignmentSubmission.find({ assignmentId: this._id });
  const submittedCount = submissions.length;
  const pendingCount = totalStudents - submittedCount;
  const gradedCount = submissions.filter(s => s.status === 'graded').length;
  
  return {
    totalStudents,
    submittedCount,
    pendingCount,
    gradedCount,
    submissionRate: totalStudents > 0 ? (submittedCount / totalStudents) * 100 : 0
  };
};

assignmentSchema.index({ teacherId: 1, createdAt: -1 });
assignmentSchema.index({ 'assignedStudents.studentId': 1 });
assignmentSchema.index({ dueDate: 1 });

export default mongoose.model('Assignment', assignmentSchema);