import mongoose from 'mongoose';

const assignmentSubmissionSchema = new mongoose.Schema({
  assignmentId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Assignment',
    required: true
  },
  studentId: {
    type: String,
    required: true
  },
  studentName: {
    type: String,
    required: true
  },
  submissionText: {
    type: String,
    trim: true
  },
  attachments: [{
    filename: {
      type: String,
      required: true
    },
    originalName: {
      type: String,
      required: true
    },
    path: {
      type: String,
      required: true
    },
    size: {
      type: Number
    },
    mimetype: {
      type: String
    },
    uploadedAt: {
      type: Date,
      default: Date.now
    }
  }],
  submittedAt: {
    type: Date,
    default: Date.now
  },
  isLateSubmission: {
    type: Boolean,
    default: false
  },
  status: {
    type: String,
    enum: ['submitted', 'under_review', 'graded', 'returned'],
    default: 'submitted'
  },
  grade: {
    type: Number,
    min: 0
  },
  feedback: {
    type: String,
    trim: true
  },
  teacherComments: [{
    comment: {
      type: String,
      required: true,
      trim: true
    },
    addedAt: {
      type: Date,
      default: Date.now
    },
    isPrivate: {
      type: Boolean,
      default: false
    }
  }],
  gradedAt: {
    type: Date
  },
  gradedBy: {
    type: String
  },
  revisionRequested: {
    type: Boolean,
    default: false
  },
  revisionDeadline: {
    type: Date
  },
  revisionComments: {
    type: String,
    trim: true
  },
  version: {
    type: Number,
    default: 1
  },
  previousVersions: [{
    submissionText: String,
    attachments: [{
      filename: String,
      originalName: String,
      path: String,
      size: Number,
      mimetype: String,
      uploadedAt: Date
    }],
    submittedAt: Date,
    version: Number
  }],
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  }
});

assignmentSubmissionSchema.pre('save', function(next) {
  this.updatedAt = Date.now();
  
  if (this.status === 'graded' && !this.gradedAt) {
    this.gradedAt = new Date();
  }
  
  next();
});

assignmentSubmissionSchema.methods.calculateLatePenalty = async function() {
  const Assignment = mongoose.model('Assignment');
  const assignment = await Assignment.findById(this.assignmentId);
  
  if (!assignment || !this.isLateSubmission || !assignment.lateSubmissionPenalty) {
    return 0;
  }
  
  return (this.grade * assignment.lateSubmissionPenalty) / 100;
};

assignmentSubmissionSchema.methods.getFinalGrade = async function() {
  if (!this.grade) return null;
  
  const penalty = await this.calculateLatePenalty();
  return Math.max(0, this.grade - penalty);
};

assignmentSubmissionSchema.methods.addComment = function(comment, isPrivate = false) {
  this.teacherComments.push({
    comment,
    isPrivate,
    addedAt: new Date()
  });
  return this.save();
};

assignmentSubmissionSchema.methods.requestRevision = function(deadline, comments) {
  this.revisionRequested = true;
  this.revisionDeadline = deadline;
  this.revisionComments = comments;
  this.status = 'returned';
  return this.save();
};

assignmentSubmissionSchema.methods.createNewVersion = function(submissionData) {
  // Save current version to history
  this.previousVersions.push({
    submissionText: this.submissionText,
    attachments: this.attachments,
    submittedAt: this.submittedAt,
    version: this.version
  });
  
  // Update with new version
  this.submissionText = submissionData.submissionText;
  this.attachments = submissionData.attachments;
  this.submittedAt = new Date();
  this.version += 1;
  this.revisionRequested = false;
  this.status = 'submitted';
  
  return this.save();
};

assignmentSubmissionSchema.index({ assignmentId: 1, studentId: 1 }, { unique: true });
assignmentSubmissionSchema.index({ studentId: 1, submittedAt: -1 });
assignmentSubmissionSchema.index({ assignmentId: 1, status: 1 });
assignmentSubmissionSchema.index({ submittedAt: -1 });

export default mongoose.model('AssignmentSubmission', assignmentSubmissionSchema);