const helmet = require('helmet');
const SecurityConfig = require('../config/security');

class EnterpriseSecurityHeaders {
  // Configure helmet with enterprise security settings
  static getHelmetConfig() {
    return helmet({
      // Content Security Policy
      contentSecurityPolicy: {
        directives: {
          defaultSrc: ["'self'"],
          styleSrc: [
            "'self'",
            "'unsafe-inline'", // Required for Material-UI
            "https://fonts.googleapis.com",
            "https://cdn.jsdelivr.net"
          ],
          fontSrc: [
            "'self'",
            "https://fonts.gstatic.com",
            "https://cdn.jsdelivr.net"
          ],
          imgSrc: [
            "'self'",
            "data:",
            "https:",
            "blob:"
          ],
          scriptSrc: [
            "'self'",
            ...(process.env.NODE_ENV === 'development' ? ["'unsafe-eval'"] : [])
          ],
          connectSrc: [
            "'self'",
            "ws:",
            "wss:",
            "http://localhost:*",
            "https://api.edumetrix.uk"
          ],
          frameSrc: ["'none'"],
          objectSrc: ["'none'"],
          baseUri: ["'self'"],
          formAction: ["'self'"],
          frameAncestors: ["'none'"],
          ...(process.env.NODE_ENV === 'production' && {
            upgradeInsecureRequests: []
          })
        },
        reportOnly: process.env.NODE_ENV === 'development'
      },

      // HTTP Strict Transport Security
      hsts: {
        maxAge: 31536000, // 1 year
        includeSubDomains: true,
        preload: true
      },

      // X-Frame-Options
      frameguard: {
        action: 'deny'
      },

      // X-Content-Type-Options
      noSniff: true,

      // X-XSS-Protection
      xssFilter: true,

      // Referrer Policy
      referrerPolicy: {
        policy: ['no-referrer', 'strict-origin-when-cross-origin']
      },

      // Hide X-Powered-By header
      hidePoweredBy: true,

      // DNS Prefetch Control
      dnsPrefetchControl: {
        allow: false
      },

      // Expect-CT header
      expectCt: {
        maxAge: 30,
        enforce: true,
        reportUri: process.env.EXPECT_CT_REPORT_URI
      },

      // Feature Policy / Permissions Policy
      permissionsPolicy: {
        features: {
          geolocation: ["'none'"],
          camera: ["'none'"],
          microphone: ["'none'"],
          payment: ["'none'"],
          usb: ["'none'"],
          magnetometer: ["'none'"],
          gyroscope: ["'none'"],
          accelerometer: ["'none'"],
          ambient_light_sensor: ["'none'"],
          autoplay: ["'none'"],
          encrypted_media: ["'none'"],
          fullscreen: ["'self'"],
          picture_in_picture: ["'none'"],
          sync_xhr: ["'none'"]
        }
      }
    });
  }

  // Custom security headers middleware
  static customSecurityHeaders() {
    return (req, res, next) => {
      // Security headers
      res.setHeader('X-Request-ID', req.id || require('crypto').randomUUID());
      res.setHeader('X-API-Version', '1.0');
      res.setHeader('X-RateLimit-Policy', 'enterprise');

      // CORP (Cross-Origin Resource Policy)
      res.setHeader('Cross-Origin-Resource-Policy', 'same-origin');

      // COEP (Cross-Origin Embedder Policy)
      if (process.env.NODE_ENV === 'production') {
        res.setHeader('Cross-Origin-Embedder-Policy', 'require-corp');
      }

      // COOP (Cross-Origin Opener Policy)
      res.setHeader('Cross-Origin-Opener-Policy', 'same-origin');

      // Clear-Site-Data (for logout endpoints)
      if (req.path.includes('/logout')) {
        res.setHeader('Clear-Site-Data', '"cache", "cookies", "storage"');
      }

      // Prevent MIME sniffing
      res.setHeader('X-Content-Type-Options', 'nosniff');

      // Cache control for sensitive endpoints
      if (req.path.includes('/api/')) {
        res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private');
        res.setHeader('Pragma', 'no-cache');
        res.setHeader('Expires', '0');
      }

      // Server information hiding
      res.removeHeader('Server');
      res.removeHeader('X-Powered-By');

      next();
    };
  }

  // CORS configuration middleware
  static configureCORS() {
    const cors = require('cors');

    return cors({
      origin: function (origin, callback) {
        const allowedOrigins = [
          'http://localhost:3000',
          'http://localhost:3001',
          'http://localhost:3002',
          'http://localhost:3003',
          'https://kaaryasthan.edumetrix.uk', 'https://student.edumetrix.uk', 'https://teacher.edumetrix.uk',

        ].filter(Boolean);

        // Allow requests with no origin (mobile apps, Postman, etc.)
        if (!origin) return callback(null, true);

        if (allowedOrigins.includes(origin)) {
          callback(null, true);
        } else {
          const error = new Error(`CORS: Origin ${origin} not allowed`);
          error.status = 403;
          callback(error);
        }
      },

      credentials: true,

      methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],

      allowedHeaders: [
        'Origin',
        'X-Requested-With',
        'Content-Type',
        'Accept',
        'Authorization',
        'X-CSRF-Token',
        'X-Request-ID',
        'X-API-Key'
      ],

      exposedHeaders: [
        'X-RateLimit-Limit',
        'X-RateLimit-Remaining',
        'X-RateLimit-Reset',
        'X-Request-ID',
        'X-Response-Time'
      ],

      maxAge: 86400, // 24 hours

      preflightContinue: false,
      optionsSuccessStatus: 204
    });
  }

  // Content-Type validation middleware
  static validateContentType() {
    return (req, res, next) => {
      if (['POST', 'PUT', 'PATCH'].includes(req.method)) {
        const contentType = req.get('Content-Type');

        if (!contentType) {
          return res.status(400).json({
            error: 'Content-Type header required'
          });
        }

        const allowedTypes = [
          'application/json',
          'multipart/form-data',
          'application/x-www-form-urlencoded'
        ];

        const isValidType = allowedTypes.some(type =>
          contentType.startsWith(type)
        );

        if (!isValidType) {
          return res.status(415).json({
            error: 'Unsupported Content-Type',
            supported: allowedTypes
          });
        }
      }

      next();
    };
  }

  // Request size limiting
  static limitRequestSize() {
    return (req, res, next) => {
      const maxSizes = {
        'application/json': 1024 * 1024, // 1MB
        'application/x-www-form-urlencoded': 1024 * 1024, // 1MB
        'multipart/form-data': 10 * 1024 * 1024 // 10MB for file uploads
      };

      const contentType = req.get('Content-Type') || '';
      const contentLength = parseInt(req.get('Content-Length') || '0');

      let maxSize = 1024 * 1024; // Default 1MB

      for (const [type, size] of Object.entries(maxSizes)) {
        if (contentType.startsWith(type)) {
          maxSize = size;
          break;
        }
      }

      if (contentLength > maxSize) {
        return res.status(413).json({
          error: 'Request entity too large',
          maxSize: maxSize,
          received: contentLength
        });
      }

      next();
    };
  }

  // Security response headers for API responses
  static apiResponseHeaders() {
    return (req, res, next) => {
      // Add response timing header for monitoring
      const start = Date.now();

      res.on('finish', () => {
        const duration = Date.now() - start;
        res.setHeader('X-Response-Time', `${duration}ms`);
      });

      // Security headers for API responses
      res.setHeader('X-Frame-Options', 'DENY');
      res.setHeader('X-XSS-Protection', '1; mode=block');
      res.setHeader('X-Content-Type-Options', 'nosniff');
      res.setHeader('Referrer-Policy', 'no-referrer');

      // Prevent caching of sensitive data
      if (req.path.includes('/api/')) {
        res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private');
        res.setHeader('Expires', '0');
      }

      next();
    };
  }

  // IP validation middleware
  static validateClientIP() {
    return (req, res, next) => {
      const clientIP = req.ip;
      const forwardedIP = req.get('X-Forwarded-For');

      // Log potential IP spoofing attempts
      if (forwardedIP && forwardedIP !== clientIP) {
        const auditLogger = require('../utils/auditLogger');
        auditLogger.warn('Potential IP spoofing detected', {
          clientIP: clientIP,
          forwardedIP: forwardedIP,
          userAgent: req.get('User-Agent'),
          endpoint: req.path
        });
      }

      // Block requests from localhost in production (except for health checks)
      if (process.env.NODE_ENV === 'production' &&
        clientIP === '127.0.0.1' &&
        !req.path.includes('/health')) {
        return res.status(403).json({
          error: 'Access denied from localhost in production'
        });
      }

      next();
    };
  }

  // Get all security middleware in order
  static getAllMiddleware() {
    return [
      this.validateClientIP(),
      this.configureCORS(),
      this.getHelmetConfig(),
      this.customSecurityHeaders(),
      this.validateContentType(),
      this.limitRequestSize(),
      this.apiResponseHeaders()
    ];
  }
}

module.exports = EnterpriseSecurityHeaders;