import helmet from 'helmet';
import rateLimit from 'express-rate-limit';
import slowDown from 'express-slow-down';
import mongoSanitize from 'express-mongo-sanitize';
import xss from 'xss';
import hpp from 'hpp';
import cors from 'cors';
import { body, validationResult } from 'express-validator';
import winston from 'winston';

// Create security logger
const securityLogger = winston.createLogger({
  level: 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.errors({ stack: true }),
    winston.format.json()
  ),
  defaultMeta: { service: 'edumetrix-security' },
  transports: [
    new winston.transports.File({ filename: 'logs/security.log' }),
    new winston.transports.File({ filename: 'logs/error.log', level: 'error' }),
    new winston.transports.Console({
      format: winston.format.simple()
    })
  ],
});

// Enhanced brute force protection with express-slow-down
const bruteForceProtection = slowDown({
  windowMs: 15 * 60 * 1000, // 15 minutes
  delayAfter: 3, // Allow 3 requests per windowMs without delay
  delayMs: (hits) => {
    return hits * 1000; // Increase delay by 1 second for each attempt
  },
  maxDelayMs: 10000, // Maximum delay of 10 seconds
  skipSuccessfulRequests: true,
  skipFailedRequests: false,
  keyGenerator: (req) => {
    return req.ip + ':' + req.path;
  }
});

// 1. HELMET - Security Headers
const helmetConfig = helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'", "https://fonts.googleapis.com"],
      fontSrc: ["'self'", "https://fonts.gstatic.com"],
      imgSrc: ["'self'", "data:", "https:", "http://localhost:3000", "http://localhost:3001", "http://localhost:3002"],
      scriptSrc: ["'self'"],
      connectSrc: ["'self'"],
      frameSrc: ["'none'"],
      objectSrc: ["'none'"],
      mediaSrc: ["'self'", "http://localhost:3000", "http://localhost:3001", "http://localhost:3002"],
      manifestSrc: ["'self'"]
    }
  },
  crossOriginEmbedderPolicy: false,
  crossOriginResourcePolicy: false, // Allow cross-origin resource access for development
  hsts: {
    maxAge: 31536000,
    includeSubDomains: true,
    preload: true
  }
});

// 2. CORS Configuration - Development Mode (Allow All Origins)
const corsOptions = {
  origin: true, // Allow all origins for development
  credentials: true,
  optionsSuccessStatus: 200
};

// 3. Rate Limiting
const createRateLimit = (windowMs, max, message) => {
  return rateLimit({
    windowMs,
    max,
    message: { error: message },
    standardHeaders: true,
    legacyHeaders: false,
    handler: (req, res) => {
      securityLogger.warn('Rate limit exceeded', {
        ip: req.ip,
        userAgent: req.get('User-Agent'),
        url: req.originalUrl,
        timestamp: new Date()
      });
      res.status(429).json({ error: message });
    },
    skip: (req) => {
      // Skip rate limiting for development tokens in development mode
      if (process.env.NODE_ENV !== 'production') {
        const authHeader = req.header('Authorization');
        if (authHeader && authHeader.includes('dev-')) {
          return true;
        }
      }
      return false;
    }
  });
};

// Different rate limits for different endpoints (Development Mode - More Permissive)
const generalLimiter = createRateLimit(15 * 60 * 1000, 1000, 'Too many requests, please try again later');
const authLimiter = createRateLimit(15 * 60 * 1000, 100, 'Too many authentication attempts, please try again later');
const otpLimiter = createRateLimit(5 * 60 * 1000, 50, 'Too many OTP requests, please try again later');
const strictLimiter = createRateLimit(15 * 60 * 1000, 500, 'Too many requests to sensitive endpoint');

// 4. Enhanced Rate Limiting for Authentication (Development Mode)
const authBruteForce = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // Limit each IP to 100 requests per windowMs for development
  message: {
    error: 'Too many authentication attempts, please try again later.'
  },
  standardHeaders: true,
  legacyHeaders: false,
  handler: (req, res) => {
    securityLogger.error('Authentication rate limit exceeded', {
      ip: req.ip,
      userAgent: req.get('User-Agent'),
      url: req.originalUrl,
      timestamp: new Date()
    });
    res.status(429).json({
      error: 'Too many authentication attempts, please try again later.'
    });
  },
  skip: (req) => {
    // Skip rate limiting for development tokens in development mode
    if (process.env.NODE_ENV !== 'production') {
      const authHeader = req.header('Authorization');
      if (authHeader && authHeader.includes('dev-')) {
        return true;
      }
    }
    return false;
  }
});

// 5. Input Validation Middleware
const validateInput = (validations) => {
  return async (req, res, next) => {
    // Run all validations
    await Promise.all(validations.map(validation => validation.run(req)));
    
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      securityLogger.warn('Input validation failed', {
        ip: req.ip,
        url: req.originalUrl,
        errors: errors.array(),
        body: req.body,
        timestamp: new Date()
      });
      return res.status(400).json({
        error: 'Invalid input data',
        details: errors.array()
      });
    }
    next();
  };
};

// 6. XSS Protection Middleware
const xssProtection = (req, res, next) => {
  if (req.body) {
    for (let key in req.body) {
      if (typeof req.body[key] === 'string') {
        req.body[key] = xss(req.body[key]);
      }
    }
  }
  if (req.query) {
    for (let key in req.query) {
      if (typeof req.query[key] === 'string') {
        req.query[key] = xss(req.query[key]);
      }
    }
  }
  if (req.params) {
    for (let key in req.params) {
      if (typeof req.params[key] === 'string') {
        req.params[key] = xss(req.params[key]);
      }
    }
  }
  next();
};

// 7. MongoDB Injection Protection
const mongoSanitizer = mongoSanitize({
  replaceWith: '_'
});

// 8. Security Audit Middleware
const securityAudit = (req, res, next) => {
  const securityData = {
    ip: req.ip,
    userAgent: req.get('User-Agent'),
    url: req.originalUrl,
    method: req.method,
    timestamp: new Date(),
    headers: {
      authorization: req.get('Authorization') ? 'Bearer [REDACTED]' : undefined,
      'content-type': req.get('Content-Type'),
      origin: req.get('Origin'),
      referer: req.get('Referer')
    }
  };

  // Log sensitive operations
  if (req.originalUrl.includes('/auth/') || 
      req.originalUrl.includes('/admin/') ||
      req.method === 'DELETE' ||
      req.method === 'PUT') {
    securityLogger.info('Security audit log', securityData);
  }

  // Detect suspicious patterns
  const suspiciousPatterns = [
    /(<script|javascript:|data:)/i,
    /(union|select|insert|delete|update|drop)/i,
    /(\.\.|\/etc\/|\/bin\/)/i,
    /(eval\(|setTimeout\(|setInterval\()/i
  ];

  const requestString = JSON.stringify(req.body) + req.originalUrl + JSON.stringify(req.query);
  
  for (let pattern of suspiciousPatterns) {
    if (pattern.test(requestString)) {
      securityLogger.error('Suspicious activity detected', {
        ...securityData,
        pattern: pattern.toString(),
        suspiciousContent: requestString.substring(0, 200)
      });
      return res.status(403).json({ error: 'Suspicious activity detected' });
    }
  }

  next();
};

// 9. Admin Route Protection
const adminRouteProtection = (req, res, next) => {
  // Additional logging for admin routes
  securityLogger.info('Admin route access attempt', {
    ip: req.ip,
    userAgent: req.get('User-Agent'),
    url: req.originalUrl,
    method: req.method,
    user: req.user?.id || 'anonymous',
    timestamp: new Date()
  });
  next();
};

// 10. Request Size Limiter
const requestSizeLimiter = (req, res, next) => {
  const contentLength = req.get('Content-Length');
  if (contentLength && parseInt(contentLength) > 10 * 1024 * 1024) { // 10MB limit
    securityLogger.warn('Request size exceeded', {
      ip: req.ip,
      contentLength,
      url: req.originalUrl,
      timestamp: new Date()
    });
    return res.status(413).json({ error: 'Request too large' });
  }
  next();
};

// Validation schemas
const validationSchemas = {
  email: body('email').isEmail().normalizeEmail().escape(),
  otp: body('otp').isLength({ min: 6, max: 6 }).isNumeric().escape(),
  username: body('username').isLength({ min: 3, max: 50 }).matches(/^[a-zA-Z0-9@._-]+$/).escape(),
  password: body('password').isLength({ min: 8, max: 128 }),
  id: body('id').optional().isMongoId().escape(),
  name: body('name').optional().isLength({ min: 1, max: 100 }).matches(/^[a-zA-Z\s]+$/).escape(),
  phoneNumber: body('phoneNumber').optional().isMobilePhone().escape()
};

export {
  helmetConfig,
  corsOptions,
  generalLimiter,
  authLimiter,
  otpLimiter,
  strictLimiter,
  bruteForceProtection,
  authBruteForce,
  validateInput,
  validationSchemas,
  xssProtection,
  mongoSanitizer,
  securityAudit,
  adminRouteProtection,
  requestSizeLimiter,
  securityLogger
};

export const hppMiddleware = hpp();