const auditLogger = require('../utils/auditLogger');

class EnterpriseRBAC {
  constructor() {
    this.initializePermissions();
    this.initializeRoles();
  }

  // Define all system permissions
  initializePermissions() {
    this.permissions = {
      // User Management
      'users.create': 'Create new users',
      'users.read': 'View user information',
      'users.update': 'Update user information',
      'users.delete': 'Delete users',
      'users.activate': 'Activate/deactivate users',
      'users.impersonate': 'Impersonate other users',

      // Student Management
      'students.create': 'Create new students',
      'students.read': 'View student information',
      'students.update': 'Update student information',
      'students.delete': 'Delete students',
      'students.assign': 'Assign students to teachers',
      'students.progress': 'View student progress',
      'students.reports': 'Generate student reports',

      // Teacher Management
      'teachers.create': 'Create new teachers',
      'teachers.read': 'View teacher information',
      'teachers.update': 'Update teacher information',
      'teachers.delete': 'Delete teachers',
      'teachers.assign': 'Assign teachers to students',
      'teachers.schedule': 'Manage teacher schedules',
      'teachers.performance': 'View teacher performance',

      // Class Management
      'classes.create': 'Schedule new classes',
      'classes.read': 'View class information',
      'classes.update': 'Update class details',
      'classes.delete': 'Cancel classes',
      'classes.start': 'Start classes',
      'classes.complete': 'Mark classes as complete',
      'classes.reschedule': 'Reschedule classes',

      // Assignment Management
      'assignments.create': 'Create assignments',
      'assignments.read': 'View assignments',
      'assignments.update': 'Update assignments',
      'assignments.delete': 'Delete assignments',
      'assignments.grade': 'Grade assignments',
      'assignments.submit': 'Submit assignments',

      // Exam Management
      'exams.create': 'Create exams',
      'exams.read': 'View exams',
      'exams.update': 'Update exams',
      'exams.delete': 'Delete exams',
      'exams.start': 'Start exams',
      'exams.grade': 'Grade exams',
      'exams.take': 'Take exams',

      // Financial Management
      'finance.read': 'View financial data',
      'finance.create': 'Create invoices/payments',
      'finance.update': 'Update financial records',
      'finance.reports': 'Generate financial reports',
      'finance.settings': 'Manage financial settings',

      // Communication
      'chat.read': 'View chat messages',
      'chat.send': 'Send chat messages',
      'chat.moderate': 'Moderate chat content',
      'notifications.send': 'Send notifications',

      // System Administration
      'system.logs': 'View system logs',
      'system.settings': 'Manage system settings',
      'system.backup': 'Perform system backups',
      'system.maintenance': 'Perform system maintenance',
      'system.monitor': 'Monitor system health',

      // File Management
      'files.upload': 'Upload files',
      'files.download': 'Download files',
      'files.delete': 'Delete files',
      'files.manage': 'Manage file permissions',

      // Profile Management
      'profile.read': 'View own profile',
      'profile.update': 'Update own profile',
      'profile.password': 'Change own password',

      // Calendar Management
      'calendar.read': 'View calendar',
      'calendar.create': 'Create calendar events',
      'calendar.update': 'Update calendar events',
      'calendar.delete': 'Delete calendar events'
    };
  }

  // Define roles and their permissions
  initializeRoles() {
    this.roles = {
      admin: {
        name: 'Administrator',
        description: 'Full system access',
        permissions: Object.keys(this.permissions), // All permissions
        inherits: []
      },

      teacher: {
        name: 'Teacher',
        description: 'Teaching and student management',
        permissions: [
          // Student Management (limited)
          'students.read',
          'students.progress',
          'students.reports',

          // Class Management
          'classes.read',
          'classes.start',
          'classes.complete',
          'classes.update',

          // Assignment Management
          'assignments.create',
          'assignments.read',
          'assignments.update',
          'assignments.delete',
          'assignments.grade',

          // Exam Management
          'exams.create',
          'exams.read',
          'exams.update',
          'exams.delete',
          'exams.start',
          'exams.grade',

          // Communication
          'chat.read',
          'chat.send',
          'notifications.send',

          // File Management
          'files.upload',
          'files.download',
          'files.delete',

          // Profile Management
          'profile.read',
          'profile.update',
          'profile.password',

          // Calendar Management
          'calendar.read',
          'calendar.create',
          'calendar.update',
          'calendar.delete'
        ],
        inherits: []
      },

      student: {
        name: 'Student',
        description: 'Student portal access',
        permissions: [
          // Limited class access
          'classes.read',

          // Assignment access
          'assignments.read',
          'assignments.submit',

          // Exam access
          'exams.read',
          'exams.take',

          // Communication
          'chat.read',
          'chat.send',

          // File Management (limited)
          'files.upload',
          'files.download',

          // Profile Management
          'profile.read',
          'profile.update',
          'profile.password',

          // Calendar Management (read-only)
          'calendar.read'
        ],
        inherits: []
      },

      parent: {
        name: 'Parent/Guardian',
        description: 'Parent access to student information',
        permissions: [
          // Student progress (read-only)
          'students.read',
          'students.progress',
          'students.reports',

          // Class information
          'classes.read',

          // Assignment viewing
          'assignments.read',

          // Communication
          'chat.read',
          'chat.send',

          // Financial information
          'finance.read',

          // Profile Management
          'profile.read',
          'profile.update',
          'profile.password',

          // Calendar viewing
          'calendar.read'
        ],
        inherits: []
      },

      moderator: {
        name: 'Moderator',
        description: 'Content moderation and limited admin access',
        permissions: [
          // User management (limited)
          'users.read',
          'users.activate',

          // Student/Teacher viewing
          'students.read',
          'teachers.read',

          // Communication moderation
          'chat.read',
          'chat.send',
          'chat.moderate',
          'notifications.send',

          // File management
          'files.manage',

          // Basic system monitoring
          'system.monitor',

          // Profile Management
          'profile.read',
          'profile.update',
          'profile.password'
        ],
        inherits: []
      }
    };
  }

  // Check if user has permission
  hasPermission(user, permission, resource = null) {
    try {
      if (!user || !user.userType) {
        return false;
      }

      const userRole = this.roles[user.userType];
      if (!userRole) {
        auditLogger.logAuthorizationFailure({
          userId: user._id,
          userType: user.userType,
          permission,
          reason: 'Invalid user role'
        });
        return false;
      }

      // Check direct permission
      if (userRole.permissions.includes(permission)) {
        return this.checkResourceAccess(user, permission, resource);
      }

      // Check inherited permissions
      for (const inheritedRole of userRole.inherits) {
        if (this.roles[inheritedRole]?.permissions.includes(permission)) {
          return this.checkResourceAccess(user, permission, resource);
        }
      }

      auditLogger.logAuthorizationFailure({
        userId: user._id,
        userType: user.userType,
        permission,
        resource: resource?.id || resource,
        reason: 'Permission denied'
      });

      return false;
    } catch (error) {
      auditLogger.error('Permission check failed', {
        error: error.message,
        userId: user?._id,
        permission
      });
      return false;
    }
  }

  // Resource-level access control
  checkResourceAccess(user, permission, resource) {
    if (!resource) {
      return true; // No resource-specific check needed
    }

    const resourceType = resource.type || this.inferResourceType(permission);
    const resourceOwnerId = resource.ownerId || resource.userId || resource.studentId || resource.teacherId;

    switch (resourceType) {
      case 'student':
        return this.checkStudentAccess(user, resource);
      
      case 'teacher':
        return this.checkTeacherAccess(user, resource);
      
      case 'class':
        return this.checkClassAccess(user, resource);
      
      case 'assignment':
        return this.checkAssignmentAccess(user, resource);
      
      case 'exam':
        return this.checkExamAccess(user, resource);
      
      case 'profile':
        return this.checkProfileAccess(user, resource);
      
      default:
        // For other resources, check ownership or admin access
        return user.userType === 'admin' || user._id === resourceOwnerId;
    }
  }

  inferResourceType(permission) {
    const prefix = permission.split('.')[0];
    return prefix === 'profiles' ? 'profile' : prefix.slice(0, -1); // Remove 's' from plural
  }

  checkStudentAccess(user, resource) {
    if (user.userType === 'admin') return true;
    
    if (user.userType === 'teacher') {
      // Teachers can access their assigned students
      return resource.assignedTeachers?.some(t => t.teacherId === user._id) || false;
    }
    
    if (user.userType === 'student') {
      // Students can only access their own data
      return resource._id === user._id || resource.userId === user._id;
    }
    
    if (user.userType === 'parent') {
      // Parents can access their children's data
      return resource.guardianId === user._id;
    }
    
    return false;
  }

  checkTeacherAccess(user, resource) {
    if (user.userType === 'admin') return true;
    
    if (user.userType === 'teacher') {
      // Teachers can access their own data
      return resource._id === user._id || resource.userId === user._id;
    }
    
    return false;
  }

  checkClassAccess(user, resource) {
    if (user.userType === 'admin') return true;
    
    if (user.userType === 'teacher') {
      return resource.teacherId === user._id;
    }
    
    if (user.userType === 'student') {
      return resource.studentId === user._id;
    }
    
    return false;
  }

  checkAssignmentAccess(user, resource) {
    if (user.userType === 'admin') return true;
    
    if (user.userType === 'teacher') {
      return resource.teacherId === user._id;
    }
    
    if (user.userType === 'student') {
      return resource.studentIds?.includes(user._id) || resource.studentId === user._id;
    }
    
    return false;
  }

  checkExamAccess(user, resource) {
    if (user.userType === 'admin') return true;
    
    if (user.userType === 'teacher') {
      return resource.createdBy === user._id;
    }
    
    if (user.userType === 'student') {
      return resource.participants?.includes(user._id);
    }
    
    return false;
  }

  checkProfileAccess(user, resource) {
    if (user.userType === 'admin') return true;
    
    // Users can access their own profile
    return resource._id === user._id || resource.userId === user._id;
  }

  // Middleware factory for permission checking
  requirePermission(permission, options = {}) {
    return async (req, res, next) => {
      try {
        if (!req.user) {
          return res.status(401).json({
            error: 'Authentication required',
            code: 'AUTH_REQUIRED'
          });
        }

        // Get resource information
        let resource = null;
        if (options.getResource) {
          resource = await options.getResource(req);
        } else if (req.params.id) {
          resource = { _id: req.params.id };
        }

        // Check permission
        const hasAccess = this.hasPermission(req.user, permission, resource);
        
        if (!hasAccess) {
          auditLogger.logAuthorizationFailure({
            userId: req.user._id,
            userType: req.user.userType,
            permission,
            endpoint: req.path,
            method: req.method,
            ip: req.ip,
            resource: resource?.id || resource?._id
          });

          return res.status(403).json({
            error: 'Insufficient permissions',
            code: 'PERMISSION_DENIED',
            permission,
            userRole: req.user.userType
          });
        }

        // Log successful authorization
        auditLogger.logDataAccess('AUTHORIZED', permission, {
          userId: req.user._id,
          userType: req.user.userType,
          endpoint: req.path,
          method: req.method,
          resource: resource?.id || resource?._id
        });

        next();
      } catch (error) {
        auditLogger.error('Authorization middleware error', {
          error: error.message,
          userId: req.user?._id,
          permission,
          endpoint: req.path
        });

        res.status(500).json({
          error: 'Authorization check failed',
          code: 'AUTH_ERROR'
        });
      }
    };
  }

  // Multiple permissions check (AND logic)
  requireAllPermissions(permissions, options = {}) {
    return async (req, res, next) => {
      for (const permission of permissions) {
        const middleware = this.requirePermission(permission, options);
        
        try {
          await new Promise((resolve, reject) => {
            middleware(req, res, (error) => {
              if (error) reject(error);
              else resolve();
            });
          });
        } catch (error) {
          return; // Error already handled by requirePermission
        }
      }
      
      next();
    };
  }

  // Multiple permissions check (OR logic)
  requireAnyPermission(permissions, options = {}) {
    return async (req, res, next) => {
      let hasAnyPermission = false;
      
      for (const permission of permissions) {
        let resource = null;
        if (options.getResource) {
          resource = await options.getResource(req);
        } else if (req.params.id) {
          resource = { _id: req.params.id };
        }

        if (this.hasPermission(req.user, permission, resource)) {
          hasAnyPermission = true;
          break;
        }
      }

      if (!hasAnyPermission) {
        auditLogger.logAuthorizationFailure({
          userId: req.user._id,
          userType: req.user.userType,
          permissions,
          endpoint: req.path,
          method: req.method,
          ip: req.ip,
          reason: 'None of the required permissions granted'
        });

        return res.status(403).json({
          error: 'Insufficient permissions',
          code: 'PERMISSION_DENIED',
          requiredPermissions: permissions,
          userRole: req.user.userType
        });
      }

      next();
    };
  }

  // Get user permissions
  getUserPermissions(user) {
    if (!user || !user.userType) {
      return [];
    }

    const role = this.roles[user.userType];
    if (!role) {
      return [];
    }

    let permissions = [...role.permissions];

    // Add inherited permissions
    for (const inheritedRole of role.inherits) {
      if (this.roles[inheritedRole]) {
        permissions.push(...this.roles[inheritedRole].permissions);
      }
    }

    return [...new Set(permissions)]; // Remove duplicates
  }

  // Get role information
  getRoleInfo(roleName) {
    return this.roles[roleName] || null;
  }

  // Validate role
  isValidRole(roleName) {
    return this.roles.hasOwnProperty(roleName);
  }

  // Dynamic permission checking for UI
  createPermissionChecker(user) {
    return {
      can: (permission, resource = null) => this.hasPermission(user, permission, resource),
      cannot: (permission, resource = null) => !this.hasPermission(user, permission, resource),
      hasRole: (roleName) => user.userType === roleName,
      hasAnyRole: (roleNames) => roleNames.includes(user.userType),
      permissions: this.getUserPermissions(user)
    };
  }
}

// Create singleton instance
const rbac = new EnterpriseRBAC();

module.exports = rbac;