const rateLimit = require('express-rate-limit');
const slowDown = require('express-slow-down');
const SecurityConfig = require('../config/security');
const auditLogger = require('../utils/auditLogger');

class EnterpriseRateLimiter {
  constructor() {
    this.suspiciousIPs = new Map();
    this.blockedIPs = new Set();
  }

  // General API rate limiter
  createGeneralLimiter() {
    return rateLimit({
      windowMs: SecurityConfig.rateLimiting.general.windowMs,
      max: SecurityConfig.rateLimiting.general.max,
      message: {
        error: SecurityConfig.rateLimiting.general.message,
        retryAfter: Math.ceil(SecurityConfig.rateLimiting.general.windowMs / 1000)
      },
      standardHeaders: true,
      legacyHeaders: false,
      handler: (req, res) => {
        this.trackSuspiciousActivity(req.ip, 'RATE_LIMIT_EXCEEDED');
        auditLogger.warn('Rate limit exceeded', {
          ip: req.ip,
          userAgent: req.get('User-Agent'),
          endpoint: req.path,
          method: req.method
        });
        res.status(429).json({
          error: 'Too many requests',
          retryAfter: Math.ceil(SecurityConfig.rateLimiting.general.windowMs / 1000)
        });
      },
      skip: (req) => {
        // Skip rate limiting for development tokens in development mode
        if (process.env.NODE_ENV !== 'production') {
          const authHeader = req.header('Authorization');
          if (authHeader && authHeader.includes('dev-')) {
            return true;
          }
        }
        return false;
      }
    });
  }

  // Authentication specific rate limiter (stricter)
  createAuthLimiter() {
    return rateLimit({
      windowMs: SecurityConfig.rateLimiting.auth.windowMs,
      max: SecurityConfig.rateLimiting.auth.max,
      message: {
        error: SecurityConfig.rateLimiting.auth.message,
        retryAfter: Math.ceil(SecurityConfig.rateLimiting.auth.windowMs / 1000)
      },
      standardHeaders: true,
      legacyHeaders: false,
      keyGenerator: (req) => {
        // Rate limit by IP and username combination for login attempts
        const username = req.body?.username || req.body?.email || 'unknown';
        return `${req.ip}-${username}`;
      },
      handler: (req, res) => {
        const username = req.body?.username || req.body?.email || 'unknown';
        this.trackSuspiciousActivity(req.ip, 'AUTH_RATE_LIMIT_EXCEEDED', { username });
        
        auditLogger.warn('Authentication rate limit exceeded', {
          ip: req.ip,
          username: username,
          userAgent: req.get('User-Agent')
        });

        // Temporarily block IP after multiple violations
        this.handleSuspiciousIP(req.ip);

        res.status(429).json({
          error: 'Too many login attempts',
          retryAfter: Math.ceil(SecurityConfig.rateLimiting.auth.windowMs / 1000)
        });
      }
    });
  }

  // API specific rate limiter
  createAPILimiter() {
    return rateLimit({
      windowMs: SecurityConfig.rateLimiting.api.windowMs,
      max: SecurityConfig.rateLimiting.api.max,
      message: {
        error: 'API rate limit exceeded',
        retryAfter: Math.ceil(SecurityConfig.rateLimiting.api.windowMs / 1000)
      },
      standardHeaders: true,
      legacyHeaders: false,
      keyGenerator: (req) => {
        // Use user ID if authenticated, otherwise IP
        return req.user?.id || req.ip;
      },
      handler: (req, res) => {
        auditLogger.warn('API rate limit exceeded', {
          ip: req.ip,
          userId: req.user?.id,
          endpoint: req.path,
          method: req.method
        });
        res.status(429).json({
          error: 'API rate limit exceeded',
          retryAfter: Math.ceil(SecurityConfig.rateLimiting.api.windowMs / 1000)
        });
      },
      skip: (req) => {
        // Skip for development mode
        return process.env.NODE_ENV !== 'production' && 
               req.header('Authorization')?.includes('dev-');
      }
    });
  }

  // Progressive delay for repeated requests
  createSlowDown() {
    return slowDown({
      windowMs: 15 * 60 * 1000, // 15 minutes
      delayAfter: 50, // Allow 50 requests per window without delay
      delayMs: 500, // Add 500ms delay for each request after delayAfter
      maxDelayMs: 5000, // Maximum delay of 5 seconds
      onLimitReached: (req, res, options) => {
        auditLogger.warn('Slow down activated', {
          ip: req.ip,
          userAgent: req.get('User-Agent'),
          endpoint: req.path
        });
      }
    });
  }

  // DDoS protection middleware
  createDDoSProtection() {
    return (req, res, next) => {
      const ip = req.ip;
      
      // Check if IP is blocked
      if (this.blockedIPs.has(ip)) {
        auditLogger.error('Blocked IP attempted access', {
          ip: ip,
          endpoint: req.path,
          userAgent: req.get('User-Agent')
        });
        return res.status(403).json({
          error: 'Access denied',
          reason: 'IP temporarily blocked due to suspicious activity'
        });
      }

      // Check for suspicious patterns
      if (this.detectDDoSPattern(req)) {
        this.handleSuspiciousIP(ip);
        return res.status(429).json({
          error: 'Request rejected',
          reason: 'Suspicious request pattern detected'
        });
      }

      next();
    };
  }

  // Track suspicious activity
  trackSuspiciousActivity(ip, type, metadata = {}) {
    if (!this.suspiciousIPs.has(ip)) {
      this.suspiciousIPs.set(ip, []);
    }

    const activities = this.suspiciousIPs.get(ip);
    activities.push({
      type,
      timestamp: Date.now(),
      metadata
    });

    // Keep only recent activities (last hour)
    const oneHourAgo = Date.now() - (60 * 60 * 1000);
    const recentActivities = activities.filter(a => a.timestamp > oneHourAgo);
    this.suspiciousIPs.set(ip, recentActivities);

    // Auto-block if too many suspicious activities
    if (recentActivities.length >= 10) {
      this.blockIP(ip, '1 hour');
    }
  }

  // Handle suspicious IP addresses
  handleSuspiciousIP(ip) {
    const activities = this.suspiciousIPs.get(ip) || [];
    const recentCount = activities.filter(a => 
      Date.now() - a.timestamp < 15 * 60 * 1000 // Last 15 minutes
    ).length;

    if (recentCount >= 5) {
      this.blockIP(ip, '30 minutes');
    }
  }

  // Block IP address
  blockIP(ip, duration) {
    this.blockedIPs.add(ip);
    
    auditLogger.error('IP address blocked', {
      ip: ip,
      duration: duration,
      reason: 'Excessive suspicious activity'
    });

    // Auto-unblock after duration
    const durationMs = this.parseDuration(duration);
    setTimeout(() => {
      this.blockedIPs.delete(ip);
      auditLogger.info('IP address unblocked', { ip: ip });
    }, durationMs);
  }

  // Detect DDoS patterns
  detectDDoSPattern(req) {
    const userAgent = req.get('User-Agent');
    const ip = req.ip;

    // Check for suspicious user agents
    const suspiciousAgents = [
      'bot', 'crawler', 'spider', 'scraper', 
      'curl', 'wget', 'python', 'go-http-client'
    ];

    if (!userAgent || suspiciousAgents.some(agent => 
      userAgent.toLowerCase().includes(agent)
    )) {
      this.trackSuspiciousActivity(ip, 'SUSPICIOUS_USER_AGENT', { userAgent });
      return true;
    }

    // Check for suspicious request patterns
    if (req.path.includes('..') || 
        req.path.includes('<script>') ||
        req.path.includes('eval(') ||
        req.path.length > 2000) {
      this.trackSuspiciousActivity(ip, 'SUSPICIOUS_REQUEST_PATH', { path: req.path });
      return true;
    }

    return false;
  }

  // Parse duration string to milliseconds
  parseDuration(duration) {
    const units = {
      'minute': 60 * 1000,
      'minutes': 60 * 1000,
      'hour': 60 * 60 * 1000,
      'hours': 60 * 60 * 1000,
      'day': 24 * 60 * 60 * 1000,
      'days': 24 * 60 * 60 * 1000
    };

    const match = duration.match(/(\d+)\s*(minute|minutes|hour|hours|day|days)/);
    if (match) {
      const value = parseInt(match[1]);
      const unit = match[2];
      return value * units[unit];
    }

    return 30 * 60 * 1000; // Default to 30 minutes
  }

  // Cleanup old data periodically
  cleanup() {
    const oneHourAgo = Date.now() - (60 * 60 * 1000);
    
    // Clean suspicious IPs data
    for (const [ip, activities] of this.suspiciousIPs.entries()) {
      const recentActivities = activities.filter(a => a.timestamp > oneHourAgo);
      if (recentActivities.length === 0) {
        this.suspiciousIPs.delete(ip);
      } else {
        this.suspiciousIPs.set(ip, recentActivities);
      }
    }
  }

  // Get middleware for specific use case
  getMiddleware(type = 'general') {
    switch (type) {
      case 'auth':
        return this.createAuthLimiter();
      case 'api':
        return this.createAPILimiter();
      case 'slowdown':
        return this.createSlowDown();
      case 'ddos':
        return this.createDDoSProtection();
      default:
        return this.createGeneralLimiter();
    }
  }
}

// Create singleton instance
const rateLimiter = new EnterpriseRateLimiter();

// Cleanup old data every 30 minutes
setInterval(() => {
  rateLimiter.cleanup();
}, 30 * 60 * 1000);

module.exports = rateLimiter;