const winston = require('winston');

// Performance logger
const performanceLogger = winston.createLogger({
  level: 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.json()
  ),
  defaultMeta: { service: 'edumetrix-performance' },
  transports: [
    new winston.transports.File({ filename: 'logs/performance.log' }),
    new winston.transports.Console({
      format: winston.format.simple(),
      level: 'warn' // Only log warnings and errors to console
    })
  ]
});

// Request timing middleware
const requestTimer = (req, res, next) => {
  const startTime = Date.now();
  
  // Track memory usage
  const startMemory = process.memoryUsage();
  
  res.on('finish', () => {
    const duration = Date.now() - startTime;
    const endMemory = process.memoryUsage();
    
    // Calculate memory delta
    const memoryDelta = {
      heapUsed: endMemory.heapUsed - startMemory.heapUsed,
      heapTotal: endMemory.heapTotal - startMemory.heapTotal,
      external: endMemory.external - startMemory.external
    };
    
    const logData = {
      method: req.method,
      url: req.originalUrl,
      statusCode: res.statusCode,
      duration: `${duration}ms`,
      ip: req.ip,
      userAgent: req.get('User-Agent'),
      contentLength: res.get('Content-Length') || 0,
      memory: {
        current: endMemory,
        delta: memoryDelta
      },
      timestamp: new Date()
    };
    
    // Log slow requests (>1000ms) as warnings
    if (duration > 1000) {
      performanceLogger.warn('Slow request detected', logData);
    } else if (duration > 500) {
      performanceLogger.info('Moderate request time', logData);
    }
    
    // Log memory spikes
    if (memoryDelta.heapUsed > 10 * 1024 * 1024) { // 10MB
      performanceLogger.warn('High memory usage detected', {
        ...logData,
        memoryWarning: 'Heap usage increased by more than 10MB'
      });
    }
  });
  
  next();
};

// Database query performance monitor
const dbQueryMonitor = {
  logSlowQuery: (operation, duration, query = {}) => {
    if (duration > 1000) { // Queries slower than 1 second
      performanceLogger.warn('Slow database query', {
        operation,
        duration: `${duration}ms`,
        query: JSON.stringify(query).substring(0, 500), // Truncate long queries
        timestamp: new Date()
      });
    }
  }
};

// Memory monitoring
const memoryMonitor = () => {
  const usage = process.memoryUsage();
  const threshold = 500 * 1024 * 1024; // 500MB
  
  if (usage.heapUsed > threshold) {
    performanceLogger.warn('High memory usage', {
      heapUsed: `${Math.round(usage.heapUsed / 1024 / 1024)}MB`,
      heapTotal: `${Math.round(usage.heapTotal / 1024 / 1024)}MB`,
      external: `${Math.round(usage.external / 1024 / 1024)}MB`,
      rss: `${Math.round(usage.rss / 1024 / 1024)}MB`,
      threshold: `${Math.round(threshold / 1024 / 1024)}MB`,
      timestamp: new Date()
    });
  }
};

// Start memory monitoring (check every 30 seconds)
const startMemoryMonitoring = () => {
  setInterval(memoryMonitor, 30000);
};

// CPU monitoring
const cpuMonitor = () => {
  const usage = process.cpuUsage();
  const totalCpu = usage.user + usage.system;
  
  performanceLogger.info('CPU usage', {
    user: usage.user,
    system: usage.system,
    total: totalCpu,
    timestamp: new Date()
  });
};

// Health check endpoint data
const getHealthMetrics = () => {
  const uptime = process.uptime();
  const memory = process.memoryUsage();
  const cpu = process.cpuUsage();
  
  return {
    status: 'healthy',
    uptime: {
      seconds: Math.floor(uptime),
      formatted: `${Math.floor(uptime / 3600)}h ${Math.floor((uptime % 3600) / 60)}m ${Math.floor(uptime % 60)}s`
    },
    memory: {
      heapUsed: `${Math.round(memory.heapUsed / 1024 / 1024)}MB`,
      heapTotal: `${Math.round(memory.heapTotal / 1024 / 1024)}MB`,
      external: `${Math.round(memory.external / 1024 / 1024)}MB`,
      rss: `${Math.round(memory.rss / 1024 / 1024)}MB`
    },
    cpu: {
      user: cpu.user,
      system: cpu.system
    },
    nodeVersion: process.version,
    platform: process.platform,
    pid: process.pid,
    timestamp: new Date()
  };
};

module.exports = {
  requestTimer,
  dbQueryMonitor,
  memoryMonitor,
  startMemoryMonitoring,
  cpuMonitor,
  getHealthMetrics,
  performanceLogger
};